/* tsc-command-interface.c: RPC interface to/from TSC for commands */
/*

Copyright (C) 2010 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <errno.h>

#include <melco/tsc-message.h>
#include <tws4/command-delegation.h>
#include <tws4/command-delegator.h>

/* include path to be specified in Makefile */
#include "rpc-hosts.h"
#include "tws_tsc_command.h"
#include "tws_tsc_reply.h"

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define RPC_TIMEOUT 2

typedef struct {
	char *server;	/* tsc-c */
	char *client;	/* tws4-c */
} sender_args_s;

/* functions for tws4/command-delegator.h */
void usage(FILE *stream);
void show_command(CommandMessage* message);
void show_reception(ReceptionResponseMessage* message);
void show_completion(CompletionResponseMessage* message);
void show_error(char *string);
static void from_tsc_prog_1(struct svc_req *rqstp, register SVCXPRT *transp);
bool_t* from_tsc_1_svc(MF_NetForm* form, struct svc_req* svcdata);
void waiter(int fd, void *args);
ssize_t sender(CommandMessage *message, int uid, int gid, void *args);

struct _waiter_args {
};

/* global variables */
static int _reply_fd;

void
usage(FILE *stream)
{
	fputs("usage: tsc-command-interface [options]\n", stream);
  fputs("\tpaths TSC commands from socket to RPC and sends back responses.\n", stream);
	fputs("Telnet to 127.0.0.1 and send command like:\n", stream);
	fputs("CDTSC%TWS4000020031222092442600270000001A1901naoj **** % % %\n\n", stream);
	fputs("Commands, responses, and warnings are displayed as well as sent to client.\n", stream);
	fputs("OPTIONS:\n", stream);
	fprintf(stream, "-p port : listen to PORT (default:%d)\n", COMMMAND_DELGATOR_PORT);
	fprintf(stream, "-s servername : send commands to server (default:%s)\n", TWS4_HOST_NAME);
	fprintf(stream, "-c clientname : name of client (default:%s)\n", TSC_HOST_NAME);
}

void
show_command(CommandMessage* message)
{
	fputs("TWS4:sending ", stderr);
	fwrite(CommandMessage_tostr(message), CommandMessage_strlen(message), 1, stderr);
	fputc('\n', stderr);
}

void
show_reception(ReceptionResponseMessage* message)
{
	fputs("TWS4:got     ", stderr);
	fwrite(ReceptionResponseMessage_tostr(message), ReceptionResponseMessage_strlen(message), 1, stderr);
	fputc('\n', stderr);
}

void
show_completion(CompletionResponseMessage* message)
{
	fputs("TWS4:got     ", stderr);
	fwrite(CompletionResponseMessage_tostr(message), CompletionResponseMessage_strlen(message), 1, stderr);
	fputc('\n', stderr);
}

void
show_error(char *string)
{
	fprintf(stderr, "%s\n", string);
}

ssize_t
sender(CommandMessage *message, int uid, int gid, void *args)
{
	bool_t *result;
	FM_NetForm form;
	struct timeval time_out;
	char *server, *client;

	CLIENT* cl;

	server = ((sender_args_s *) args)->server;
	client = ((sender_args_s *) args)->client;

	form.FM_NetForm_val = CommandMessage_tostr(message);
	form.FM_NetForm_len = CommandMessage_strlen(message);

	cl = clnt_create(server, TOWARD_TSC_PROG, TOWARD_TSC_VERS, "tcp");
	if (!cl) return 0;

	cl->cl_auth = authunix_create(client, uid, gid, 0, NULL);
	time_out.tv_sec = RPC_TIMEOUT;
	time_out.tv_usec = 0;
	clnt_control(cl, CLSET_TIMEOUT, (char *) &time_out);

	result = toward_tsc_1(&form, cl);
	if (!result) return 0;

	auth_destroy(cl->cl_auth);
	clnt_destroy(cl);
	return form.FM_NetForm_len;
}

/* copied from tws_tsc_reply_svc.c */
static void
from_tsc_prog_1(struct svc_req *rqstp, register SVCXPRT *transp)
{
	union {
		MF_NetForm from_tsc_1_arg;
	} argument;
	char *result;
	xdrproc_t _xdr_argument, _xdr_result;
	char *(*local)(char *, struct svc_req *);

	switch (rqstp->rq_proc) {
	case NULLPROC:
		(void) svc_sendreply (transp, (xdrproc_t) xdr_void, (char *)NULL);
		return;

	case FROM_TSC:
		_xdr_argument = (xdrproc_t) xdr_MF_NetForm;
		_xdr_result = (xdrproc_t) xdr_bool;
		local = (char *(*)(char *, struct svc_req *)) from_tsc_1_svc;
		break;

	default:
		svcerr_noproc (transp);
		return;
	}
	memset ((char *)&argument, 0, sizeof (argument));
	if (!svc_getargs (transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument)) {
		svcerr_decode (transp);
		return;
	}
	result = (*local)((char *)&argument, rqstp);
	if (result != NULL && !svc_sendreply(transp, (xdrproc_t) _xdr_result, result)) {
		svcerr_systemerr (transp);
	}
	if (!svc_freeargs (transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument)) {
		fprintf (stderr, "%s", "unable to free arguments");
		exit (1);
	}
	return;
}

bool_t*
from_tsc_1_svc(MF_NetForm* form, struct svc_req* svcdata __attribute__ ((unused)))
{
	static bool_t result = TRUE;
	if (write(_reply_fd, form->MF_NetForm_val, form->MF_NetForm_len) < 0)
		{
			perror("write()");
			result = FALSE;
		}
	return &result;
}

void
waiter(int fd, void *args __attribute__ ((unused)))
{
	register SVCXPRT *transp;

	_reply_fd = fd;

	pmap_unset(FROM_TSC_PROG, FROM_TSC_VERS);
	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (!transp)
		{
			perror("svctcp_create()");
			exit(EXIT_FAILURE);
		}
	if (!svc_register(transp, FROM_TSC_PROG, FROM_TSC_VERS, from_tsc_prog_1, IPPROTO_TCP))
		{
			perror("svctcp_create()");
			exit(EXIT_FAILURE);
		}

	svc_run();
	perror("svc_run()");
	exit(EXIT_FAILURE);
}

int
main(int argc, char *argv[])
{
	long r;
	struct sockaddr_in myaddr;
	struct _waiter_args waiter_args;
	logger_functions_s loggers;
	short port;
	char *tschostname = TSC_HOST_NAME;
	char *tws4hostname = TWS4_HOST_NAME;
	sender_args_s sender_args;

	/* command line options */
	port = COMMMAND_DELGATOR_PORT;
	{
		int o;
		while((o = getopt(argc, argv, "hp:c:s:")) != -1)
			{
				switch(o)
					{
					case 'h':	/* help */
						usage(stdout);
						return EXIT_SUCCESS;
					case 'p':	/* port */
						port = atoi(optarg);
						break;
					case 'c': /* client */
						tws4hostname = optarg;
						break;
					case 's': /* server */
						tschostname = optarg;
						break;
					default:
						usage(stderr);
						return EXIT_FAILURE;
					}
			}
	}
	
	/* loggers */
	loggers.command = (command_logger_callback_t) show_command;
	loggers.reception = (reception_logger_callback_t) show_reception;
	loggers.completion =(completion_logger_callback_t) show_completion;
	loggers.error = (error_logger_callback_t) show_error;

	/* tws4 server setting */
	memset(&myaddr, 0, sizeof(myaddr));
	myaddr.sin_family = AF_INET;
	myaddr.sin_port = htons(port);
	myaddr.sin_addr.s_addr = htonl(0x7f000001);	/* 127.0.0.1 */

	sender_args.server = tschostname;
	sender_args.client = tws4hostname;

	fprintf(stderr, "waiting for commands at 127.0.0.1 port %d ...\n", port);
	r = command_delegator(
				&myaddr,
				(CommandMessage_send_t) sender, (void *) &sender_args,
				(message_waiter_t) waiter, (void *) &waiter_args,
				&loggers);
	if (r >= 0)
		fprintf(stdout, "%ld messages sent\n", r);
	else
		perror("");

	return EXIT_SUCCESS;
}
