#!/usr/bin/python
import time
import calendar
import unittest
import checkalarmchange

# This test can only be run on TWS4
class FetchHistoryTestCase(unittest.TestCase):
	def notestFetchHistory(self):
		assert(None != checkalarmchange.fetchHistory())

class StatusTestCase(unittest.TestCase):
	def setUp(self):
		self.stat = checkalarmchange.Status(0x00B300B)
		self.stat.name = 'Cs Bolt - MLP3 Fault'
	def testId(self):
		assert(0x00B300B == self.stat.status_id)
		assert('Cs Bolt - MLP3 Fault' == self.stat.name)
	def testNotRaised(self):
		self.assertFalse(self.stat.hasRaised)
		self.assertFalse(self.stat.beingRaised)
		self.assertFalse(self.stat.isAlarm)
	def testAlarm(self):
		self.stat.isAlarm = True
		assert(self.stat.isAlarm)
	def testRaised(self):
		t = calendar.timegm((2010, 12, 18, 1, 15, 00))
		self.stat.setRaisedAt(t)
		assert(t == self.stat.raisedAt)
		assert(self.stat.hasRaised)
		assert(self.stat.beingRaised)
	def testCleared(self):
		t = calendar.timegm((2010, 12, 18, 1, 15, 00))
		self.stat.setClearedAt(t)
		assert(t == self.stat.clearedAt)
		self.assertFalse(self.stat.hasRaised)
		self.assertFalse(self.stat.beingRaised)
	def testRaisedAndCleared(self):
		self.stat.setRaisedAt(calendar.timegm((2010, 12, 18, 1, 15, 00)))
		self.stat.setClearedAt(calendar.timegm((2010, 12, 18, 1, 16, 00)))
		assert(self.stat.hasRaised)
		self.assertFalse(self.stat.beingRaised)
	def testClearedAndRaise(self):
		self.stat.setClearedAt(calendar.timegm((2010, 12, 18, 1, 15, 00)))
		self.stat.setRaisedAt(calendar.timegm((2010, 12, 18, 1, 16, 00)))
		assert(self.stat.hasRaised)
		assert(self.stat.beingRaised)
	def testRaisedAndClearedReverse(self):
		self.stat.setClearedAt(calendar.timegm((2010, 12, 18, 1, 16, 00)))
		self.stat.setRaisedAt(calendar.timegm((2010, 12, 18, 1, 15, 00)))
		assert(self.stat.hasRaised)
		self.assertFalse(self.stat.beingRaised)
	def testClearedAndRaiseReverse(self):
		self.stat.setRaisedAt(calendar.timegm((2010, 12, 18, 1, 16, 00)))
		self.stat.setClearedAt(calendar.timegm((2010, 12, 18, 1, 15, 00)))
		assert(self.stat.hasRaised)
		assert(self.stat.beingRaised)

class HistoryTestCase(unittest.TestCase):
	def setUp(self):
		self.hist = checkalarmchange.History()
	def testEmpty(self):
		assert(0 == len(self.hist.status))
	def testStrToTime(self):
		assert(calendar.timegm((2010, 12, 28, 2, 17, 3)) + 0.9 == checkalarmchange.strToTime("201012280217039"))
	def testParse(self):
		self.hist.parse("201012280217036 201012280217039 OCR 1 00020D0 EL Cable(IR) Slave Warning Up\n")
		stat = self.hist.status[0x00020D0]
		assert('EL Cable(IR) Slave Warning Up' == stat.name)
		assert(stat.isAlarm)
		assert(calendar.timegm((2010, 12, 28, 2, 17, 3)) + 0.9 == stat.raisedAt)
		assert(stat.beingRaised)
		self.hist.parse("201012280217046 201012280217049 RCV 1 00020D0 EL Cable(IR) Slave Warning Up\n")
		assert(calendar.timegm((2010, 12, 28, 2, 17, 4)) + 0.9 == stat.clearedAt)
		self.assertFalse(stat.beingRaised)
	def testTimeRange(self):
		self.hist.setTimeRange(calendar.timegm((2010, 12, 28, 2, 17, 3)) + 0.95, None)
		self.hist.parse("201012280217036 201012280217039 OCR 1 00020D0 EL Cable(IR) Slave Warning Up\n")
		assert(0 == len(self.hist.status))
		assert(self.hist.parse("201012280217046 201012280217049 RCV 1 00020D0 EL Cable(IR) Slave Warning Up\n"))
		stat = self.hist.status[0x00020D0]
		self.assertFalse(stat.hasRaised)
		self.assertFalse(stat.beingRaised)
	def testRead(self):
		self.hist.setTimeRange(calendar.timegm((2010, 12, 18, 17, 37, 32)) + 0.6, calendar.timegm((2010, 12, 18, 17, 39, 25)) + 0.0)
		self.hist.read('test-APPARATUS_STATUS_ALL.LOG')
		assert(3 == len(self.hist.status))
	def testFileTooNew(self):
		self.hist.setTimeRange(calendar.timegm((2000, 12, 18, 0, 15, 0)), None)
		self.assertRaises(RuntimeError, self.hist.read, ('test-APPARATUS_STATUS_ALL.LOG'))
	def testFileTooOld(self):
		self.hist.setTimeRange(None, calendar.timegm((2010, 12, 28, 3, 35, 0)))
		self.assertRaises(RuntimeError, self.hist.read, ('test-APPARATUS_STATUS_ALL.LOG'))

class TimeParserTestCase(unittest.TestCase):
	def testToday(self):
		c = time.time()
		d = time.mktime(((time.localtime(c)[0:3]) + (12, 34, 0, 0, 0, -1)))
		t = checkalarmchange.timeParse('12:34', c)
		assert(d == t)
	def testAbsoluteday(self):
		d = time.mktime((2010, 12, 14, 12, 34, 0, 0, 0, -1))
		t = checkalarmchange.timeParse('2010-12-14 12:34')
		assert(d == t)
	def testRelative(self):
		c = time.time()
		d = c - 30*60
		t = checkalarmchange.timeParse('30 min ago', c)
		assert(d == t)
		d = c - 3*60*60
		t = checkalarmchange.timeParse('3 hours ago', c)
		assert(d == t)

if __name__ == "__main__":
	unittest.main(testRunner = unittest.TextTestRunner(verbosity = 1))
