/* exec-tsc.c : user interface to send command to TSC */
/*

Copyright (C) 2006 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#define COMMAND_DST "TSC"
#define COMMAND_SRC "TWS4"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>

#include <melco/tsc-message.h>
#include <tws4/command-delegation.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define RX_BUF_SIZE 1025
#define TX_BUF_SIZE 1025

typedef enum {
	exec_tsc_success = 0,
	exec_tsc_connection_closed,	/* socket closed */
	exec_tsc_err_malloc,	/* memory over */
	exec_tsc_err_syscall,	/* error from system call */
	exec_tsc_err_response,	/* error response from TSC */
	exec_tsc_err_malformed,	/* TSC command syntax error */
	exec_tsc_err_timeout,	/* timeout out */
	exec_tsc_quit,	/* quit */
	exec_tsc_err_malformed_id,	/* command ID is malformed */
} exec_tsc_error_t;

typedef struct _pid_list {
	pid_t pid;
	struct _pid_list *next;
} pid_list_s;

typedef struct {
	int verbosity;	/* verbosity */
	int async;	/* 1 if asyncrounous */
	int interactive;	/* 1 if interactive */
	FILE *output;	/* output stream */
	pid_list_s worker_list_head;	/* list of workers */
	short port;	/* port */
} exec_tsc_conf_s;

#define DEF_TSC_CONF {\
	1, /* verbosity */ \
	0, /* 1 if asyncronous */ \
	0, /* 1 if interactive */ \
	stdout, /* output stream */ \
	{0, NULL}, /* list of workers */ \
	COMMMAND_DELGATOR_PORT, /* port */ \
}

#define VERBOSITY_SHOW_ERROR_SUMMARY 1
#define VERBOSITY_SHOW_ERROR_DETAIL 2
#define VERBOSITY_SHOW_PROGRESS 3

exec_tsc_conf_s _exec_tsc_conf;

/* private functions */
void usage(FILE *stream);
void command_usage(FILE *stream);
ssize_t skgets(int sk, char *buf, ssize_t n);
static void sigchld(int x);

/* open and close socket to delegator */
int open_socket_to_delegator(exec_tsc_conf_s *conf);

/* sends a command and returns after getting and showing result */
exec_tsc_error_t exec_tsc_command(char *string, exec_tsc_conf_s *conf);
exec_tsc_error_t get_delegator_status(exec_tsc_conf_s *conf);
exec_tsc_error_t set_delegator_id(exec_tsc_conf_s *conf, int type, int uid);
exec_tsc_error_t send_and_wait_tsc_command(exec_tsc_conf_s *conf, char *string);
exec_tsc_error_t send_cd_message(int sk, exec_tsc_conf_s *conf, char *string);
exec_tsc_error_t get_ca_message(int sk, exec_tsc_conf_s *conf);
exec_tsc_error_t get_ce_message(int sk, exec_tsc_conf_s *conf);

static void
sigchld(int x __attribute__ ((unused)))
{
	while(waitpid(-1, NULL, WNOHANG) > 0);
	signal(SIGCHLD, sigchld);
}

/* open the socket to command delegator */
int
open_socket_to_delegator(exec_tsc_conf_s *conf)
{
	int sk;
	struct sockaddr_in server;

	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_port = htons(conf->port);
	server.sin_addr.s_addr = htonl(0x7f000001); /* 127.0.0.1 */

	if ((sk = socket(AF_INET, SOCK_STREAM, 0)) == -1)
		{
			perror("socket()");
			return -1;
		}
	if (connect(sk, (struct sockaddr *) &server, sizeof(server)) == -1)
		{
			perror("connect()");
			fputs("You have to start command delegator or specify port number with -p option.\n", stderr);
			return -1;
		}

	return sk;
}

/* wait for a line from delegator */
ssize_t
skgets(int sk, char *buf, ssize_t n)
{
	ssize_t r, c;

	c = 0;
	do {
		r = read(sk, buf + c, n - c - 2);
		if (0 > r)
			{
				perror("read()");
				break;
			}
		c += r;
	} while(0 < n - c - 2 && buf[c - 1] != '\n');
	buf[c] = '\0';

	return r < 0 ? r : c;
}

void
usage(FILE *stream)
{
	fputs("usage: exec-tsc [options] [command]\n", stream);
	fputs("\tsends TSC command(s), e.g. 1A1901loginame password % % %,\n", stream);
	fputs("\tthrough command delegator on localhost.\n", stream);
	fputs("OPTIONS:\n", stream);
	fputs("-v      : increase verbosity\n", stream);
	fputs("-q      : decrease verbosity\n", stream);
	fprintf(stream, "-p port : listen to PORT (default:%d)\n", COMMMAND_DELGATOR_PORT);
	fputs("-a      : work asynchronously, exit after receiving reception message\n", stream);
	fputs("-i      : work interactively reading commands from stdin\n", stream);
	fputs("-h      : show this\n", stream);
}

void
command_usage(FILE *stream)
{
	fputs("LOCAL AND DELEGATOR COMMANDS\n", stream);
	fputs("v, verbose   : increase verbosity\n", stream);
	fputs("q, quiet     : decrease verbosity\n", stream);
	fputs("exit         : exit exec-tsc\n", stream);
	fputs("h, help      : show this\n\n", stream);
	fputs("s, status    : show status of command delegator\n", stream);
	fputs("u, uid NUMBER: set uid of commands sent to TSC\n", stream);
	fputs("g, gid NUMBER: set gid of commands sent to TSC\n", stream);
}

exec_tsc_error_t
get_delegator_status(exec_tsc_conf_s *conf)
{
	char command = DELEGATOR_SHOW_STATUS;
	char buf[RX_BUF_SIZE];
	int sk;

	sk = open_socket_to_delegator(conf);
	if (sk == -1) return exec_tsc_err_syscall;

	if (sizeof(command) != write(sk, &command, sizeof(command)))
		{
			perror("write()");
			close(sk);
			return exec_tsc_err_syscall;
		}

	if (skgets(sk, buf, sizeof(buf)) <= 0)
		{
			close(sk);
			return exec_tsc_err_syscall;
		}

	fputs(buf, conf->output);
	fflush(conf->output);

	if (close(sk) == -1) return exec_tsc_err_syscall;

	return exec_tsc_success;
}

exec_tsc_error_t
set_delegator_id(exec_tsc_conf_s *conf, int type, int uid)
{
	char command[TX_BUF_SIZE];
	int sk;
	int len;

	len = snprintf(command, sizeof(command) - 1, "%c%c%d", DELEGATOR_SET_ID, type, uid);
	command[len] = '\0';

	sk = open_socket_to_delegator(conf);
	if (sk == -1) return exec_tsc_err_syscall;

	if (len + 1 != write(sk, &command, len + 1))
		{
			perror("write()");
			close(sk);
			return exec_tsc_err_syscall;
		}

	if (close(sk) == -1) return exec_tsc_err_syscall;

	return exec_tsc_success;
}

/* send a command, wait for response, fork to wait for completion */
exec_tsc_error_t
send_and_wait_tsc_command(exec_tsc_conf_s *conf, char *string)
{
	int sk;
	exec_tsc_error_t r;
	pid_t pid = -1;

	sk = open_socket_to_delegator(conf);
	if (sk == -1) return exec_tsc_err_syscall;

	/* send the command message */
	if ((r = send_cd_message(sk, conf, string)) != exec_tsc_success)
		{
			if (r != exec_tsc_err_malformed_id)
				close(sk);	/* make sure no date from delegator */
			else
				close(sk);	/* we have to close from our side because no data is sent */
			return r;
		}

	/* get the reception response */
	if ((r = get_ca_message(sk, conf)) != exec_tsc_success)
		{
			close(sk);
			return r;
		}

	/* get the copmletion response */
	if (conf->interactive) pid = fork();
	if (!conf->interactive || 0 == pid || -1 == pid)
		{
			r = get_ce_message(sk, conf);
			if (0 == pid) exit(EXIT_SUCCESS);	/* nothing expected from child process */
			if (r != exec_tsc_success)
				{
					close(sk);
					return r;
				}
			if (close(sk) == -1) return exec_tsc_err_syscall;
		}

	return exec_tsc_success;
}

/* send a command message to TSC */
exec_tsc_error_t
send_cd_message(int sk, exec_tsc_conf_s *conf, char *string)
{
	CommandMessage command;
	long id;
	char *pars;
	int parlen;
	struct timeval txtime;

	/* command ID */
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1], *ptr;
		memcpy(buf, string, COMMANDMESSAGE_ID_SIZE);
		buf[COMMANDMESSAGE_ID_SIZE] = '\0';
		id = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_ID_SIZE)
			{
				if (conf->verbosity >= VERBOSITY_SHOW_ERROR_SUMMARY)
					fputs("malformed command ID\n", stderr);
				return exec_tsc_err_malformed_id;
			}
	}

	/* parameter */
	{
		pars = string + COMMANDMESSAGE_ID_SIZE;
		parlen = strlen(pars);
	}

	/* time stamp */
	{
		memset(&txtime, 0, sizeof(txtime));
		gettimeofday(&txtime, NULL);
	}

	/* format */
	{
		CommandMessage_init(&command);
		if (CommandMessage_set(&command,
				COMMAND_DEMAND_MESSAGE_TYPE,
				COMMAND_DST,
				COMMAND_SRC,
				1, // This has to be the same for 1A1901 and 1A1902 (int) getpid(),
				&txtime,
				0,	/* seqnum to be set in delgator */
				id,
				pars,
				parlen))
			{
				if (conf->verbosity >= VERBOSITY_SHOW_ERROR_SUMMARY)
					fputs("malformed command\n", stderr);
				return exec_tsc_err_malformed;
			}
	}
	CommandMessage_incref(&command);

	/* send */
	{
		size_t len;
		len = CommandMessage_strlen(&command);
		if ((ssize_t) len != write(sk, CommandMessage_tostr(&command), len))
			{
				perror("write()");
				close(sk);
				return exec_tsc_err_syscall;
			}
	}

	CommandMessage_decref(&command);

	/* get the echo back and check it */
	{
		tsc_error_t e;
		char buf[RX_BUF_SIZE];
		ssize_t n;

		n = skgets(sk, buf, sizeof(buf));
		if (n == 0) return exec_tsc_connection_closed;
		if (n < 0) return exec_tsc_err_syscall;
		if (buf[n - 1] != '\n')
			{
				fprintf(stderr, "Malformed communication: got \"%s\" from delegator without \\n:", buf);
				{
					ssize_t i;
					for(i = 0; i < n; i++) fprintf(stderr, "[%02X]", buf[i]);
					fputs("\n", stderr);
				}
				return exec_tsc_err_malformed;
			}

		CommandMessage_init(&command);
		e = strto_CommandMessage(&command, buf, NULL);
		switch(e)
			{
				case tsc_success:
					if (conf->verbosity >= VERBOSITY_SHOW_PROGRESS)
						fprintf(conf->output, "sent: %s", buf);
					break;
				case tsc_error_malformed:	/* should be time out message from delgator */
					if (conf->verbosity >= VERBOSITY_SHOW_ERROR_SUMMARY)
						{
							fputs(buf, conf->output);
							fflush(conf->output);
						}
					return exec_tsc_err_timeout;
				case tsc_error_malloc:
					perror("error");
					return exec_tsc_err_syscall;
				default:
					fprintf(stderr, "unsupported error at %s:%d\n", __FILE__, __LINE__);
					return exec_tsc_err_syscall;
			}
	}

	return exec_tsc_success;
}

/* get a response message from TSC */
exec_tsc_error_t
get_ca_message(int sk, exec_tsc_conf_s *conf)
{
	tsc_error_t e;
	char buf[RX_BUF_SIZE];
	ReceptionResponseMessage response;

	if (skgets(sk, buf, sizeof(buf)) <= 0)
		{
			return exec_tsc_err_syscall;
		}

	ReceptionResponseMessage_init(&response);
	e = strto_ReceptionResponseMessage(&response, buf, NULL);
	switch(e)
		{
		case tsc_success:
			if(conf->verbosity >= VERBOSITY_SHOW_ERROR_DETAIL)
				{
					fputs("got:  ",conf->output);
					fputs(buf, conf->output);
					fflush(conf->output);
				}
			else
				{
					if(conf->verbosity == VERBOSITY_SHOW_ERROR_SUMMARY)
						{
							if (response_ok != response.result)
								{
									fputs("Reception response: ",conf->output);
									fputs(ReceptionResponseMessage_strerr(&response), conf->output);
									fputs("\n", conf->output);
									fflush(conf->output);
								}
						}
				}
			return response.result == response_ok ? exec_tsc_success : exec_tsc_err_response;
		case tsc_error_malformed:	/* should be time out message from delgator */
			if (conf->verbosity >= VERBOSITY_SHOW_ERROR_SUMMARY)
				{
					fputs(buf, conf->output);
					fflush(conf->output);
				}
			return exec_tsc_err_timeout;
		case tsc_error_malloc:
			perror("error");
			return exec_tsc_err_syscall;
		default:
			fprintf(stderr, "unsupported error at %s:%d\n", __FILE__, __LINE__);
			return exec_tsc_err_syscall;
		}
}


/* get a completion message from TSC */
exec_tsc_error_t
get_ce_message(int sk, exec_tsc_conf_s *conf)
{
	tsc_error_t e;
	char buf[RX_BUF_SIZE];
	CompletionResponseMessage response;

	if (skgets(sk, buf, sizeof(buf)) <= 0)
		{
			return exec_tsc_err_syscall;
		}

	CompletionResponseMessage_init(&response);
	e = strto_CompletionResponseMessage(&response, buf, NULL);
	switch(e)
		{
		case tsc_success:
			if(conf->verbosity >= VERBOSITY_SHOW_ERROR_DETAIL)
				{
					fputs("got:  ",conf->output);
					fputs(buf, conf->output);
					fflush(conf->output);
				}
			else
				{
					if(conf->verbosity == VERBOSITY_SHOW_ERROR_SUMMARY)
						{
							if (response_ok != response.result)
								{
									fputs("Completion response: ",conf->output);
									fputs(CompletionResponseMessage_strerr(&response), conf->output);
									fputs("\n", conf->output);
									fflush(conf->output);
								}
						}
				}
			return response.result == response_complete ? exec_tsc_success : exec_tsc_err_response;
		case tsc_error_malformed:	/* should be time out message from delgator */
			if (conf->verbosity >= VERBOSITY_SHOW_ERROR_SUMMARY)
				{
					fputs(buf, conf->output);
					fflush(conf->output);
				}
			return exec_tsc_err_timeout;
		case tsc_error_malloc:
			perror("strto_CompletionResponseMessage()");
			return exec_tsc_err_syscall;
		default:
			fprintf(stderr, "unsupported error at %s:%d\n", __FILE__, __LINE__);
			return exec_tsc_err_syscall;
		}
}

exec_tsc_error_t
exec_tsc_command(char *string, exec_tsc_conf_s *conf)
{
	int id = 0;
	int offset = 0;
	exec_tsc_error_t r = exec_tsc_success;

	do {
		/* commands to delegator */
		/* status */
		if (!strcmp("s", string) || !strcmp("status", string))
			{
				r = get_delegator_status(conf);
				break;
			}
		/* set id */
		if ((!strncmp("uid", string, 3) && (offset = 3)) || (!strncmp("u", string, 1) && (offset = 1)))
			{
				id = atoi(string + offset);
				r = set_delegator_id(conf, DELEGATOR_SET_UID, id);
				break;
			}
		if ((!strncmp("gid", string, 3) && (offset = 3)) || (!strncmp("g", string, 1) && (offset = 1)))
			{
				id = atoi(string + offset);
				r = set_delegator_id(conf, DELEGATOR_SET_GID, id);
				break;
			}
		/* commands to this command */
		if (!strcmp("v", string) || !strcmp("verbose", string))
			{
				conf->verbosity++;
				if (conf->verbosity >= VERBOSITY_SHOW_PROGRESS)
					fprintf(conf->output, "Increased verbosity\n");
				break;
			}
		if (!strcmp("q", string) || !strcmp("quiet", string))
			{
				conf->verbosity--;
				if (conf->verbosity >= VERBOSITY_SHOW_PROGRESS)
					fprintf(conf->output, "Decreased verbosity\n");
				break;
			}
		if (!strcmp("h", string) || !strcmp("help", string))
			{
				command_usage(stdout);
				break;
			}
		if (!strcmp("exit", string))
			{
				r = exec_tsc_quit;
				if (conf->verbosity >= VERBOSITY_SHOW_PROGRESS)
					fprintf(conf->output, "Exiting\n");
				break;
			}
		/* otherwise: command */
		{
			r = send_and_wait_tsc_command(conf, string);
			break;
		}
	} while(0);
	return r;
}

int
main(int argc, char *argv[])
{
	exec_tsc_conf_s conf = DEF_TSC_CONF;

	/* command line options */
	{
		int r;
		while((r = getopt(argc, argv, "vqp:aisth")) != -1)
			{
				switch(r)
					{
					case 'v': conf.verbosity++; break;
					case 'q': conf.verbosity--; break;
					case 'p': conf.port = atoi(optarg); break;
					case 'a': conf.async = 1; break;
					case 'i': conf.interactive = 1; break;
					case 'h':
						usage(stdout);
						fputs("\n", stdout);
						command_usage(stdout);
						return EXIT_SUCCESS;
					default:
						usage(stderr);
						return EXIT_FAILURE;
					}
			}
		if (optind >= argc && !conf.interactive)
			{
				fputs("No commands are specified.\n", stderr);
				return EXIT_FAILURE;
			}
	}

	/* send the first command from the command line */
	signal(SIGCHLD, sigchld);
	if (optind < argc)
		{
			exec_tsc_error_t r;
			char *buf;
			size_t len;
			int i;

			/* length */
			len = 0;
			for(i = optind; i < argc; i++)
				len += strlen(argv[i]) + 1;	/* one more byte for space or \0 */

			if (!(buf = (char *) malloc(len)))
				{
					perror("malloc()");
					return EXIT_FAILURE;
				}

			/* concatenate */
			buf[0] = '\0';
			for(i = optind; i < argc; i++)
				{
					strcat(buf, argv[i]);
					if (i < argc - 1) strcat(buf, " ");
				}

			/* send */
			r = exec_tsc_command(buf, &conf);

			/* result */
			if (r == exec_tsc_connection_closed || r == exec_tsc_err_malloc || r == exec_tsc_err_syscall)
				return EXIT_FAILURE;
			if (r == exec_tsc_quit)
				return EXIT_SUCCESS;
			if (!conf.interactive && r != exec_tsc_success)
				return EXIT_FAILURE;
		}

	/* interactive execution */
	if (conf.interactive)
		{
			exec_tsc_error_t r = 0;
			char buf[TX_BUF_SIZE];
			int istty;
			istty = isatty(fileno(stdin));
			do {
				if (istty) fputs("> ", stdout);
				fgets(buf, sizeof(buf), stdin);
				if (buf[0] == '\n' || buf[0] == '\0') continue;
				buf[strlen(buf) - 1] = '\0';
				r = exec_tsc_command(buf, &conf);
			} while(r != exec_tsc_quit);
		}

	return EXIT_SUCCESS;
}
