/* command-delegator.c : delegates command from local applications */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifndef HAVE_PSELECT
#include "pselect.h"
#endif

#include <melco/tsc-message.h>
#include <tws4/command-delegator.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define NLISTEN 1

#define ERROR_LOG(func, fd, string) {\
	char _buf[ERROR_LINE_SIZE];\
	snprintf(_buf, sizeof(_buf), "%s near %s:%d", (string), __FILE__, __LINE__);\
	if(func)(func)(_buf);\
	if (fd != -1) {\
		snprintf(_buf, sizeof(_buf), "delegator error: %s\n", (string));\
		_buf[sizeof(_buf)-1] = '\0';\
		write(fd, _buf, strlen(_buf));\
	}\
}

/* private types */
/* fd list */
typedef struct _fd_list {
	int fd;
	struct _fd_list *next;
} fd_list_s;

/* sequential number list */
typedef struct _seq_list {
	int fd;
	long seqnum;
	long id;
	time_t endtime;
	struct _seq_list *next;
} seq_list_s;

/* global variables */
static int _tws4_seqnum = 0;
static logger_functions_s _logger_functions = { NULL, NULL, NULL, NULL };
static int timedout;	/* timeout before getting reception response */
static int got_sigchld;
static int responsewaiting;	/* -1 when not waiting, socket fd when waiting */
/* default values of uid and gid as seen from TWS2 */
#define NOLOGIN_UID 999
#define NOLOGIN_GID 99
static int _rpc_uid = NOLOGIN_UID;	/* user ID to be sent through RPC */
static int _rpc_gid = NOLOGIN_GID;	/* group ID to be sent through RPC */

/* private functions */
static void timeout(int x);
static void sigchld(int x);
inline void _add_fd_set(int fd, fd_set *fds, int *maxfd);

inline fd_list_s* _add_fd(fd_list_s* head, int fd);
inline void _remove_fd(fd_list_s* head, int fd);
inline void _free_fd(fd_list_s* head);
inline void _close_and_unregister_fd(fd_list_s *fdhead, seq_list_s *seqhead, int fd);

inline seq_list_s* _add_seqnum(seq_list_s* head, int fd, long seqnum, long id, time_t endtime);
inline void _refresh_seqnum(seq_list_s* head, long seqnum, time_t endtime);
inline void _remove_seqnum(seq_list_s* head, long seqnum);
inline void _remove_seqnum_withfd(seq_list_s* head, int fd);
inline int _fd_from_seqnum(seq_list_s* head, long seqnum);
inline void _free_seqnum(seq_list_s* head);
inline long _count_seqnum(seq_list_s *head);

static int _read_and_redirect(fd_list_s *head, seq_list_s *seqhead, fd_set *fdset, CommandMessage_send_t sender, void *sender_args, int *tobecontinued);
static int _receive_response(int fd, fd_list_s *head, seq_list_s *seqhead);

static void _write_status(int fd, seq_list_s *seqhead);

/* private functions */
static void
timeout(int x __attribute__ ((unused)))
{
	timedout = 1;
}

static void
sigchld(int x __attribute__ ((unused)))
{
	while(waitpid(-1, NULL, WNOHANG) > 0);
	got_sigchld = 1;
	signal(SIGCHLD, sigchld);
}

inline void
_add_fd_set(int fd, fd_set *fds, int *maxfd)
{
	FD_SET(fd, fds);
	if (*maxfd < fd) *maxfd = fd;
}

inline seq_list_s*
_add_seqnum(seq_list_s* head, int fd, long seqnum, long id, time_t endtime)
{
	seq_list_s *cur, *new;
	cur = head;
	while(cur->next) cur = cur->next;
	new = (seq_list_s*) malloc(sizeof(seq_list_s));
	if (new)
		{
			new->fd = fd;
			new->next = NULL;
			new->seqnum = seqnum;
			new->id = id;
			new->endtime = endtime;
			cur->next = new;
		}
	return new;
}

inline void
_refresh_seqnum(seq_list_s* head, long seqnum, time_t endtime)
{
	seq_list_s *cur;
	cur = head->next;
	while(cur && cur->seqnum != seqnum) cur = cur->next;
	if (cur && cur->seqnum == seqnum) cur->endtime = endtime;
}

inline void
_remove_seqnum(seq_list_s* head, long seqnum)
{
	seq_list_s *cur, *next;
	cur = head;
	while(cur->next && cur->next->seqnum != seqnum) cur = cur->next;
	if (cur->next && cur->next->seqnum == seqnum)
		{
			next = cur->next->next;
			free(cur->next);
			cur->next = next;
		}
}

inline void
_remove_seqnum_withfd(seq_list_s* head, int fd)
{
	seq_list_s *cur, *next;
	cur = head;
	while(cur->next)
		{
			if (cur->next->fd == fd)
				{
					next = cur->next->next;
					free(cur->next);
					cur->next = next;
				}
			else
				{
					cur = cur->next;
				}
		}
}

inline int
_fd_from_seqnum(seq_list_s* head, long seqnum)
{
	seq_list_s *cur;
	cur = head->next;
	while(cur && cur->seqnum != seqnum) cur = cur->next;
	if (cur && cur->seqnum == seqnum)
		return cur->fd;
	else
		return -1;
}

inline fd_list_s*
_add_fd(fd_list_s* head, int fd)
{
	fd_list_s *cur, *new;
	cur = head;
	while(cur->next) cur = cur->next;
	new = (fd_list_s*) malloc(sizeof(fd_list_s));
	if (new)
		{
			new->fd = fd;
			new->next = NULL;
			cur->next = new;
		}
	return new;
}

inline void
_free_seqnum(seq_list_s* head)
{
	seq_list_s *cur, *next;
	cur = head->next;
	while(cur)
		{
			next = cur->next;
			free(cur);
			cur = next;
		}
}

inline long
_count_seqnum(seq_list_s *head)
{
	seq_list_s *cur;
	long r = 0;
	cur = head->next;
	while(cur)
		{
			r++;
			cur = cur->next;
		}
	return r;
}

inline void
_remove_fd(fd_list_s* head, int fd)
{
	fd_list_s *cur, *next;
	cur = head;
	while(cur->next && cur->next->fd != fd) cur = cur->next;
	if (cur->next && cur->next->fd == fd)
		{
			next = cur->next->next;
			free(cur->next);
			cur->next = next;
		}
}

inline void
_free_fd(fd_list_s* head)
{
	fd_list_s *cur, *next;
	cur = head->next;
	while(cur) {
		next = cur->next;
		close(cur->fd);
		free(cur);
		cur = next;
	}
}

inline void
_close_and_unregister_fd(fd_list_s *fdhead, seq_list_s *seqhead, int fd)
{
	close(fd);
	if (fdhead) _remove_fd(fdhead, fd);
	if (seqhead) _remove_seqnum_withfd(seqhead, fd);
}

static int
_read_and_redirect(fd_list_s *head, seq_list_s *seqhead, fd_set *fdset, CommandMessage_send_t sender, void *sender_args, int *tobecontinued)
{
	int nmsg;
	fd_list_s *cur;
	cur = head->next;
	nmsg = 0;
	while(cur)
		{
			if (FD_ISSET(cur->fd, fdset))
				{
					ssize_t r;
					unsigned char command;
					int close_and_continue = 0;

					/* skip space charactors */
					r = recv(cur->fd, &command, sizeof(command), MSG_PEEK);
					while(r > 0 && errno != EBADF && strchr(DELEGATOR_IGNORE, command))
						{
							read(cur->fd, &command, sizeof(command));
							r = recv(cur->fd, &command, sizeof(command), MSG_PEEK);
						}

					/* check if this is a command to myself */
					do {
						if (r < 1 || errno == EBADF)
							{
								close_and_continue = 1;
								continue;
							}
						if (command == DELEGATOR_EOT)
							{
								r = read(cur->fd, &command, sizeof(command));
								if (r < 1 || errno == EBADF)
									{
										close_and_continue = 1;
										continue;
									}
								*tobecontinued = 0;
								close_and_continue = 1;
								continue;
							}
						if (command == IAC)
							{
								unsigned char telnetcommand[IACBUFLEN];
								r = read(cur->fd, &telnetcommand, sizeof(telnetcommand));
								if (r < 1 || errno == EBADF)
									{
										close_and_continue = 1;
										continue;
									}
								if (strchr(IAC_EOTS, telnetcommand[1])) *tobecontinued = 0;
								close_and_continue = 1;
								continue;
							}
						if (command == DELEGATOR_SHOW_STATUS)
							{
								r = read(cur->fd, &command, sizeof(command));
								if (r < 1 || errno == EBADF)
									{
										close_and_continue = 1;
										continue;
									}
								_write_status(cur->fd, seqhead);
								close_and_continue = 1;
								continue;
							}
						if (command == DELEGATOR_SET_ID)
							{
								char idbuf[DELEGATOR_ID_BUF_LEN];
								char type;
								char *eptr;
								int id;
								r = read(cur->fd, &idbuf, sizeof(idbuf) - 1);
								if (r < 1 || errno == EBADF)
									{
										close_and_continue = 1;
										continue;
									}
								idbuf[r] = '\0';
								type = idbuf[1];
								id = strtol(idbuf + 2, &eptr, 10);
								if (*eptr && !strchr(DELEGATOR_IGNORE, *eptr))	/* number should end at \0, CR, or LF */
									{
										ERROR_LOG(_logger_functions.error, cur->fd, "error in stream from client");
									}
								else
									{
										switch(type)	/* following char to specify uid or gid */
											{
												case DELEGATOR_SET_UID: _rpc_uid = id; break;
												case DELEGATOR_SET_GID: _rpc_gid = id; break;
												default:
													ERROR_LOG(_logger_functions.error, cur->fd, "error in stream from client");
											}
									}
								close_and_continue = 1;
								continue;
							}
					} while(0);

					/* ensure */
					if (close_and_continue)
						{
							fd_list_s *next;
							next = cur->next;
							_close_and_unregister_fd(head, seqhead, cur->fd);
							cur = next;
							continue;
						}

					/* read command and directed it to TSC */
					{
						fd_list_s *next;
						tsc_error_t tsc;
						CommandMessage msg;
						
						/* parse command */
						CommandMessage_init(&msg);
						tsc = read_CommandMessage(&msg, cur->fd);
						if (tsc != tsc_success)
							{
								CommandMessage_decref(&msg);
								if (_logger_functions.error)
									{
										switch(tsc)
											{
												case tsc_error_nostream:
													ERROR_LOG(_logger_functions.error, cur->fd, "error in stream");
													break;
												case tsc_error_malloc:
													ERROR_LOG(_logger_functions.error, cur->fd, "out of memory");
													break;
												case tsc_error_malformed:
													ERROR_LOG(_logger_functions.error, cur->fd, "malformed message header");
													break;
												default:
													ERROR_LOG(_logger_functions.error, cur->fd, "unknown error");
													break;
											}
									}
								next = cur->next;
								_close_and_unregister_fd(head, seqhead, cur->fd);
								cur = next;
								continue;
							}

						/* allocate sequential number */
						CommandMessage_set_seqnum(&msg, _tws4_seqnum);
						if (!_add_seqnum(seqhead, cur->fd, _tws4_seqnum, msg.id, time(NULL) + FD_FORGET_AFTER))
							{
								ERROR_LOG(_logger_functions.error, cur->fd, "out of memory");
								next = cur->next;
								_close_and_unregister_fd(head, seqhead, cur->fd);
								cur = next;
								continue;
							}
						_tws4_seqnum++;
						if (_tws4_seqnum > MAX_SEQNUM) _tws4_seqnum = 0;

						/* send command */
						errno = 0;
						if (sender(&msg, _rpc_uid, _rpc_gid, sender_args) > 0)
							{
								if (_logger_functions.command) _logger_functions.command(&msg);
							}
						else
							{
								char *message;
								ssize_t res;
								if (errno)
									message = strerror(errno);
								else
									message = "Error in sending command";
								ERROR_LOG(_logger_functions.error, cur->fd, message);
								next = cur->next;
								_close_and_unregister_fd(head, seqhead, cur->fd);
								cur = next;
								continue;
							}
						if (fork() == 0)
							{
								ssize_t res;
								res = write(cur->fd, CommandMessage_tostr(&msg), CommandMessage_strlen(&msg));
								if (res > 0) write(cur->fd, "\n", 1);
								_free_fd(head);
								_free_seqnum(seqhead);
								CommandMessage_decref(&msg);
								exit(res > 0 ? EXIT_SUCCESS : EXIT_FAILURE);
							}
						CommandMessage_decref(&msg);
						responsewaiting = cur->fd;
						signal(SIGALRM, timeout);
						alarm(RESPONSE_TIMEOUT);
						nmsg++;
					}
				}
				cur = cur->next;
		}
	return nmsg;
}

static int
_receive_response(int fd, fd_list_s *head, seq_list_s *seqhead)
{
	char *data;
	char type[3];

	/* receive message */
	data = read_Message(fd);
	if (data == NULL) 
		{
			ERROR_LOG(_logger_functions.error, -1, "out of memory");
			return -1;
		}
	if (data == (char *) 1)
		{
			ERROR_LOG(_logger_functions.error, -1, strerror(errno));
			return 0;
		}
	if (data == (char *) 2)
		{
			ERROR_LOG(_logger_functions.error, -1, "malformed header from TSC");
			return 0;
		}
	if (data == (char *) 3)
		{
			ERROR_LOG(_logger_functions.error, -1, "connection from TSC closed");
			return -1;
		}

	/* process message */
	message_header_type((message_header_s*) data, type);
	while(1)
		{
			if (!strcmp(type, "CA"))	/* Reception Response */
				{
					int output_fd;
					ReceptionResponseMessage msg;

					alarm(0);
					ReceptionResponseMessage_init(&msg);
					strto_ReceptionResponseMessage(&msg, data, free);
					output_fd = _fd_from_seqnum(seqhead, msg.seqnum);
					responsewaiting = -1;
					if (msg.result == response_ok)
						_refresh_seqnum(seqhead, msg.seqnum, msg.endtime + COMMANDMESSAGE_COMPLETION_TIMEOUT + FD_FORGET_AFTER);
					if (_logger_functions.reception) _logger_functions.reception(&msg);
					if (output_fd != -1)
						{
							if (fork() == 0)
								{
									ssize_t r;
									r = write(output_fd, ReceptionResponseMessage_tostr(&msg), ReceptionResponseMessage_strlen(&msg));
									if (r > 0) write(output_fd, "\n", 1);
									_free_fd(head);
									_free_seqnum(seqhead);
									ReceptionResponseMessage_decref(&msg);
									exit(r > 0 ? EXIT_SUCCESS : EXIT_FAILURE);
								}
							if (msg.result != response_ok)
								_close_and_unregister_fd(head, seqhead, output_fd);
						}
					else
						{
							_remove_seqnum(seqhead, msg.seqnum);
						}

					ReceptionResponseMessage_decref(&msg);
					break;
				}
			if (!strcmp(type, "CE"))	/* Completion Response */
				{
					int output_fd;
					CompletionResponseMessage msg;

					CompletionResponseMessage_init(&msg);
					strto_CompletionResponseMessage(&msg, data, free);
					output_fd = _fd_from_seqnum(seqhead, msg.seqnum);
					_remove_seqnum(seqhead, msg.seqnum);
					if (_logger_functions.completion) _logger_functions.completion(&msg);
					if (msg.result == response_complete)
						{
							const char *argptr = NULL;
							size_t arglen;
							int gid;
							switch(msg.id)
								{
									case 0x1A1901:	/* login */
										if (!CompletionResponseMessage_arg(&msg, &argptr, &arglen)) break;
										if (arglen < 1) break;
										gid = atoi(argptr);
										argptr += arglen;
										if (!CompletionResponseMessage_arg(&msg, &argptr, &arglen)) break;
										if (arglen < 1) break;
										_rpc_uid = atoi(argptr);
										_rpc_gid = gid;
										break;
									case 0x1A1902:	/* logout */
										_rpc_uid = NOLOGIN_UID;
										_rpc_gid = NOLOGIN_GID;
										break;
									default:
										break;
								}
						}
					if (output_fd != -1 && fork() == 0)
						{
							ssize_t r;
							r = write(output_fd, CompletionResponseMessage_tostr(&msg), CompletionResponseMessage_strlen(&msg));
							if (r > 0) write(output_fd, "\n", 1);
							_free_fd(head);
							_free_seqnum(seqhead);
							CompletionResponseMessage_decref(&msg);
							exit(r > 0 ? EXIT_SUCCESS : EXIT_FAILURE);
						}
					_close_and_unregister_fd(head, seqhead, output_fd);
					CompletionResponseMessage_decref(&msg);
					break;
				}

			/* default */
			if (_logger_functions.error)
				{
					char buf[ERROR_LINE_SIZE];
					snprintf(buf, sizeof(buf), "unknown message type:%s", type);
					ERROR_LOG(_logger_functions.error, -1, buf);
				}
			free(data);
			return 0;
		}

	return 1;
}

static void
_write_status(int fd, seq_list_s *head)
{
	int len;
	char buf[ERROR_LINE_SIZE];
	len = snprintf(buf, sizeof(buf), "delegator status: pid:%d uid:%d gid:%d next-seq-num:%d waiting-completion:%ld\n", getpid(), _rpc_uid, _rpc_gid,_tws4_seqnum, _count_seqnum(head));
	if (len < 0 || len >= (int) sizeof(buf))
		{
			buf[sizeof(buf) - 1] = '\0';
			len = sizeof(buf) - 1;
		}
	write(fd, buf, len);
}

/* public functions */
long
command_delegator(
	struct sockaddr_in *myaddr,
	/* worker functions to send and receive commands */
	CommandMessage_send_t sender,
	void *sender_args,
	message_waiter_t waiter,
	void *waiter_args,
	/* callbacks for log: this or some member can be NULL */
	logger_functions_s *logger_functions
)
{
	long nmsg = 0;
	int waiter_fd[2];
	int listner;
	int tobecontinued;
	pid_t waiter_pid;
	int maxfd;
	fd_list_s connections = {-1, NULL};
	seq_list_s seqnums = {-1, -1, 0, 0, NULL};
	fd_set readfds, suspendedfds;
	sigset_t sigmask, orig_sigmask;
	struct timespec loop_period;

	/* timeout for pselect */
	loop_period.tv_sec = LOOP_PERIOD;
	loop_period.tv_nsec = 0;

	/* register loggers */
	if (logger_functions) _logger_functions = *logger_functions;

	/* create a worker process `waiter' to receive messages from TSC */
	if (pipe(waiter_fd)) return -1;
	waiter_pid = fork();
	if (waiter_pid == -1) return -1;
	if (waiter_pid == 0)
		{
			/* waiter process */
			close(waiter_fd[0]);
			waiter(waiter_fd[1], waiter_args);
		}
	else
		{
			close(waiter_fd[1]);
		}

	/* create a listner socket to wait for connections from clients on localhost */
	if ((listner = socket(AF_INET, SOCK_STREAM, 0)) == -1)
		{
			kill(waiter_pid, SIGKILL);
			waitpid(waiter_pid, NULL, 0);
			return -1;
		}

	/* avoid TIME_WAIT */
	{
		int val = 1;
		if (setsockopt(listner, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val)) < 0)
			{
				kill(waiter_pid, SIGKILL);
				waitpid(waiter_pid, NULL, 0);
				return -1;
			}
	}

	/* get the listner socket ready */
	if (bind(listner, (struct sockaddr *) myaddr, sizeof(struct sockaddr_in)))
		{
			close(listner);
			kill(waiter_pid, SIGKILL);
			waitpid(waiter_pid, NULL, 0);
			return -1;
		}
	if (listen(listner, NLISTEN))
		{
			close(listner);
			kill(waiter_pid, SIGKILL);
			waitpid(waiter_pid, NULL, 0);
			return -1;
		}

	/* receive SIGALRM and SIGCHLD only during select() */
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGALRM);
	sigaddset(&sigmask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &sigmask, &orig_sigmask);

	signal(SIGCHLD, sigchld);

	/* ignore SIGPIPE */
	signal(SIGPIPE, SIG_IGN);

	/* event loop */
	FD_ZERO(&suspendedfds);
	tobecontinued = 1;
	timedout = 0;
	responsewaiting = -1;
	got_sigchld = 0;
	while(tobecontinued)
		{
			/* wait for: */
			FD_ZERO(&readfds);
			maxfd = 0;
			/* - response from TSC */
			_add_fd_set(waiter_fd[0], &readfds, &maxfd);
			/* - socket connection */
			_add_fd_set(listner, &readfds, &maxfd);
			/* - commands from connected sockets */
			{
				fd_list_s *cur;
				cur = connections.next;
				while(cur)
					{
						if (!FD_ISSET(cur->fd, &suspendedfds))
							{
								_add_fd_set(cur->fd, &readfds, &maxfd);
							}
						cur = cur->next;
					}
			}

			/* wait */
			errno = 0;
			if (pselect(maxfd + 1 ,&readfds, NULL, NULL, &loop_period, &orig_sigmask) < 0)
				{
					if (errno == EINTR)	/* got interruption */
						{
							if (timedout)	/* SIGALRM */
								{
									/* notify */
									ERROR_LOG(_logger_functions.error, responsewaiting, "timeout before receiving Reception Response Message");
									_close_and_unregister_fd(&connections, &seqnums, responsewaiting);
									timedout = 0;
									/* process suspended inputs */
									nmsg += _read_and_redirect(&connections, &seqnums, &suspendedfds, sender, sender_args, &tobecontinued);
									FD_ZERO(&suspendedfds);
									responsewaiting = -1;
									continue;
								}
							if (got_sigchld)	/* SIGCHLD */
								{
									got_sigchld = 0;
									continue;
								}
						}
					ERROR_LOG(_logger_functions.error, -1, strerror(errno));	/* unknown */
						continue;
				}

			/* response from TSC */
			if (FD_ISSET(waiter_fd[0], &readfds))
				{
					int r;
					r = _receive_response(waiter_fd[0], &connections, &seqnums);
					if (r < 0) tobecontinued = 0;
					nmsg += r;
				}

			/* from socket: new connection */
			if (FD_ISSET(listner, &readfds))
				{
					int sk;
					struct sockaddr_in peer;
					unsigned int paddrlen;

					paddrlen = sizeof(peer);
					sk = accept(listner, (struct sockaddr*)&peer, &paddrlen);
					if (sk != -1)
						{
							if (!_add_fd(&connections, sk))
								{
									ERROR_LOG(_logger_functions.error, sk, "out of memory");
								}
						}
					else
						{
							ERROR_LOG(_logger_functions.error, -1, strerror(errno));
						}
				}

			/* from socket that has already connected */
			if (-1 == responsewaiting)
				{
					/* process commands from known clients */
					nmsg += _read_and_redirect(&connections, &seqnums, &readfds, sender, sender_args, &tobecontinued);
				}
			else
				{
					/* wait for an CA remembering fds into a list */
					fd_list_s *cur;
					cur = connections.next;
					while(cur)
						{
							if (FD_ISSET(cur->fd, &readfds)) FD_SET(cur->fd, &suspendedfds);
							cur = cur->next;
						}
				}

			/* clean up old sequential numbers */
			{
				seq_list_s *cur, *tmp;
				time_t now;
				time(&now);
				cur = seqnums.next;
				while(cur)
					{
						if (cur->endtime < now)
							{
								if (_logger_functions.error)
									{
										char buf[ERROR_LINE_SIZE];
										snprintf(buf, sizeof(buf), "Completion message for %0*lX #%0*lX timedout", COMMANDMESSAGE_ID_SIZE, cur->id, COMMANDMESSAGE_SEQ_SIZE, cur->seqnum);
										ERROR_LOG(_logger_functions.error, cur->fd, buf);
									}
								tmp = cur->next;
								_close_and_unregister_fd(&connections, &seqnums, cur->fd);
								cur = tmp;
							}
						else
							{
								cur = cur->next;
							}
					}
			}
		}

	/* clean up */
	if (_logger_functions.error)
		{
			char buf[ERROR_LINE_SIZE];
			snprintf(buf, sizeof(buf), "command-delegator ending after processing %ld messages", nmsg);
			ERROR_LOG(_logger_functions.error, -1, buf);
		}
	_free_fd(&connections);
	_free_seqnum(&seqnums);
	close(listner);
	close(waiter_fd[0]);
	kill(waiter_pid, SIGKILL);
	waitpid(waiter_pid, NULL, 0);
	while(waitpid(-1, NULL, WNOHANG) > 0);
	signal(SIGPIPE, SIG_DFL);
	return nmsg;
}
