/* tsc-message.c : functions in tsc-message.h not included in status-bank.c or memory-manage.c */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#define __USE_XOPEN
#include <time.h>

#include <melco/tsc-message.h>
#include <melco/tree-hash.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

/* private functions */
inline void _init_scan_table(void);
inline char* _next_instrument_status_header(char* cur, char *end);
inline int _at_status_header(char* cur, char *end);
inline int _MonitorDataMessage_register(MonitorDataMessage* parent, void* arg, struct melco_tree_s* info, instrument_status_header_s* h);
const char* _message_arg(const char *pars, int parlen, const char **startptr, size_t *length);

/* parameters */
#define PARAMETER_ARG_IGNORED "%"
#define PARAMETER_ARG_DELIMITOR " "

/* error codes: NULL:malloc 1:stream 2:malformed header 3:EBADF */
inline char* _read_Message(int fd);	/* returns a malloced memory filled with message read from fd, or NULL for an error */

/* instrument status scanner */
char *_shortid_chars = "0123456789ABCDEF";
char *_time_chars = "0123456789";
static int _shortid_table[256], _time_table[256];
int _table_init = 0;
/* the 19 byte header */
static int* _char_tables[sizeof(instrument_status_header_s)] = {
	/* 4-byte short ID */
	_shortid_table, _shortid_table, _shortid_table, _shortid_table,
	/* 15-byte rx time */
	_time_table, _time_table, _time_table, _time_table, _time_table,
	_time_table, _time_table, _time_table, _time_table, _time_table,
	_time_table, _time_table, _time_table, _time_table, _time_table
};

/* initialize the lookup tables */
inline void
_init_scan_table(void)
{
	register unsigned int c;
	if (!_table_init)
		{
			for(c = 0; c < 256; c++)
				{
					_shortid_table[c] = 0;
					_time_table[c] = 0;
				}
			for(c = 0; c < strlen(_shortid_chars); c++)
				{
					_shortid_table[(unsigned int) _shortid_chars[c]] = 1;
				}
			for(c = 0; c < strlen(_time_chars); c++)
				{
					_time_table[(unsigned int) _shortid_chars[c]] = 1;
				}
			_table_init = 1;
		}
}

inline char*
_next_instrument_status_header(char* cur, char *end)
{
	register unsigned int c;
	struct timeval time_check;
	ASC15_TIME t;

	/* scan for the next instrument status header */
	c = 0;
	while(cur < end && c < sizeof(instrument_status_header_s))
		{
			while(cur < end && c < sizeof(instrument_status_header_s))
				{
					if (_char_tables[c][(unsigned int) *((unsigned char*) cur)]) c++; else c = 0;
					cur++;
				}
			if (c == sizeof(instrument_status_header_s))
				{
					memcpy(&t, &((((instrument_status_header_s *) (cur - sizeof(instrument_status_header_s)))->s.rxtime)), sizeof(t));
					if (ASC15_TIME_to_t(&t, 0, 15, &time_check))
						{
							/* valid time stamp */
							return cur - sizeof(instrument_status_header_s);
						}
					else
						{
							/* invalid time stamp */
							c = 0;
							cur = cur - sizeof(instrument_status_header_s) + 1;
						}
				}
		}

	return NULL;
}

inline int
_at_status_header(char* cur, char *end)
{
	register unsigned int c;

	c = 0;
	while(cur < end && c < sizeof(instrument_status_header_s) && _char_tables[c][(unsigned int) *cur])
		{
			cur++;
		}

	return (c == sizeof(instrument_status_header_s));
}

inline char*
_read_Message(int fd)
{
	ssize_t r;
	int content_length;
	message_header_s *data;
	char *types[] = SUPPORTED_MESSAGE_TYPES;

	/* allocate memory for the header */
	data = (message_header_s*) malloc(sizeof(message_header_s));
	if (!data) return NULL;

	/* read the header */
	if ((r = read(fd, data, sizeof(message_header_s))) != sizeof(message_header_s))
		{
			free(data);
			if (r == 0 || errno == EBADF)
				return (char *) 3;
			else
				return (char *) 1;
		}

	/* check type */
	{
		int i;
		for(i = 0; types[i]; i++)
			{
				if (memcmp(types[i], data->s.type, sizeof(data->s.type)) == 0) break;
			}
		if (!types[i])
			{
				free(data);
				return (char*) 2;
			}
	}

	/* check length */
	content_length = message_header_length(data);

	/* allocate memory for the content */
	data = (message_header_s*) realloc(data, sizeof(message_header_s) + content_length);
	if (!data) return NULL;

	/* read the content */
	if (read(fd, (char*) data + sizeof(message_header_s), content_length) != content_length)
		{
			free(data);
			if (errno == EBADF)
				return (char *) 3;
			else
				return (char *) 1;
		}

	return (char *) data;
}

char *
read_Message(int fd)
{
	return _read_Message(fd);
}

int
instrument_status_total_size(const char *type, const unsigned int shortid)
{
	char key[MELCO_TREE_MAX_WORD_LENGTH+1];
	struct melco_tree_s* info;

	treekey(key, type, shortid);
	info = melco_tree_lookup(key, strlen(key));
	if (!info) return -1;	/* error! */
	return sizeof(instrument_status_header_s) + info->bytes;
}

void
InstrumentStatus_set(InstrumentStatus *self, instrument_status_header_s* d, void (*d_free)(void *ptr), struct melco_tree_s* info)
{
	self->d = d;
	self->d_free = d_free;
	self->shortid = instrument_status_header_shortid(d);
	if (self->parent)
		{
			message_header_status_content_type(self->parent->d, self->type);
			self->length = sizeof(instrument_status_header_s) + info->bytes;
		}
	else
		{
			self->type[0] = '\0';
			self->length = 0;
		}
}

StatusEntry*
InstrumentStatus_get_status_entry(InstrumentStatus *self, const char *keyword)
{
	StatusEntry *entry;
	struct melco_status_s* info;
	info = melco_status_lookup(keyword, strlen(keyword));
	if (!info) return NULL;
	if (!InstrumentStatus_is(self, info->type, info->shortid)) return NULL;
	entry = StatusEntry_alloc(self);
	if (!entry) return NULL;
	entry->d = (char*) self->d + sizeof(instrument_status_header_s) + info->offset;
	entry->d_free = NULL;
	entry->info = info;
	return entry;
}

tsc_error_t
read_MonitorDataMessage(MonitorDataMessage* dst, int fd)
{
	char *data;
	errno = 0;
	data = _read_Message(fd);
	if (data == NULL) return tsc_error_malloc;
	if (data == (char *) 1) return tsc_error_stream;
	if (data == (char *) 2) return tsc_error_malformed;
	if (data == (char *) 3) return tsc_error_nostream;
	return strto_MonitorDataMessage(dst, data, free);
}

tsc_error_t
fread_MonitorDataMessage_header(MonitorDataMessage* dst, FILE* stream, int rewind_after_try)
{
	long len, offset;

	/* allocate memory for the header and one-byte type */
	if (!(dst->d))
		{
			dst->d = (message_header_s*) malloc(sizeof(message_header_s) + 1);
			if (!(dst->d))
				{
					return tsc_error_malloc; 
				}
			dst->d_free = free;
		}

	/* read the message header and one-byte type */
	len = sizeof(message_header_s) + 1;
	offset = -len;
	if (fread(dst->d, len, 1, stream) != 1)
		{
			MonitorDataMessage_free_if_noref(dst);
			if (feof(stream))
				return tsc_error_nostream;
			else
				return tsc_error_stream;
		}

	/* check message type */
	if (strncmp(MONITOR_DATA_MESSAGE_TYPE, dst->d->s.type, strlen(MONITOR_DATA_MESSAGE_TYPE)))
		{
			MonitorDataMessage_free_if_noref(dst);
			if (rewind_after_try) fseek(stream, offset, SEEK_CUR);
			return tsc_error_malformed;
		}

	/* type */
	dst->type[0] = *((char *)(dst->d)+sizeof(message_header_s));
	switch(dst->type[0])
		{
			case 'L':
				dst->d = (message_header_s*) realloc(dst->d, sizeof(message_header_s) + 2);
				if(!dst->d)
					{
						if (rewind_after_try) fseek(stream, offset, SEEK_CUR);
						dst->d_free = NULL;
						return tsc_error_malloc;
					}
				len = 1;
				if (fread((char *)(dst->d)+sizeof(message_header_s)+1, len, 1, stream) != 1)
					{
						MonitorDataMessage_free_if_noref(dst);
						if (rewind_after_try) fseek(stream, offset, SEEK_CUR);
						if (feof(stream))
							return tsc_error_nostream;
						else
							return tsc_error_stream;
					}
				offset -= len;
				dst->type[1] = *((char *)(dst->d)+sizeof(message_header_s)+1);
				dst->type[2] = '\0';
				dst->typelen = 2;
				break;
			case 'S':
				/* fall through */
			case 'E':
				dst->type[1] = '\0';
				dst->typelen = 1;
				break;
			default:
				MonitorDataMessage_free_if_noref(dst);
				if (rewind_after_try) fseek(stream, offset, SEEK_CUR);
				return tsc_error_malformed;
		}
	if (rewind_after_try) fseek(stream, offset, SEEK_CUR);
	return tsc_success;
}

tsc_error_t
fread_MonitorDataMessage(MonitorDataMessage* dst, FILE* stream, message_blocking_t blocking, const long blocksize)
{
	tsc_error_t r;
	int content_length;
	long real_blocksize = 0, offset;

	assert(dst->refcount < 1);

	/* read and check the current header */
	r = fread_MonitorDataMessage_header(dst, stream, 0);
	if (tsc_success != r) return r;

	/* content length */
	content_length = message_header_length(dst->d);

	/* block size */
	switch(blocking)
		{
			case block_none:
			case block_none_or_fujitsu:
				real_blocksize = sizeof(message_header_s) + content_length;
				break;
			case block_exact:
				real_blocksize = blocksize;
				break;
			case block_fujitsu:
				switch(dst->type[0])
					{
						case 'L':
							real_blocksize = FUJITSU_TSCL_BLOCK_SIZE;
							break;
						case 'S':
							real_blocksize = FUJITSU_TSCS_BLOCK_SIZE;
							break;
						case 'E':
							real_blocksize = FUJITSU_TSCV_BLOCK_SIZE;
							break;
					}
				break;
			case block_tws4:
				switch(dst->type[0])
					{
						case 'L':
							real_blocksize = TWS4_TSCL_BLOCK_SIZE;
							break;
						case 'S':
							real_blocksize = TWS4_TSCS_BLOCK_SIZE;
							break;
						case 'E':
							real_blocksize = TWS4_TSCV_BLOCK_SIZE;
							content_length = TWS4_TSCV_BLOCK_SIZE - sizeof(message_header_s);
							break;
					}
				break;
			default:
				assert(0);
		}
	dst->length = content_length;

	/* allocate memory for the content */
	dst->d = (message_header_s*) realloc(dst->d, sizeof(message_header_s) + content_length);
	if(!dst->d)
		{
			dst->d_free = NULL;
			MonitorDataMessage_free(dst);
			return tsc_error_malloc;
		}

	/* read the content */
	if (fread((char *)(dst->d) + sizeof(message_header_s) + dst->typelen, content_length - dst->typelen, 1, stream) != 1)
		{
			MonitorDataMessage_free_if_noref(dst);
			if (feof(stream))
				return tsc_error_nostream;
			else
				return tsc_error_stream;
		}

	/* move to the next block */
	if (block_none_or_fujitsu == blocking)
		{
			MonitorDataMessage t;
			tsc_error_t peek_result;

			MonitorDataMessage_init(&t);
			peek_result = fread_MonitorDataMessage_header(&t, stream, 1);
			MonitorDataMessage_decref(&t);

			if (tsc_error_malformed == peek_result)
				{
					switch(dst->type[0])
						{
							case 'L':
								real_blocksize = FUJITSU_TSCL_BLOCK_SIZE;
								break;
							case 'S':
								real_blocksize = FUJITSU_TSCS_BLOCK_SIZE;
								break;
							case 'E':
								real_blocksize = FUJITSU_TSCV_BLOCK_SIZE;
								break;
						}
				}
		}

	offset = real_blocksize - content_length - sizeof(message_header_s);
	if (offset > 0) fseek(stream, offset, SEEK_CUR);

	return tsc_success;
}

tsc_error_t
strto_MonitorDataMessage(MonitorDataMessage* dst, char *src, void (*src_free)(void* ptr))
{
	/* get the header in hand */
	dst->d = (message_header_s*) src;
	dst->d_free = src_free;

	/* check message type */
	if (strncmp(MONITOR_DATA_MESSAGE_TYPE, dst->d->s.type, strlen(MONITOR_DATA_MESSAGE_TYPE)))
		{
			return tsc_error_malformed;
		}

	/* type */
	dst->type[0] = *((char *)(dst->d)+sizeof(message_header_s));
	switch(dst->type[0])
		{
			case 'L':
				dst->type[1] = *((char *)(dst->d)+sizeof(message_header_s)+1);
				dst->type[2] = '\0';
				dst->typelen = 2;
				break;
			case 'S':
				/* fall through */
			case 'E':
				dst->type[1] = '\0';
				dst->typelen = 1;
				break;
			default:
				return tsc_error_malformed;
		}

	/* content length */
	dst->length = message_header_length(dst->d);
	return tsc_success;
}

int
MonitorDataMessage_scanner(MonitorDataMessage* self, void* arg, scan_option_t opt, MonitorDataMessage_scanner_callback_t callback)
{
	int r, count;
	register char *cur, *end, *buf;
	char key[MELCO_TREE_MAX_WORD_LENGTH+1];
	struct melco_tree_s* info;

	if (!self->d || self->length <= 0) return 0;
	MonitorDataMessage_incref(self);

	_init_scan_table();

	/* data limits */
	cur = (char *) self->d + sizeof(message_header_s) + self->typelen;
	end = cur + self->length - self->typelen;

	/* find the first header */
	switch(opt)
		{
			case scan_all:
			case scan_guess_and_all:
				cur = _next_instrument_status_header(cur, end);
				break;
			default:
				;
		}

	count = 0;
	while(cur && cur < end)
		{
			/* check coming status and register */
			treekey_str(key, self->type, ((instrument_status_header_s *) cur)->s.shortid);
			info = melco_tree_lookup(key, strlen(key));
			if (info)
				{
					r = callback(self, (void*) arg, info, (instrument_status_header_s*) cur);
					if (r < 0)
						{
							count = r;
							goto finish;
						}
					count += r;
				}

			/* move to the next bunch or stop */
			switch(opt)
				{
					case scan_guess_and_all:
						if (info)
							{
								buf = cur;
								/* first belive the registered length */
								cur += sizeof(instrument_status_header_s) + info->bytes;
								if (_at_status_header(cur, end)) break;

								cur = buf;
							}
						/* fall through if we can not guess the length or guess was wrong */
					case scan_all:
						cur = _next_instrument_status_header(cur + sizeof(instrument_status_header_s), end);
						break;
					case scan_strict:
						if (!info)
							{
								count = -1;
								goto finish;
							}
						cur += sizeof(instrument_status_header_s) + info->bytes;
						break;
					default:
						assert(0);
				}
		}
	finish:
	MonitorDataMessage_decref(self);
	return count;
}

inline int
_MonitorDataMessage_register(MonitorDataMessage* parent, void* arg, struct melco_tree_s* info, instrument_status_header_s* h)
{
	InstrumentStatus *inst;
	inst = InstrumentStatus_alloc(parent);
	if (!inst) return -1;
	InstrumentStatus_set(inst, h, NULL, info);
	if ((((StatusBank*) arg)->data)[info->index])
		InstrumentStatus_decref((((StatusBank*) arg)->data)[info->index]);
	(((StatusBank*) arg)->data)[info->index] = inst;
	return 1;
}

int
MonitorDataMessage_scan(MonitorDataMessage* self, StatusBank* dst, scan_option_t opt)
{
	return MonitorDataMessage_scanner(self, dst, opt, _MonitorDataMessage_register);
}

StatusEntry*
StatusBank_get_status_entry(StatusBank *self, const char *keyword)
{
	StatusEntry *entry;
	InstrumentStatus *inst;
	struct melco_status_s* info;
	info = melco_status_lookup(keyword, strlen(keyword));
	if (!info) return NULL;
	inst = StatusBank_get_inst(self, info->type, info->shortid);
	if (!inst) return NULL;
	entry = StatusEntry_alloc(inst);
	if (!entry)
		{
			InstrumentStatus_free(inst);
			return NULL;
		}
	entry->d = (char*) inst->d + sizeof(instrument_status_header_s) + info->offset;
	entry->d_free = NULL;
	entry->info = info;
	return entry;
}

double*
StatusEntry_to_d(const StatusEntry* entry, double* dst)
{
	if (!entry) return NULL;
	if (!entry->info) return NULL;
	if (!entry->info->to_d) return NULL;
	if (!(entry->info->to_d)(entry->d, entry->info->mask, entry->info->bytes, (void*) dst)) return NULL;
	return dst;
}

long long*
StatusEntry_to_i(const StatusEntry* entry, long long* dst)
{
	if (!entry) return NULL;
	if (!entry->info) return NULL;
	if (!entry->info->to_i) return NULL;
	if (!(entry->info->to_i)(entry->d, entry->info->mask, entry->info->bytes, (void*) dst)) return NULL;
	return dst;
}

char*
StatusEntry_to_s(const StatusEntry* entry, char* dst)
{
	if (!entry) return NULL;
	if (!entry->info) return NULL;
	if (!entry->info->to_s) return NULL;
	if (!(entry->info->to_s)(entry->d, entry->info->mask, entry->info->bytes, (void*) dst)) return NULL;
	return dst;
}

int
StatusEntry_to_s_length(const StatusEntry* entry)
{
	int r;
	if (!entry) return 0;
	if (!entry->info) return 0;
	if (!entry->info->to_s_length) return 0;
	r = (entry->info->to_s_length)(entry->d, entry->info->mask, entry->info->bytes, NULL);
	return r;
}

char*
StatusEntry_to_hex(const StatusEntry* entry, char* dst)
{
	if (!entry) return NULL;
	if (!entry->info) return NULL;
	if (!to_hex(entry->d, entry->info->mask, entry->info->bytes, (void*) dst)) return NULL;
	return dst;
}

int
StatusEntry_to_hex_length(const StatusEntry* entry)
{
	int r;
	if (!entry) return 0;
	if (!entry->info) return 0;
	r = to_hex_length(entry->d, entry->info->mask, entry->info->bytes, NULL);
	return r;
}

struct timeval*
StatusEntry_to_tv(const StatusEntry* entry, struct timeval* dst)
{
	if (!entry) return NULL;
	if (!entry->info) return NULL;
	if (!entry->info->to_t) return NULL;
	if (!(entry->info->to_t)(entry->d, entry->info->mask, entry->info->bytes, (void*) dst)) return NULL;
	return dst;
}

time_t*
StatusEntry_to_t(const StatusEntry* entry, time_t* dst)
{
	struct timeval tv;
	if (!StatusEntry_to_tv(entry, &tv)) return NULL;
	*dst = tv.tv_sec;
	return dst;
}

/* reception time from the instrument header */
struct timeval*
StatusEntry_rxtime_tv(const StatusEntry* entry, struct timeval* dst)
{
	if (!(ASC15_TIME_to_t((void*) &(entry->parent->d->s.rxtime), 0, 15, dst))) return NULL;
	return dst;
}

time_t*
StatusEntry_rxtime(const StatusEntry* entry, time_t* dst)
{
	struct timeval tv;
	if (!StatusEntry_rxtime_tv(entry, &tv)) return NULL;
	*dst = tv.tv_sec;
	return dst;
}

double*
StatusEntry_rxtime_d(const StatusEntry* entry, double* dst)
{
	struct timeval tv;
	if (!StatusEntry_rxtime_tv(entry, &tv)) return NULL;
	*dst = (double) tv.tv_sec + 1e-6 * tv.tv_usec;
	return dst;
}

struct timeval*
InstrumentStatus_rxtime_tv(const InstrumentStatus* inst, struct timeval* dst)
{
	if (!(ASC15_TIME_to_t((void*) &(inst->d->s.rxtime), 0, 15, dst))) return NULL;
	return dst;
}

time_t*
InstrumentStatus_rxtime(const InstrumentStatus* inst, time_t* dst)
{
	struct timeval tv;
	if (!InstrumentStatus_rxtime_tv(inst, &tv)) return NULL;
	*dst = tv.tv_sec;
	return dst;
}

double*
InstrumentStatus_rxtime_d(const InstrumentStatus* inst, double* dst)
{
	struct timeval tv;
	if (!InstrumentStatus_rxtime_tv(inst, &tv)) return NULL;
	*dst = (double) tv.tv_sec + 1e-6 * tv.tv_usec;
	return dst;
}

char*
InstrumentStatus_treekey(const InstrumentStatus* ptr, char *seven_byte_buf)
{
	treekey(seven_byte_buf, ptr->type, ptr->shortid);
	return seven_byte_buf;
}

int
InstrumentStatus_is(const InstrumentStatus* self, const char *type, const unsigned int shortid)
{
	char selfkey[MELCO_TREE_MAX_WORD_LENGTH+1], otherkey[MELCO_TREE_MAX_WORD_LENGTH+1];
	treekey(selfkey, self->type, self->shortid);
	treekey(otherkey, type, shortid);
	return !strncmp(selfkey, otherkey, sizeof(selfkey));
}

const char*
_message_arg(const char *pars, int parlen, const char **startptr, size_t *length)
{
	const char *cptr;

	/* starting */
	if (!*startptr)
		{
			/* first argument */
			if (parlen < 1)
				{
					*startptr = NULL;
					*length = 0;
					return *startptr;
				}
			cptr = pars;
			if (strchr(PARAMETER_ARG_IGNORED, *cptr))
				{
					/* first arg is an empty string */
					*startptr = cptr;
					*length = 0;
					return *startptr;
				}
		}
	else
		{
			/* second+ argument */
			cptr = *startptr;
			/* skip %s */
			while(cptr - pars < parlen && strchr(PARAMETER_ARG_IGNORED, *cptr)) cptr++;
			/* skip delimitors */
			while(cptr - pars < parlen && strchr(PARAMETER_ARG_DELIMITOR, *cptr)) cptr++;
			if (cptr - pars >= parlen)
				{
					*startptr = NULL;
					*length = 0;
					return *startptr;
				}
		}

	/* find end */
	*startptr = cptr;
	while(cptr - pars < parlen && !strchr(PARAMETER_ARG_IGNORED, *cptr) && !strchr(PARAMETER_ARG_DELIMITOR, *cptr)) cptr++;
	*length = cptr - *startptr;

	return *startptr;
}

tsc_error_t
strto_CommandMessage(CommandMessage* dst, char *src, void (*src_free)(void* ptr))
{
	/* get the header in hand */
	if (dst->d_free) dst->d_free(dst->d);
	dst->d = (message_header_s*) src;
	dst->d_free = src_free;

	/* check message type */
	if (strncmp(PRIORITY_MESSAGE_TYPE, dst->d->s.type, strlen(PRIORITY_MESSAGE_TYPE)) && strncmp(COMMAND_DEMAND_MESSAGE_TYPE, dst->d->s.type, strlen(COMMAND_DEMAND_MESSAGE_TYPE)))
		{
			return tsc_error_malformed;
		}

	/* command request number */
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1], *ptr;
		memcpy(buf, (char *)(dst->d)+sizeof(message_header_s), COMMANDMESSAGE_SEQ_SIZE);
		buf[COMMANDMESSAGE_SEQ_SIZE] = '\0';
		dst->seqnum = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_SEQ_SIZE)
			{
				return tsc_error_malformed;
			}
	}

	/* command ID */
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1], *ptr;
		memcpy(buf, (char *)(dst->d)+sizeof(message_header_s)+COMMANDMESSAGE_SEQ_SIZE, COMMANDMESSAGE_ID_SIZE);
		buf[COMMANDMESSAGE_ID_SIZE] = '\0';
		dst->id = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_ID_SIZE)
			{
				return tsc_error_malformed;
			}
	}
	
	/* parameters */
	{
		if (dst->par_free) (dst->par_free)(dst->parameters);
		dst->parameters = (char *)(dst->d) + sizeof(message_header_s) + COMMANDMESSAGE_SEQ_SIZE + COMMANDMESSAGE_ID_SIZE;
		dst->parlen = message_header_length(dst->d) - COMMANDMESSAGE_SEQ_SIZE - COMMANDMESSAGE_ID_SIZE;
		dst->par_free = NULL;
	}

	return tsc_success;
}

tsc_error_t
read_CommandMessage(CommandMessage* dst, int fd)
{
	char *data;
	errno = 0;
	data = _read_Message(fd);
	if (data == NULL) return tsc_error_malloc;
	if (data == (char *) 1) return tsc_error_stream;
	if (data == (char *) 2) return tsc_error_malformed;
	if (data == (char *) 3) return tsc_error_nostream;
	return strto_CommandMessage(dst, data, free);
}

tsc_error_t
CommandMessage_set(CommandMessage *self,
	const char *type,	/* "CC" or "CD" */
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	const char *parameters,	/* parameters */
	int parlen	/* length of the parameters in bytes */
)
{
	int content_length;
	content_length = COMMANDMESSAGE_SEQ_SIZE + COMMANDMESSAGE_ID_SIZE + parlen;

	/* allocate memory */
	if (self->d_free) (self->d_free)(self->d);
	self->d = (message_header_s*) malloc(sizeof(message_header_s) + content_length);
	if (!self->d) return tsc_error_malloc;
	self->d_free = free;

	/* header */
	message_header_set_type(self->d, type);
	message_header_set_dst(self->d, dst);
	message_header_set_src(self->d, src);
	message_header_set_pid(self->d, pid);
	message_header_set_txtime(self->d, txtime);
	message_header_set_length(self->d, content_length);

	/* content */
	self->seqnum = seqnum;
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_SEQ_SIZE, seqnum);
		memcpy((char*)(self->d) + sizeof(message_header_s), buf, COMMANDMESSAGE_SEQ_SIZE);
	}
	self->id = id;
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_ID_SIZE, id);
		memcpy((char*)(self->d) + sizeof(message_header_s) + COMMANDMESSAGE_SEQ_SIZE, buf, COMMANDMESSAGE_ID_SIZE);
	}
	if (self->par_free) (self->par_free)(self->parameters);
	self->parameters = (char*)(self->d) + sizeof(message_header_s) + COMMANDMESSAGE_SEQ_SIZE + COMMANDMESSAGE_ID_SIZE;
	self->par_free = NULL;
	memcpy(self->parameters, parameters, parlen);
	self->parlen = parlen;

	return tsc_success;
}

tsc_error_t
CommandMessage_set_seqnum(CommandMessage *self, long seqnum)
{
	char buf[COMMANDMESSAGE_SEQ_SIZE + 1];
	self->seqnum = seqnum;
	snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_SEQ_SIZE, seqnum);
	memcpy((char*)(self->d) + sizeof(message_header_s), buf, COMMANDMESSAGE_SEQ_SIZE);
	return tsc_success;
}

char*
CommandMessage_tostr(CommandMessage *self)
{
	return (char*)(self->d);
}

size_t
CommandMessage_strlen(CommandMessage *self)
{
	return sizeof(message_header_s) + message_header_length(self->d);
}

const char*
CommandMessage_arg(CommandMessage *self, const char **startptr, size_t *length)
{
	return _message_arg(self->parameters, self->parlen, startptr, length);
}

tsc_error_t
ReceptionResponseMessage_set(ReceptionResponseMessage *self,
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	long rxnum,	/* rx sequence number */
	response_message_t result,  /* result of acceptance test */
	time_t endtime  /* estimated end time */
)
{
	char *dstptr;
	int content_length;
	content_length = COMMANDMESSAGE_SEQ_SIZE + COMMANDMESSAGE_ID_SIZE + COMMANDMESSAGE_RXSEQ_SIZE + COMMANDMESSAGE_RESULT_SIZE + COMMANDMESSAGE_INFO_SIZE;

	/* allocate memory */
	if (self->d_free) (self->d_free)(self->d);
	self->d = (message_header_s*) malloc(sizeof(message_header_s) + content_length);
	if (!self->d) return tsc_error_malloc;
	self->d_free = free;

	/* header */
	message_header_set_type(self->d, RECEPTION_RESPONSE_MESSAGE_TYPE);
	message_header_set_dst(self->d, dst);
	message_header_set_src(self->d, src);
	message_header_set_pid(self->d, pid);
	message_header_set_txtime(self->d, txtime);
	message_header_set_length(self->d, content_length);

	/* content */
	dstptr = (char*)(self->d) + sizeof(message_header_s);
	self->seqnum = seqnum;
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_SEQ_SIZE, seqnum);
		memcpy(dstptr, buf, COMMANDMESSAGE_SEQ_SIZE);
		dstptr += COMMANDMESSAGE_SEQ_SIZE;
	}
	self->id = id;
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_ID_SIZE, id);
		memcpy(dstptr, buf, COMMANDMESSAGE_ID_SIZE);
		dstptr += COMMANDMESSAGE_ID_SIZE;
	}
	self->rxnum = rxnum;
	{
		char buf[COMMANDMESSAGE_RXSEQ_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_RXSEQ_SIZE, rxnum);
		memcpy(dstptr, buf, COMMANDMESSAGE_RXSEQ_SIZE);
		dstptr += COMMANDMESSAGE_RXSEQ_SIZE;
	}
	self->result = result;
	{
		char *r;
		switch(result)
			{
				case(response_ok): r = "OK"; break;
				default: r = "NG"; break;
			}
		memcpy(dstptr, r, COMMANDMESSAGE_RESULT_SIZE);
		dstptr += COMMANDMESSAGE_RESULT_SIZE;
	}
	self->endtime = endtime;
	{
		switch(result)
			{
				case(response_ok):	/* fall through */
				case(response_ng):
					if (endtime == 0)
						{
							memcpy(dstptr, "NNNNNN", COMMANDMESSAGE_INFO_SIZE);
						}
					else
						{
							struct tm tm;
							char buf[COMMANDMESSAGE_INFO_SIZE + 1];
							memset(&tm, 0, sizeof(tm));
#ifdef HAVE_GMTIME_R
							gmtime_r(&endtime, &tm);
#else
							{
								struct tm *tmptr;
								tmptr = gmtime(&endtime);
								tm = *tmptr;
							}
#endif
							strftime(buf, sizeof(buf), "%H%M%S", &tm);
							memcpy(dstptr, buf, COMMANDMESSAGE_INFO_SIZE);
						}
					break;
				case(response_no_priority):
					memcpy(dstptr, "800000", COMMANDMESSAGE_INFO_SIZE);
					break;
				case(response_locked):
					memcpy(dstptr, "400000", COMMANDMESSAGE_INFO_SIZE);
					break;
				default:
					if (self->d_free) (self->d_free)(self->d);
					self->d = NULL;
					self->d_free = NULL;
					ReceptionResponseMessage_free(self);
					return tsc_error_malformed;
			}
		dstptr += COMMANDMESSAGE_INFO_SIZE;
	}
	return tsc_success;
}

char*
ReceptionResponseMessage_tostr(ReceptionResponseMessage *self)
{
	return (char*)(self->d);
}

size_t
ReceptionResponseMessage_strlen(ReceptionResponseMessage *self)
{
	return sizeof(message_header_s) + message_header_length(self->d);
}

tsc_error_t
strto_ReceptionResponseMessage(ReceptionResponseMessage* dst, char *src, void (*src_free)(void* ptr))
{
	char *srcptr;

	/* get the header in hand */
	dst->d = (message_header_s*) src;
	dst->d_free = src_free;

	/* check message type */
	if (strncmp(RECEPTION_RESPONSE_MESSAGE_TYPE, dst->d->s.type, strlen(RECEPTION_RESPONSE_MESSAGE_TYPE)))
		{
			return tsc_error_malformed;
		}

	/* command request number */
	srcptr = (char *)(dst->d) + sizeof(message_header_s);
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_SEQ_SIZE);
		buf[COMMANDMESSAGE_SEQ_SIZE] = '\0';
		dst->seqnum = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_SEQ_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_SEQ_SIZE;
	}

	/* command ID */
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_ID_SIZE);
		buf[COMMANDMESSAGE_ID_SIZE] = '\0';
		dst->id = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_ID_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_ID_SIZE;
	}

	/* rx sequence number */
	{
		char buf[COMMANDMESSAGE_RXSEQ_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_RXSEQ_SIZE);
		buf[COMMANDMESSAGE_RXSEQ_SIZE] = '\0';
		dst->rxnum = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_RXSEQ_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_RXSEQ_SIZE;
	}

	/* result */
	{
		while(1)	/* switch - case for a string */
			{
				if (!strncmp(srcptr, "OK", COMMANDMESSAGE_RESULT_SIZE))
					{
						dst->result = response_ok;
						break;
					}
				if (!strncmp(srcptr, "NG", COMMANDMESSAGE_RESULT_SIZE))
					{
						dst->result = response_ng;
						break;
					}
				/* default - malformed input */
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_RESULT_SIZE;
	}

	if (dst->result == response_ok)
	/* end time */
		{
			struct timeval txtime;
			struct tm tm;

			/* first, assume 00:00 today in UT */
			ASC15_TIME_to_t(message_header_txtime(dst->d), 0, sizeof(ASC15_TIME), &txtime);
			dst->endtime = txtime.tv_sec - (txtime.tv_sec % (24*3600));

			/* add the time written in the packet */
			memset(&tm, 0, sizeof(tm));
#ifdef HAVE_GMTIME_R
			gmtime_r(&(txtime.tv_sec), &tm);
#else
			{
				struct tm *tmptr;
				tmptr = gmtime(&(txtime->tv_sec));
				tm = *tmptr;
			}
#endif
			strptime(srcptr, "%H%M%S", &tm);
			dst->endtime += tm.tm_hour*3600 + tm.tm_min*60 + tm.tm_sec;

			/* add one day if the estimation is already in the past */
			if(txtime.tv_sec > dst->endtime) dst->endtime += 24*3600;
		}
	else
	/* error code */
		{
			dst->endtime = 0;
		}
	srcptr += COMMANDMESSAGE_INFO_SIZE;
		
	return tsc_success;
}

tsc_error_t
read_ReceptionResponseMessage(ReceptionResponseMessage* dst, int fd)
{
	char *data;
	errno = 0;
	data = _read_Message(fd);
	if (data == NULL) return tsc_error_malloc;
	if (data == (char *) 1) return tsc_error_stream;
	if (data == (char *) 2) return tsc_error_malformed;
	if (data == (char *) 3) return tsc_error_nostream;
	return strto_ReceptionResponseMessage(dst, data, free);
}

const char *
ReceptionResponseMessage_strerr(ReceptionResponseMessage *self)
{
	return completion_error_message(self->result, 0);
}

tsc_error_t
strto_CompletionResponseMessage(CompletionResponseMessage* dst, char *src, void (*src_free)(void* ptr))
{
	char *srcptr;

	/* get the header in hand */
	dst->d = (message_header_s*) src;
	dst->d_free = src_free;

	/* check message type */
	if (strncmp(COMPLETION_RESPONSE_MESSAGE_TYPE, dst->d->s.type, strlen(COMPLETION_RESPONSE_MESSAGE_TYPE)))
		{
			return tsc_error_malformed;
		}

	/* command request number */
	srcptr = (char *)(dst->d) + sizeof(message_header_s);
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_SEQ_SIZE);
		buf[COMMANDMESSAGE_SEQ_SIZE] = '\0';
		dst->seqnum = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_SEQ_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_SEQ_SIZE;
	}

	/* command ID */
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_ID_SIZE);
		buf[COMMANDMESSAGE_ID_SIZE] = '\0';
		dst->id = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_ID_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_ID_SIZE;
	}

	/* rx sequence number */
	{
		char buf[COMMANDMESSAGE_RXSEQ_SIZE + 1], *ptr;
		memcpy(buf, srcptr, COMMANDMESSAGE_RXSEQ_SIZE);
		buf[COMMANDMESSAGE_RXSEQ_SIZE] = '\0';
		dst->rxnum = strtol(buf, &ptr, 16);
		if (ptr - buf != COMMANDMESSAGE_RXSEQ_SIZE)
			{
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_RXSEQ_SIZE;
	}

	/* result */
	{
		while(1)	/* switch - case for a string */
			{
				if (!strncmp(srcptr, "COMPLETE%%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_complete;
						break;
					}
				if (!strncmp(srcptr, "CANCEL%%%%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_cancel;
						break;
					}
				if (!strncmp(srcptr, "TIMEOUT%%%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_timeout;
						break;
					}
				if (!strncmp(srcptr, "FAIL%%%%%%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_fail;
						break;
					}
				if (!strncmp(srcptr, "ERR(FMT)%%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_err_format;
						break;
					}
				if (!strncmp(srcptr, "ERR(COMM)%", COMMANDMESSAGE_COMPLETION_RESULT_SIZE))
					{
						dst->result = response_err_comm;
						break;
					}
				/* default - malformed input */
				return tsc_error_malformed;
			}
		srcptr += COMMANDMESSAGE_COMPLETION_RESULT_SIZE;
	}

	/* errror code if applicable */
	if (dst->result != response_complete && dst->result != response_err_format)
		{
			char buf[COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE + 1], *ptr;
			memcpy(buf, srcptr, COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE);
			buf[COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE] = '\0';
			dst->errorcode = strtol(buf, &ptr, 16);
			if (ptr - buf != COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE)
				{
					return tsc_error_malformed;
				}
			srcptr += COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE;
		}	
	else
		{
			dst->errorcode = 0;
		}

	/* data */
	dst->datalen = (char *)(dst->d) + sizeof(message_header_s) + message_header_length(dst->d) - srcptr;
	if (dst->datalen < 0)
		{
			return tsc_error_malformed;
		}
	if (dst->datalen > 0)
		{
			dst->data = srcptr;
			dst->data_free = NULL;
		}
	else
		{
			dst->data = NULL;
			dst->data_free = NULL;
		}

	return tsc_success;
}

tsc_error_t
read_CompletionResponseMessage(CompletionResponseMessage* dst, int fd)
{
	char *data;
	errno = 0;
	data = _read_Message(fd);
	if (data == NULL) return tsc_error_malloc;
	if (data == (char *) 1) return tsc_error_stream;
	if (data == (char *) 2) return tsc_error_malformed;
	if (data == (char *) 3) return tsc_error_nostream;
	return strto_CompletionResponseMessage(dst, data, free);
}

char*
CompletionResponseMessage_tostr(CompletionResponseMessage *self)
{
	return (char*)(self->d);
}

size_t
CompletionResponseMessage_strlen(CompletionResponseMessage *self)
{
	return sizeof(message_header_s) + message_header_length(self->d);
}

const char*
CompletionResponseMessage_arg(CompletionResponseMessage *self, const char **startptr, size_t *length)
{
	return _message_arg(self->data, self->datalen, startptr, length);
}

tsc_error_t
CompletionResponseMessage_set(CompletionResponseMessage *self,
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	long rxnum,	/* rx sequence number */
	response_message_t result,	/* result of execution */
	long errorcode,	/* error code */
	char *data,	/* status data, command, or communication data as defined in Table 2.5-1 in "F-M interface" documentation, excluding error code */
	int datalen	/* length of data */
)
{
	char *dstptr;
	int content_length;
	content_length = COMMANDMESSAGE_SEQ_SIZE + COMMANDMESSAGE_ID_SIZE + COMMANDMESSAGE_RXSEQ_SIZE + COMMANDMESSAGE_COMPLETION_RESULT_SIZE + datalen;
	if (result != response_complete && result != response_err_format && id != 0x1A1901) content_length += COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE;

	/* allocate memory */
	if (self->d_free) (self->d_free)(self->d);
	self->d = (message_header_s*) malloc(sizeof(message_header_s) + content_length);
	if (!self->d) return tsc_error_malloc;
	self->d_free = free;

	/* header */
	message_header_set_type(self->d, COMPLETION_RESPONSE_MESSAGE_TYPE);
	message_header_set_dst(self->d, dst);
	message_header_set_src(self->d, src);
	message_header_set_pid(self->d, pid);
	message_header_set_txtime(self->d, txtime);
	message_header_set_length(self->d, content_length);

	/* content */
	dstptr = (char*)(self->d) + sizeof(message_header_s);
	self->seqnum = seqnum;
	{
		char buf[COMMANDMESSAGE_SEQ_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_SEQ_SIZE, seqnum);
		memcpy(dstptr, buf, COMMANDMESSAGE_SEQ_SIZE);
		dstptr += COMMANDMESSAGE_SEQ_SIZE;
	}
	self->id = id;
	{
		char buf[COMMANDMESSAGE_ID_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_ID_SIZE, id);
		memcpy(dstptr, buf, COMMANDMESSAGE_ID_SIZE);
		dstptr += COMMANDMESSAGE_ID_SIZE;
	}
	self->rxnum = rxnum;
	{
		char buf[COMMANDMESSAGE_RXSEQ_SIZE + 1];
		snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_RXSEQ_SIZE, rxnum);
		memcpy(dstptr, buf, COMMANDMESSAGE_RXSEQ_SIZE);
		dstptr += COMMANDMESSAGE_RXSEQ_SIZE;
	}
	self->result = result;
	{
		char *r;
		switch(result)
			{
				case(response_complete): r = "COMPLETE%%"; break;
				case(response_cancel): r = "CANCEL%%%%"; break;
				case(response_timeout): r = "TIMEOUT%%%"; break;
				case(response_fail): r = "FAIL%%%%%%"; break;
				case(response_err_format): r = "ERR(FMT)%%"; break;
				case(response_err_comm): r = "ERR(COMM)%"; break;
				default:
					(self->d_free)(self->d);
					self->d = NULL;
					self->d_free = NULL;
					CompletionResponseMessage_free(self);
					return tsc_error_malformed;
			}
		memcpy(dstptr, r, COMMANDMESSAGE_COMPLETION_RESULT_SIZE);
		dstptr += COMMANDMESSAGE_COMPLETION_RESULT_SIZE;
	}
	if (result != response_complete && result != response_err_format && id != 0x1A1901)
		{
			char buf[COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE + 1];
			self->errorcode = errorcode;
			snprintf(buf, sizeof(buf), "%0*lX", COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE, errorcode);
			memcpy(dstptr, buf, COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE);
			dstptr += COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE;
		}
	else
		{
			self->errorcode = 0;
		}

	if (self->data_free) (self->data_free)(self->data);
	self->data = NULL;
	self->data_free = NULL;
	self->datalen = datalen;
	if (datalen > 0)
		{
			self->data = dstptr;
			self->datalen = datalen;
			memcpy(dstptr, data, datalen);
			dstptr += datalen;
		}

	return tsc_success;
}

const char *
CompletionResponseMessage_strerr(CompletionResponseMessage *self)
{
	return completion_error_message(self->result, self->errorcode);
}

const char *
completion_error_message(response_message_t response, long errorcode)
{
	const char *r;
	r = "unknown error";
	switch(response)
		{
			case(response_ok):
				r = "ok";
				break;
			case(response_no_priority):
				r = "NG because no priority in Rx machine";
				break;
			case(response_locked):
				r = "NG because the machine is locked";
				break;
			case(response_complete):
				r = "completed";
				break;
			/* cancel - in Table 2.5-1 in "F-M interface" documentation */
			case(response_cancel):
				switch(errorcode)
					{
						case(0x1): r = "canceled with a duplicated command"; break;
						case(0x2): r = "canceled with an inconsistent command"; break;
						default: r = "canceled"; break;
					}
				break;
			/* timeout - in Table 2.5-1 in "F-M interface" documentation */
			case(response_timeout):
				switch(errorcode)
					{
						case(0x1): r = "can not confirm the check pattern on local"; break;
						case(0x2): r = "error in file transfer"; break;
						case(0x11): r = "timeout detected on TSC"; break;
						default: r = "timeout"; break;
					}
				break;
			/* */
			case(response_fail):
				r = "interlocked";
				break;
			/* */
			case(response_err_format):
				r = "format error";
				break;
			/* err(comm) - in Table 2.5-1 in "F-M interface" documentation */
			case(response_err_comm):
				switch(errorcode)
					{
						case(0x1): r = "communication error with MLP1"; break;
						case(0x2): r = "communication error with MLP2"; break;
						case(0x3): r = "communication error with MLP3"; break;
						case(0x4): r = "communication error with TWS1"; break;
						case(0x5): r = "communication error with TWS2"; break;
						case(0x6): r = "communication error with OBS"; break;
						default: r = "communication error";
					}
				break;
			/* fail from http://www1.naoj.org/operation/documents/TSC_NG_Codes.txt */
				switch(errorcode)
					{

						case(0x100100): r = "malformed transmission time"; break;
						case(0x100200): r = "format error"; break;
						case(0x100300): r = "malformed parameter"; break;
						case(0x100400): r = "parameter out of range"; break;
						case(0x200100): r = "not logged in"; break;
						case(0x200200): r = "incorrect operator level"; break;
						case(0x300100): r = "TSC malfunction (PA and MA)"; break;
						case(0x300200): r = "MLP malfunction"; break;
						case(0x300300): r = "controlled device malfunction"; break;
						case(0x300301): r = "error in communication to controlled device"; break;
						case(0x400100): r = "controlled device in local mode"; break;
						case(0x400200): r = "drive not ready (local condition)"; break;
						case(0x400301): r = "number of process exceeds 10"; break;
						case(0x400302): r = "file/catalog not found in TSC"; break;
						case(0x400303): r = "specified macro/schedule not running/stopped with an error"; break;
						case(0x400304): r = "specified macro not running in simulation mode"; break;
						case(0x400305): r = "current observation condition inconsistent with specified parameter"; break;
						case(0x400307): r = "specified file not owned by the operators group"; break;
						case(0x400308): r = "command sender different from the one started PA/MA"; break;
						case(0x400309): r = "telescope parameters not ready"; break;
						case(0x40030A): r = "telescope parameter type not selected for PA"; break;
						case(0x40030C): r = "PA parameters being measured"; break;
						case(0x40030D): r = "not waiting for MA operator confirmation"; break;
						case(0x40030E): r = "executing macro/schdule"; break;
						case(0x40030F): r = "permision denied to edit file"; break;
						case(0x400310): r = "number of registered file exceeded maximum"; break;
						case(0x400311): r = "error in coordinate conversion"; break;
						case(0x400312): r = "schedule not registered"; break;
						case(0x400313): r = "plot number/telescope paraeter calculation not specified"; break;
						case(0x400314): r = "MA not interruptted"; break;
						case(0x400401): r = "device not found/focus not supported"; break;
						case(0x400402): r = "not in controllable mode"; break;
						case(0x400403): r = "Cassegrain rotator not selected"; break;
						case(0x40040C): r = "drive on/controlling"; break;
						case(0x40040D): r = "Cs bolt drive on"; break;
						case(0x500102): r = "dome upper part door open"; break;
						case(0x500103): r = "dome upper part door open"; break;
						case(0x500104): r = "80/40t crane not in up finish position"; break;
						case(0x500106): r = "80/40t crane not in store position"; break;
						case(0x500108): r = "1t hoist not up in finish position"; break;
						case(0x500109): r = "1t hoist not up in finish position"; break;
						case(0x500201): r = "AZ dome gap limit"; break;
						case(0x500B01): r = "TUE in use"; break;
						case(0x500B02): r = "TUE in use"; break;
						case(0x500E02): r = "AO cal in optical axis"; break;
						case(0x502401): r = "mirror cover opening/closing"; break;
						case(0x502701): r = "not balanced"; break;
						case(0x503A01): r = "CIAX at position 1, 2, or 3"; break;
						case(0x503A02): r = "instrument not selected in CIAX-3"; break;
						case(0x503A03): r = "Cs bolt not anchored"; break;
						case(0x503A04): r = "CsInR not in bolt clamping position"; break;
						case(0x506101): r = "dome shutter open"; break;
						case(0x50A102): r = "wind speed more than 7m/s"; break;
						case(0x50B203): r = "actuator calibration failed"; break;
						case(0x800000): r = "TSC not in controllable mode"; break;
						case(0x900000): r = "device locked"; break;
					}
				break;
			default:
				break;
		}
	return r;
}

