/* test-tsc-message.c : test cases for tsc-message.c */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <ctype.h>
#include <unistd.h>

#include <melco/tsc-message.h>
#include <count-malloc.h>

void test_fread_MonitorDataMessage(void);
void test_strto_MonitorDataMessage(void);
void test_strto_CommandMessage(void);
void test_CommandMessage_set(void);
void test_CommandMessage_arg(char *pars, char *args[]);
void test_strto_ReceptionResponseMessage(void);
void test_ReceptionResponseMessage_set(void);
void test_strto_CompletionResponseMessage(void);
void test_CompletionResponseMessage_set(void);
void test_CompletionResponseMessage_arg(char *pars, char *args[]);
void test_read(void);
void test_fread_MonitorDataMessage_memory_manage(void);
void test_validate_InstrumentStatus(void);

void
test_fread_MonitorDataMessage(void)
{
	FILE *file;
	MonitorDataMessage *m;
	StatusBank *s;
	int i;

	s = (StatusBank*) malloc(sizeof(StatusBank));
	if (!s)
		{
			fputs("memory exhausted\n", stderr);
			exit(EXIT_FAILURE);
		}
	assert(StatusBank_init(s));

	assert(file = fopen("test-data/TSCL20050209-L1-L7.bin", "r"));
	for(i = 0; i < 8; i++)
		{
			tsc_error_t res;
			assert(m = MonitorDataMessage_alloc());
			res = fread_MonitorDataMessage(m, file, block_fujitsu, 0);
			if (res == tsc_error_malloc)
				{
					fputs("memory exhausted\n", stderr);
					exit(EXIT_FAILURE);
				}
			assert(res == (tsc_error_t) (i < 7 ? tsc_success : tsc_error_nostream));
			if (res == tsc_success)
				assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
		}
	fclose(file);

	assert(file = fopen("test-data/TSCS20050209-2sets.bin", "r"));
	for(i = 0; i < 3; i++)
		{
			tsc_error_t res;
			assert(m = MonitorDataMessage_alloc());
			res = fread_MonitorDataMessage(m, file, block_fujitsu, 0);
			assert(res == (tsc_error_t) (i < 2 ? tsc_success : tsc_error_nostream));
			if (res == tsc_success)
				assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
		}
	assert(feof(file));
	fclose(file);

	assert(file = fopen("test-data/TSCV20050209-10sets.bin", "r"));
	for(i = 0; i < 11; i++)
		{
			tsc_error_t res;
			assert(m = MonitorDataMessage_alloc());
			res = fread_MonitorDataMessage(m, file, block_fujitsu, 0);
			assert(res == (tsc_error_t) (i < 10 ? tsc_success : tsc_error_nostream));
			if (res == tsc_success)
				assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
		}
	assert(feof(file));
	fclose(file);

	assert(file = fopen("test-data/050418-1615xx-shm-0x100.bin", "r"));
	assert(m = MonitorDataMessage_alloc());
	assert(fread_MonitorDataMessage(m, file, block_tws4, 0) == tsc_success);
	assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
	fclose(file);

	assert(file = fopen("test-data/050418-1615xx-shm-0x101.bin", "r"));
	assert(m = MonitorDataMessage_alloc());
	assert(fread_MonitorDataMessage(m, file, block_tws4, 0) == tsc_success);
	assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
	fclose(file);

	assert(file = fopen("test-data/050418-1615xx-shm-0x102.bin", "r"));
	assert(m = MonitorDataMessage_alloc());
	assert(fread_MonitorDataMessage(m, file, block_tws4, 0) == tsc_success);
	assert(MonitorDataMessage_scan(m, s, scan_all) >= 0);
	fclose(file);

	{
		StatusEntry *e;
		char *buf;
		double d;
		assert(e = StatusBank_get_status_entry(s, "Right Ascention"));
		buf = (char*) malloc(StatusEntry_to_s_length(e) + 1);
		if (!buf)
			{
				fputs("memory exhausted\n", stderr);
				exit(EXIT_FAILURE);
			}
		assert(!strcmp("+05:43:22.459000", StatusEntry_to_s(e, buf)));
		assert(StatusEntry_to_d(e, &d));
		assert(fabs(1.498253 - d) < 1e-6);
		free(buf);
		StatusEntry_decref(e);
	}

	StatusBank_discard_all_inst(s);
	free(s);
}

void
test_fread_MonitorDataMessage_memory_manage(void)
{
	FILE *file;
	MonitorDataMessage *m;
	StatusBank *s;
	s = (StatusBank*) malloc(sizeof(StatusBank));
	if (!s)
		{
			fputs("memory exhausted\n", stderr);
			exit(EXIT_FAILURE);
		}
	assert(StatusBank_init(s));
	assert(file = fopen("test-data/TSCV20100302-1set.bin", "r"));
	assert(m = MonitorDataMessage_alloc());
	assert(fread_MonitorDataMessage(m, file, block_fujitsu, 0) == tsc_success);
	assert(MonitorDataMessage_scan(m, s, scan_strict) == 1);
	assert(m = MonitorDataMessage_alloc());
	assert(fread_MonitorDataMessage(m, file, block_fujitsu, 0) == tsc_error_nostream);
	assert(feof(file));
	fclose(file);
	StatusBank_discard_all_inst(s);
	free(s);
}

void
test_strto_MonitorDataMessage(void)
{
	FILE *file;
	struct timeval txtime;
	char stat[FUJITSU_TSCL_BLOCK_SIZE];
	char buf[16];	/* change size of buf according to the tests */
	MonitorDataMessage m;
	MonitorDataMessage_init(&m);

	file = fopen("test-data/TSCL20050209-L1-L7.bin", "r");
	if (!file)
		{
			perror("fopen");
			exit(EXIT_FAILURE);
		}
	assert(fread(stat, sizeof(stat), 1, file) == 1);
	fclose(file);
	assert(strto_MonitorDataMessage(&m, stat, NULL) == tsc_success);
	message_header_type(m.d, buf);
	assert(!strcmp(buf, "MO"));
	message_header_dst(m.d, buf);
	assert(!strcmp(buf, "OBS%"));
	message_header_src(m.d, buf);
	assert(!strcmp(buf, "TSC%"));
	assert(0 == message_header_pid(m.d));
	assert(!strncmp((const char*)message_header_txtime(m.d), "200502091000005", 15));
	assert(&txtime == message_header_txtime_tv(m.d, &txtime));
	assert(1107943200 == txtime.tv_sec);
	assert(1255 == message_header_length(m.d));
	assert(1255 == m.length);
	assert(!strcmp("L1", m.type));
	assert(2 == m.typelen);
	MonitorDataMessage_decref(&m);
}

void
test_strto_CommandMessage(void)
{
	char buf[16];	/* change size of buf according to the tests */
	char command[] = "CDTSC%TWS40000200312220924426002700001C1A1901naoj **** % % %"; /* from tws4:~naoj/Readme.if_tsc_has_rebooted */
	char pars[] = "naoj **** % % %";

	CommandMessage m;
	CommandMessage_init(&m);
	assert(strto_CommandMessage(&m, command, NULL) == tsc_success);
	message_header_type(m.d, buf);
	assert(!strcmp(buf, "CD"));
	message_header_dst(m.d, buf);
	assert(!strcmp(buf, "TSC%"));
	message_header_src(m.d, buf);
	assert(!strcmp(buf, "TWS4"));
	assert(0x1C == m.seqnum);
	assert(0x1A1901 == m.id);
	assert(strlen(pars) == (unsigned) m.parlen);
	assert(!strcmp(pars, m.parameters));
	CommandMessage_decref(&m);
}

void
test_CommandMessage_set(void)
{
	char command[] = "CDTSC%TWS40000200312220924426002700001C1A1901naoj **** % % %";
	char pars[] = "naoj **** % % %";
	struct timeval txtime;

	CommandMessage m;
	CommandMessage_init(&m);

	txtime.tv_sec =	1072085082;	/* 2003-12-22 09:24:42 UTC */
	txtime.tv_usec = 600000;

	assert(CommandMessage_set(&m, "CD", "TSC%", "TWS4", 0, &txtime, 0x1C, 0x1A1901, pars, strlen(pars)) == tsc_success);
	{
		int l;
		char *b;
		b = CommandMessage_tostr(&m);
		l = CommandMessage_strlen(&m);
	}
	assert(strlen(command) == CommandMessage_strlen(&m));
	assert(!strncmp(command, CommandMessage_tostr(&m), CommandMessage_strlen(&m)));
	CommandMessage_decref(&m);
}

void
test_CommandMessage_arg(char *pars, char *args[])
{
	int argi;
	size_t len;
	const char *argstart;
	struct timeval txtime;

	CommandMessage m;
	CommandMessage_init(&m);

	txtime.tv_sec =	1072085082;	/* 2003-12-22 09:24:42 UTC */
	txtime.tv_usec = 600000;

	assert(CommandMessage_set(&m, "CD", "TSC%", "TWS4", 0, &txtime, 0x1C, 0x1A1901, pars, strlen(pars)) == tsc_success);

	argstart = NULL;
	for(argi = 0; args[argi]; argi++)
		{
			assert(CommandMessage_arg(&m, &argstart, &len));
			assert(strlen(args[argi]) == len);
			assert(!strncmp(args[argi], argstart, len));
			argstart += len;
		}
	assert(!CommandMessage_arg(&m, &argstart, &len));

	CommandMessage_decref(&m);
}

void
test_strto_ReceptionResponseMessage(void)
{
	char command[] = "CATWS4TSC%0000200312220924143002600001C1A1901069320OK092422";
	time_t endtime;
	struct timeval txtime;
	char buf[16];	/* change size of buf according to the tests */
	ReceptionResponseMessage m;

	endtime =	1072085062;	/* 2003-12-22 09:24:22 UTC */
	txtime.tv_sec =	1072085054;	/* 2003-12-22 09:24:28 UTC */
	txtime.tv_usec = 300000;

	ReceptionResponseMessage_init(&m);
	assert(strto_ReceptionResponseMessage(&m, command, NULL) == tsc_success);

	message_header_type(m.d, buf);
	assert(!strcmp(buf, "CA"));
	message_header_dst(m.d, buf);
	assert(!strcmp(buf, "TWS4"));
	message_header_src(m.d, buf);
	assert(!strcmp(buf, "TSC%"));
	assert(0 == message_header_pid(m.d));
	assert(!strncmp((const char*)message_header_txtime(m.d), "200312220924143", 15));
	assert(26 == message_header_length(m.d));
	assert(0x1C == m.seqnum);
	assert(0x1A1901 == m.id);
	assert(0x69320 == m.rxnum);
	assert(response_ok == m.result);
	assert(endtime == m.endtime);

	ReceptionResponseMessage_decref(&m);
}

void
test_ReceptionResponseMessage_set(void)
{
	char command[] = "CATWS4TSC%0000200312220924143002600001C1A1901069320OK092422";
	time_t endtime;
	struct timeval txtime;
	ReceptionResponseMessage m;

	endtime =	1072085062;	/* 2003-12-22 09:24:22 UTC */
	txtime.tv_sec =	1072085054;	/* 2003-12-22 09:24:28 UTC */
	txtime.tv_usec = 300000;

	ReceptionResponseMessage_init(&m);
	assert(ReceptionResponseMessage_set(&m, "TWS4", "TSC%", 0, &txtime, 0x1C, 0x1A1901, 0x069320, response_ok, endtime) == tsc_success);
	{
		int l;
		char *b;
		b = ReceptionResponseMessage_tostr(&m);
		l = ReceptionResponseMessage_strlen(&m);
	}
	assert(strlen(command) == ReceptionResponseMessage_strlen(&m));
	assert(!strncmp(command, ReceptionResponseMessage_tostr(&m), ReceptionResponseMessage_strlen(&m)));
	ReceptionResponseMessage_decref(&m);
}

void
test_strto_CompletionResponseMessage(void)
{
	char command[] = "CETWS4TSC%0000200312220924144002800001D1A1011069321COMPLETE%%";
	char buf[16];	/* change size of buf according to the tests */

	CompletionResponseMessage m;
	CompletionResponseMessage_init(&m);
	assert(strto_CompletionResponseMessage(&m, command, NULL) == tsc_success);

	message_header_type(m.d, buf);
	assert(!strcmp(buf, "CE"));
	message_header_dst(m.d, buf);
	assert(!strcmp(buf, "TWS4"));
	message_header_src(m.d, buf);
	assert(!strcmp(buf, "TSC%"));
	assert(0 == message_header_pid(m.d));
	assert(!strncmp((const char*)message_header_txtime(m.d), "200312220924144", 15));
	assert(28 == message_header_length(m.d));
	assert(0x1D == m.seqnum);
	assert(0x1A1011 == m.id);
	assert(0x69321 == m.rxnum);
	assert(response_complete == m.result);
	assert(NULL == m.data);
	assert(0 == m.datalen);

	CompletionResponseMessage_decref(&m);
}

void
test_CompletionResponseMessage_set(void)
{
	char command[] = "CETWS4TSC%0000200312220924144002800001D1A1011069321COMPLETE%%";
	struct timeval txtime;
	CompletionResponseMessage m;

	txtime.tv_sec =	1072085054;	/* 2003-12-22 09:24:14 UTC */
	txtime.tv_usec = 400000;

	CompletionResponseMessage_init(&m);
	assert(CompletionResponseMessage_set(&m, "TWS4", "TSC%", 0, &txtime, 0x1D, 0x1A1011, 0x069321, response_complete, 0, NULL, 0) == tsc_success);
	{
		int l;
		char *b;
		b = CompletionResponseMessage_tostr(&m);
		l = CompletionResponseMessage_strlen(&m);
	}
	assert(strlen(command) == CompletionResponseMessage_strlen(&m));
	assert(!strncmp(command, CompletionResponseMessage_tostr(&m), CompletionResponseMessage_strlen(&m)));
	CompletionResponseMessage_decref(&m);
}

void
test_CompletionResponseMessage_arg(char *data, char *args[])
{
	int argi;
	size_t len;
	const char *argstart;
	struct timeval txtime;

	CompletionResponseMessage m;
	CompletionResponseMessage_init(&m);

	txtime.tv_sec =	1072085082;	/* 2003-12-22 09:24:42 UTC */
	txtime.tv_usec = 600000;

	assert(CompletionResponseMessage_set(&m, "TWS4", "TSC%", 0, &txtime, 0x1C, 0x1A1901, 0, response_complete, 0, data, strlen(data)) == tsc_success);

	argstart = NULL;
	for(argi = 0; args[argi]; argi++)
		{
			assert(CompletionResponseMessage_arg(&m, &argstart, &len));
			assert(strlen(args[argi]) == len);
			assert(!strncmp(args[argi], argstart, len));
			argstart += len;
		}
	assert(!CompletionResponseMessage_arg(&m, &argstart, &len));

	CompletionResponseMessage_decref(&m);
}

void
test_read(void)
{
	int fds[2];
	char *messages[] = {
		"CDTSC%TWS40000200312220924426002700001C1A1901naoj **** % % %",
		"CATWS4TSC%0000200312220924143002600001C1A1901069320OK092422",
		"CETWS4TSC%0000200312220924144002800001D1A1011069321COMPLETE%%",
		NULL
	};
	assert(!pipe(fds));
	{
		CommandMessage m1;
		ReceptionResponseMessage m2;
		CompletionResponseMessage m3;
		ssize_t len;

		len = strlen(messages[0]);
		assert(write(fds[1], messages[0], len) == len);
		CommandMessage_init(&m1);
		assert(read_CommandMessage(&m1, fds[0]) == tsc_success);
		CommandMessage_decref(&m1);

		len = strlen(messages[1]);
		assert(write(fds[1], messages[1], len) == len);
		ReceptionResponseMessage_init(&m2);
		assert(read_ReceptionResponseMessage(&m2, fds[0]) == tsc_success);
		ReceptionResponseMessage_decref(&m2);

		len = strlen(messages[2]);
		assert(write(fds[1], messages[2], len) == len);
		CompletionResponseMessage_init(&m3);
		assert(read_CompletionResponseMessage(&m3, fds[0]) == tsc_success);
		assert(!strcmp("completed", CompletionResponseMessage_strerr(&m3)));
		CompletionResponseMessage_decref(&m3);

	}
}

void
test_validate_InstrumentStatus(void)
{
	FILE *file;
	MonitorDataMessage *m;
	StatusBank *s;

	s = (StatusBank*) malloc(sizeof(StatusBank));
	if (!s)
		{
			fputs("memory exhausted\n", stderr);
			exit(EXIT_FAILURE);
		}
	assert(StatusBank_init(s));

	assert(file = fopen("test-data/TSCV20110616-fake0002.bin", "r"));

	{
		tsc_error_t res;
		assert(m = MonitorDataMessage_alloc());
		res = fread_MonitorDataMessage(m, file, block_fujitsu, 0);
		if (res == tsc_error_malloc)
			{
				fputs("memory exhausted\n", stderr);
				exit(EXIT_FAILURE);
			}
		assert(res == (tsc_error_t) tsc_success);
		if (res == tsc_success)
			assert(MonitorDataMessage_scan(m, s, scan_all) == 1);
	}

	fclose(file);
	StatusBank_discard_all_inst(s);
	free(s);
}

int
main(int argc __attribute__ ((unused)), char *argv[])
{
	char pars0[] = "";
	char *args0[] = {NULL};
	char pars1[] = "%";
	char *args1[] = {""};
	char pars2[] = "naoj **** % % %";
	char *args2[] = {"naoj", "****", "", "", "", NULL};

	fprintf(stderr, "%s running\n", argv[0]);
	test_validate_InstrumentStatus();
	test_fread_MonitorDataMessage_memory_manage();
	test_fread_MonitorDataMessage();
	test_strto_MonitorDataMessage();
	test_strto_CommandMessage();
	test_CommandMessage_set();
	test_CommandMessage_arg(pars0, args0);
	test_CommandMessage_arg(pars1, args1);
	test_CommandMessage_arg(pars2, args2);
	test_strto_ReceptionResponseMessage();
	test_ReceptionResponseMessage_set();
	test_strto_CompletionResponseMessage();
	test_CompletionResponseMessage_set();
	test_CompletionResponseMessage_arg(pars2, args2);
	test_read();
	return EXIT_SUCCESS;
}
