/* test-file-interface.c: test cases for file-interface.c using regular files */
/*

Copyright (C) 2010 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <melco/tsc-message.h>
#include <melco/shm-interface.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
extern int malloc_count;
#endif

#define TESTDIR "test-file"

void to_file(void);
void test_on_file(void);
void test_on_values(void);

void
to_file(void)
{
	int i;
	char *files[] = {
		"test-data/TSCL20050209-L1-L7.bin",
		"test-data/TSCS20050209-2sets.bin",
		"test-data/TSCV20050209-10sets.bin",
		NULL
	};
#ifdef COUNT_MALLOC
	int malloc_count_orig;
	malloc_count_orig = malloc_count;
#endif

	fprintf(stderr, "%s\n", __func__);
	for(i = 0; files[i]; i++)
		{
			FILE *file;
			int c;

			file = fopen(files[i], "r");
			assert(file);
			while(!feof(file))
				{
					int r;
					MonitorDataMessage *msg;
					msg = MonitorDataMessage_alloc();
					if (!msg)
						{
							fputs("memory exhausted\n", stderr);
							exit(EXIT_FAILURE);
						}
					r = fread_MonitorDataMessage(msg, file, block_fujitsu, 0);
					if (r == tsc_error_malloc)
						{
							fputs("memory exhausted\n", stderr);
							exit(EXIT_FAILURE);
						}
					assert(r == tsc_success || r == tsc_error_nostream);
					if (feof(file)) break;
					r = MonitorDataMessage_scan_to_file(msg, scan_all, TESTDIR);
					if (r < 0) perror("MonitorDataMessage_scan_to_file()");
					assert(r >= 0);
				}
			fclose(file);
		}
	instrumentstatus_file_unmap_all(TESTDIR);
#ifdef COUNT_MALLOC
	assert(malloc_count == malloc_count_orig);
#endif
}

void
test_on_file(void)
{
	StatusEntry *entry, *sptr;
	char *buf, *cptr;

	fprintf(stderr, "%s\n", __func__);
	entry = file_get_status_entry("Guide Star1 Image Size on AG", TESTDIR);
	assert(entry);
	sptr = StatusEntry_decref(entry);
	assert(!sptr);
	entry = file_get_status_entry("Observer", TESTDIR);
	assert(entry);
	buf = (char *) malloc(StatusEntry_to_s_length(entry) + 1);
	if(!buf)
		{
			fputs("memory exhausted\n", stderr);
			exit(EXIT_FAILURE);
		}
	cptr = StatusEntry_to_s(entry, buf);
	assert(cptr);
	assert(!strcmp("Sugai", buf));
	sptr = StatusEntry_decref(entry);
	assert(!sptr);
	free(buf);
}

void
test_on_values(void)
{
	int j, c, k;
	struct name_and_goal_s {
		char *name;
		char *goal;
		time_t rxtime;
	};
	struct name_and_goal_s name_and_goal[] = {
		/* goal time (time_t) can be calculated with: date -d 'date' +%s */
		/* rx time check */
		{"Heat Exh Mirror Temp Real", "2.47500000", 1107943200}, /* L-00A1:200502091000004 */
		/* BCD6 */
		{"AZ Real Angle", "-131.84393500", 1107943200},
		{"AZ CMD Angle ANS", "-131.84392400", 1107943200},
		{"Cal Source R CMD Position", "184.20000000", 1107943198},
		{"Cal Source R Real Position", "184.20000000", 1107943198},
		{"Dome Current REF-1", "-10.96000000", 1107943199},
		{"Dome Current REF-2", "13.23000000", 1107943199},
		{"Dome Current REF-3", "-11.21000000", 1107943199},
		{"Dome Current REF-4", "13.08000000", 1107943199},
		{"TLSCP Room(NsIR) HMD Data", "15.30000000", 1107943198},
		{"TLSCP Room HMD Data", "14.80000000", 1107943198},
		{"TLSCP Room(NsOpt) HMD Data", "13.70000000", 1107943198},
		/* BCD1 */
		{"Top Unit Code on BLCU", "1", 1107943201},
		{"Top Unit Code on SMCU", "1", 1107943201},
		/* bit */
		{"2nd Mirror DPA1 Power On", "1", 1107943201},
		{"P AG/SH Safety SW", "0", 1107807517},
		{"P AG/SH X 1st Limit +", "0", 1107807517},
		{"P AG/SH Y 1st Limit -", "0", 1107807517},
		{"Cs Bolt1 Anchored", "0", 1107734097},
		/* {"Cs Bolt1 Extended", "0", 1107734097}, no longer exist? */
		{"Cs Bolt1 Stop", "1", 1107734097},
		/* BINARY1_0p01 */
		{"SH REF I", "0.00", 1107943199},
		/* BINARY1_1 */
		{"AG Binning Pattern", "34", 1107943201},
		/* BINARY2_0p01 */
		{"SH REF V", "0.00", 1107943199},
		/* BINARY2_0p1 */
		{"PMFXS8-60 Temp MON1 Data", "2.0", 1107943200},
		/* BINARY2_1 */
		{"SV Star Position1 Intensity on MLP1", "3189", 1107943200},
		{"SV Star Position2 Intensity on MLP1",  "0", 1107943200},
		{"MDB1 208V PS Voltage", "209", 1107943199},
		{"MDB2 208V PS Voltage", "208", 1107943199},
		{"MDB1 480V PS Voltage", "493", 1107943199},
		{"MDB2 480V PS Voltage", "484", 1107943199},
		/* BINARY4_0p000000000001 */
		{"Time GEN Freq S.D.", "-0.000000000408", 1107942552},
		/* BINARY4_0p000001 */
		/* can not check: all are vstat for PA */
		/* BINARY4_0p00001 */
		/* can not check: all are vstat */
		/* BINARY4_0p001 */
		{"SV Field Real Position", "0.000", 1107943199},
		{"SV Field CMD Position ANS", "0.000", 1107943199},
		/* BINARY4_0p01 */
		{"AG CCD Temperature", "-40.10", 1107943199},
		{"SV CCD Temperature", "-40.00", 1107943199},
		{"AG Star Position1 X", "239.93", 1107943201},
		{"AG Star Position1 Y", "247.53", 1107943201},
		{"AG Star Position1 Error X on AG", "0.00", 1107943199},
		{"AG Star Position1 Error Y on AG", "0.00", 1107943199},
		/* BINARY4_0p1 */
		{"Weather Atmospheric Pressure", "622.0", 1107943196},
		/* BINARY4_100 */
		/* can not check: all are vstat */
		/* BINARY4_1 */
		{"AG Exposure Time", "3980", 1107943201},
		/* REAL8 */
		{"Aluminizing Storing Pressure", "0", 1107943199},
		{"Aluminizing Chanber Pressure1(Torr)", "1e-12", 1107943199},
		{"Aluminizing Chanber Pressure1(pa)", "1.33322e-10", 1107943199},
		{"Aluminizing Chanber Pressure2", "1e-12", 1107943199},
		{"Aluminizing Cryopump A Pressure", "1e-12", 1107943199},
		{"Aluminizing Cryopump B Pressure", "1e-12", 1107943199},
		{"Aluminizing Cryopump C Pressure", "1e-12", 1107943199},
		/* BCD6_RA */
		{"Right Ascention", "+10:48:15.399000", 1107943200},
		{"Right Ascention CMD", "+10:48:15.427000", 1107943200},
		/* BCD6_DEC */
		{"Declination", "+37:39:28.8000", 1107943200},
		/* BCD6_Epoch */
		{"Epoch", "2000", 1107943200},
		/* TIME3 */
		{"AG I/F TX Last Access Time", "09:59:59", 1107943200},
		{"AG I/F RX Last Access Time", "09:59:59", 1107943200},
		/* TIME4 */
		{"MLP2 Zernike Time", "00-00:00:00", 1107943203},
		/* TIME6 */
		{"Time GEN Time", "2005/02/09 10:00:00", 1107942552},
		/* TIME8 */
		{"AG Data Time on MLP1", "2005/02/09 10:00:03", 1107943204},
		{"AG Data Time on AG", "2005/02/09 09:59:55", 1107943199},
		{"SV Data Time", "2005/01/31 16:24:03", 1107943199},
		/* PIR */
		/*{"PIR AG for SH Image Size", "0", 1107943200},*/
		{NULL, NULL, 0},
	};

	fprintf(stderr, "%s\n", __func__);
	for(k = 0; name_and_goal[k].name != NULL; k++)
		{
			StatusEntry *entry;
			fprintf(stderr, "%s:",name_and_goal[k].name);
			entry = file_get_status_entry(name_and_goal[k].name, TESTDIR);
			assert(entry);
			for(j = 0; j < entry->info->bytes; j++)
				{
					fprintf(stderr, " %02X", ((unsigned char *) entry->d)[j]);
				}
			fputs("=>", stderr);
			for(j = 0; j < entry->info->bytes; j++)
				{
					c = (int) ((unsigned char *) entry->d)[j];
					if (isprint(c))
						fprintf(stderr, "%c", ((unsigned char *) entry->d)[j]);
					else
						fputs(".", stderr);
				}
			if (entry->info->to_i)
				{
					long long buf, *ptr;
					ptr = (entry->info->to_i)(entry->d, entry->info->mask, entry->info->bytes, (void*) &buf);
					assert(ptr);
          fprintf(stderr, "=>%lld", buf);
				}
			if (entry->info->to_d)
				{
					double buf, *ptr;
					ptr = (entry->info->to_d)(entry->d, entry->info->mask, entry->info->bytes, (void*) &buf);
					assert(ptr);
					fprintf(stderr, "=>%g", buf);
				}
			if (entry->info->to_s)
				{
					char buf[100], *ptr;
					ptr = (entry->info->to_s)(entry->d, entry->info->mask, entry->info->bytes, (void*) &buf);
					assert(ptr);
					fprintf(stderr, "=>%s", buf);
					assert(!strcmp(name_and_goal[k].goal, buf));
				}
			/* higher interface function */
			{
				char buf[100], ct[100], *cptr;
				struct timeval tv, *tptr;
				struct tm tm;
				cptr = StatusEntry_to_s(entry, buf);
				assert(cptr);
				tptr = StatusEntry_rxtime_tv(entry, &tv);
				assert(tptr);
				memset(&tm, 0, sizeof(tm));
#ifdef HAVE_LOCALTIME_R
				localtime_r(&(tv.tv_sec), &tm);
#else
				{
					struct tm *tm_t;
					tm_t = localtime(&(tv.tv_sec));
					tm = *tm_t;
				}
#endif
				strftime(ct, sizeof(ct), " at %b %d %T%Z", &tm);
				fputs(ct, stderr);
				assert(name_and_goal[k].rxtime == tv.tv_sec);
			}
			StatusEntry_decref(entry);
			fputs("\n", stderr);
		}
	fprintf(stderr, "instrumentstatus_file_unmap_all(): %d\n", instrumentstatus_file_unmap_all(TESTDIR));
}

int
main(int argc, char *argv[])
{ 
	int set_f = 1;
	int test_f = 1;
	int unlink_f = 1;

  fprintf(stderr, "%s running\n", argv[0]);
	if (argc > 1)
		{
			if (!strcmp(argv[1], "set-only"))
				{
					fprintf(stderr, "only preparing POSIX shared memory objects\n");
					test_f = 0;
					unlink_f = 0;
				}
			if (!strcmp(argv[1], "unlink-only"))
				{
					fprintf(stderr, "only discarding POSIX shared memory objects\n");
					set_f = 0;
					test_f = 0;
				}
		}

	if (mkdir(TESTDIR, 0777))
		{
			perror("mkdir");
		}

	if (set_f)
		{
			to_file();
		}
	if (test_f)
		{
			test_on_file();
			test_on_values();
		}
	if (unlink_f) fprintf(stderr, "instrumentstatus_file_unlink_all(): %d\n", instrumentstatus_file_unlink_all(TESTDIR));
	return EXIT_SUCCESS;
}
