#
# a python module to handle status entries
#
# Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>
#
# Permission is granted for use, copying, modification, distribution,
# and distribution of modified versions of this work under the terms of
# GPL version 2 or later.
#
# vim: set ts=2 sw=2 ai noet:

import sys
import re
import csv
import os
import string

import mitsubishi

# type:{shortid(mitsubishi.py):{'in':'Lx', 'shortid':'shortid'}}
shortid_real = {
	'E':{
		'00B2-1':{'in':'E', 'shortid':'00B2'},
	},
	'S':{
		'00B2-1':{'in':'S', 'shortid':'00B2'},
	},
	'L':{
		'00B2-1':{'in':'L2', 'shortid':'00B2'},
		'00B2-2':{'in':'L3', 'shortid':'00B2'},
	},
}

# type including L2 or L3 for MLP2
def extype(type, shortid):
	if shortid_real.has_key(type) and shortid_real[type].has_key(shortid):
		r = shortid_real[type][shortid]['in']
	else:
		r = type
	return r

# real shortid
def realshortid(type, shortid):
	if shortid_real.has_key(type) and shortid_real[type].has_key(shortid):
		r = shortid_real[type][shortid]['shortid']
	else:
		r = shortid
	return r

# Hash name space
class NameSpace:
	# id:format
	solver_id = {
		'0001034':'%s',	# Mirror Hatch Open
		'00B314C':'%s 2',	# MDB-1 MCB UPS-3 Alarm
		'00B314D':'%s 1',	# MDB-1 MCB UPS-3 Alarm
		'00B3157':'%s 2',	# MDB-1 MCB UPS XFMR BD-1 Alarm
		'00B3158':'%s 1',	# MDB-1 MCB UPS XFMR BD-1 Alarm
		'0002123':'%s',	# COML PWR Fault
		'00B3163':'%s 1',	# MDB-2 MCB UPS XFMR BD-1 Alarm
		'00B3171':'%s 2',	# MDB-2 MCB UPS XMFR BD-2 Alarm
		'0002040':'%s 1',	# AZ Cable Overlap +
		'0002041':'%s 1',	# AZ Cable Overlap -
		'0002119':'%s 2',	# AZ Cable Overlap +
		'000211A':'%s 2',	# AZ Cable Overlap -
		'00B102F':'%s in E',	# Auto Guide AZ Error Correct Angle
		'00B111E':'%s in S',	# Auto Guide AZ Error Correct Angle
		'00B1030':'%s in E',	# Auto Guide EL Error Correct Angle
		'00B111F':'%s in S',	# Auto Guide EL Error Correct Angle
	}
	# id range: range does not include the last number
	solver_id_range = {
		# DOMEENVI.csv adds _1 or _2
		(0x002A0FD,0x002A109):'%s 1',
		(0x002A109,0x002A115):'%s 1',
		(0x002A115,0x002A121):'%s 1',
		(0x002A121,0x002A12D):'%s 1',
		(0x002A12D,0x002A139):'%s 2',
		(0x002A139,0x002A145):'%s 2',
		(0x002A145,0x002A151):'%s 2',
		(0x002A151,0x002A15D):'%s 2',
		(0x002A15D,0x002A169):'%s 2',
		(0x002A169,0x002A175):'%s 1',
		(0x002A175,0x002A181):'%s 2',
		(0x002A181,0x002A18D):'%s 1',
		(0x002A18D,0x002A199):'%s 2',
		(0x002A199,0x002A1A5):'%s 1',
		(0x002A1A5,0x002A1B1):'%s 2',
		(0x002A1B1,0x002A1BD):'%s 1',
		(0x002A1BD,0x002A1C9):'%s 2',
		(0x002A1C9,0x002A1D5):'%s 1',
		(0x002A1D5,0x002A1E1):'%s 2',
		(0x002A1E1,0x002A1ED):'%s 1',
		(0x002A1ED,0x002A1F9):'%s 2',
		(0x002A1F9,0x002A205):'%s 3',
		(0x002A205,0x002A211):'%s 4',
		(0x002A211,0x002A21D):'%s 5',
		(0x002A21D,0x002A229):'%s 6',
		# DOMECT2.csv
		(0x002B00D,0x002B019):'%s 1',
		(0x002B019,0x002B025):'%s 1',
		(0x002B031,0x002B03D):'%s 2',
		(0x002B03D,0x002B049):'%s 2',
		(0x002B049,0x002B055):'%s 1',
		(0x002B055,0x002B061):'%s 1',
		(0x002B06D,0x002B079):'%s 2',
		(0x002B079,0x002B085):'%s 2',
		(0x002B085,0x002B091):'%s 1',
		(0x002B091,0x002B09D):'%s 1',
		(0x002B09D,0x002B0A9):'%s 1',
		(0x002B0A9,0x002B0B5):'%s 1',
		(0x002B0B5,0x002B0C1):'%s 2',
		(0x002B0C1,0x002B0CD):'%s 2',
		(0x002B0CD,0x002B0D9):'%s 2',
		(0x002B0D9,0x002B0E5):'%s 1',
		(0x002B0E5,0x002B0F1):'%s 1',
		(0x002B0F1,0x002B0FD):'%s 2',
		(0x002B0FD,0x002B109):'%s 2',
		(0x002B109,0x002B115):'%s 2',
		(0x002B115,0x002B121):'%s 2',
		(0x002B121,0x002B12D):'%s 1',
		(0x002B12D,0x002B139):'%s 1',
		(0x002B139,0x002B145):'%s 2',
		(0x002B145,0x002B151):'%s 2',
		(0x002B151,0x002B15D):'%s 1',
		(0x002B15D,0x002B169):'%s 1',
		(0x002B169,0x002B175):'%s 2',
		(0x002B175,0x002B181):'%s 2',
		# TLSCPCT2.csv
		(0x002D001,0x002D00D):'%s CH1',
		(0x002D00D,0x002D019):'%s CH2',
		(0x002D019,0x002D025):'%s CH3',
		(0x002D025,0x002D031):'%s CH4',
		(0x002D031,0x002D03D):'%s CH11',
		(0x002D03D,0x002D049):'%s CH12',
		(0x002D049,0x002D055):'%s CH13',
		(0x002D055,0x002D061):'%s CH14',
		(0x002D061,0x002D06D):'%s CH21',
		(0x002D06D,0x002D079):'%s CH22',
		(0x002D079,0x002D085):'%s CH23',
		(0x002D085,0x002D091):'%s CH24',
		(0x002D091,0x002D09D):'%s CH31',
		(0x002D09D,0x002D0A9):'%s CH32',
		(0x002D0A9,0x002D0B5):'%s CH33',
		(0x002D0B5,0x002D0C1):'%s CH34',
		(0x002D0C1,0x002D0CD):'%s CH41',
		(0x002D0CD,0x002D0D9):'%s CH42',
		(0x002D0D9,0x002D0E5):'%s CH43',
		(0x002D0E5,0x002D0F1):'%s CH44',
		(0x002D0F1,0x002D0FD):'%s CH51',
		(0x002D0FD,0x002D109):'%s CH52',
	}
	# shortid:format
	solver_shortid = {
		'0006':'%s on AG',
		'0007':'%s on SV',
		'0037':'%s on ASCU',
		'0027':'%s on BLCU',
		'000E':'%s on CAL',
		'002E':'%s on CLOCK',
		'0024':'%s on CVCU',
		'0001':'%s on DRDR',
		'000D':'%s on FPCI',
		'0004':'%s on FRAD',
		'00B1':'%s on MLP1',
		'00B2':'%s on MLP2',
		'00B3':'%s on MLP3',
		'0002':'%s on MTDR',
		'0003':'%s on SMCU',
		'0033':'%s on SPU4',
		'0030':'%s on THRM',
		'00A1':'%s on TSC',
		'0028':'%s on TTCU',
		'0031':'%s on HT EXH',
	}
	# some IDs have many entries: cannot predefine
	# see mitsubishi.py
	no_predefine = {
		'00B218B':True, '0040001':True, '0040004':True, '0040015':True,
	}

	def __init__(self):
		self.entries = {}	# {name:[entry,...],...}
		# merge the two lists: NameSpace.solver_id and NameSpace.solver_id_range
		self.solver_id = NameSpace.solver_id.copy()
		for minmax in NameSpace.solver_id_range.keys():
			solution = NameSpace.solver_id_range[minmax]
			for id in range(minmax[0],minmax[1]):
				self.solver_id['%07X' % id] = solution
		self.predefined = {}
		self.newdefined = False

	def register(self, name, entry):
		if not self.entries.has_key(name):
			self.entries[name] = []
		self.entries[name].append(entry)

	def remove(self, name, entry):
		self.entries[name].remove(entry)
		if len(self.entries[name]) < 1:
			del(self.entries[name])

	def collisions(self):
		r = []
		for name in self.entries.keys():
			if len(self.entries[name]) > 1:
				r.append(name)
		return r

	def solve_collisions(self):
		changes = []
		for name in self.entries.keys():
			for e in self.entries[name]:
				try:
					e.name_wo_collision
					continue
				except AttributeError:
					if self.predefined.has_key(e.id):	# predefined names
						newname = self.predefined[e.id]
						e.name_wo_collision = newname
						if name != newname:
							changes.append([e,name,newname])
					else:
						newname = None
						if self.solver_id.has_key(e.id):	# id defined default changes
							try:
								newname = self.solver_id[e.id] % e.name
							except TypeError:
								newname = self.solver_id[e.id]
						elif len(self.entries[name]) > 1 and NameSpace.solver_shortid.has_key(e.shortid):	# short ID defined changes
							newname = NameSpace.solver_shortid[e.shortid] % e.name
						if newname:
							e.name_wo_collision = newname
							changes.append([e,name,newname])
						if not NameSpace.no_predefine.has_key(e.id):
							self.newdefined = True
							self.predefined[e.id] = e.keyword()
		for change in changes:
			self.remove(change[1],change[0])
			self.register(change[2],change[0])

	def read_predefined(self, filename):
		try:
			file = open(filename, 'r')
			predefined = file.readlines()
			file.close()
			for entry in predefined:
				if entry[0] == '#':
					continue
				e = entry.strip().split('\t')
				if not NameSpace.no_predefine.has_key(e[0]):
					self.predefined[e[0]] = e[1]
		except IOError:
			pass

	def save_predefined(self, filename):
		file = open(filename, 'w')
		ids = self.predefined.keys()
		ids.sort()
		for id in ids:
			file.write("%s\t%s\n" % (id, self.predefined[id]))
		file.close()

class TelStatusEntries:
	def __init__(self):
		# dictionary of entires:
		# for bit field: type:short-id:byte-offset:bit-offset:entry
		# otherwise:		 type:short-id:byte-offset:'normaldata':entry
		self.entries = {}
		self.namespace = NameSpace()

	# dictionary tree
	def entry_at(self, type, shortid, byteoffset):
		if not self.entries.has_key(type):
			self.entries[type] = {}
		if not self.entries[type].has_key(shortid):
			self.entries[type][shortid] = {}
		if not self.entries[type][shortid].has_key(byteoffset):
			self.entries[type][shortid][byteoffset] = {}
		return self.entries[type][shortid][byteoffset]

	# has a normal datum?
	def has_normaldata(self, type, shortid, byteoffset):
		return self.entry_at(type, shortid, byteoffset).has_key('normaldata')

	# has bit field(s)?
	def has_bitfield(self, type, shortid, byteoffset, bitoffset):
		for bit in range(0,8,1):
			if self.entry_at(type, shortid, byteoffset).has_key(bitoffset):
				return True
		return False

	# add a TelStatusEntry
	def append(self, entry):
		type = entry.type
		shortid = entry.shortid
		byteoffset = entry.byteoffset

		# store the entry
		if entry.is_bitfield():
			# bit field
			bitoffset = entry.bitoffset
			if self.entry_at(type, shortid, byteoffset).has_key(bitoffset):
				old = self.entry_at(type, shortid, byteoffset)[bitoffset]
				if entry.same(old):
					sys.stderr.write('identical entry at %s-%s offset:%d bit:%d id:%s from %s and %s\n' % (type, shortid, byteoffset, bitoffset, entry.id, entry.basename, old.basename))
				else:
					sys.stderr.write('duplicate bit entries at %s-%s offset:%d bit:%d ids:%s,%s from %s and %s\n' % (type, shortid, byteoffset, bitoffset, entry.id, old.id, entry.basename, old.basename))
			else:
				self.entry_at(type, shortid, byteoffset)[bitoffset] = entry
		else:
			# normal data
			if self.has_normaldata(type, shortid, byteoffset):
				old = self.entry_at(type, shortid, byteoffset)['normaldata']
				if entry.same(old):
					sys.stderr.write('identical entry at %s-%s offset:%d id:%s from %s and %s\n' % (type, shortid, byteoffset, entry.id, entry.basename, old.basename))
				else:
					sys.stderr.write('duplicate byte entries at %s-%s offset:%d ids:%s,%s from %s and %s\n' % (type, shortid, byteoffset, entry.id, old.id, entry.basename, old.basename))
			else:
				self.entry_at(type, shortid, byteoffset)['normaldata'] = entry

		# register to the namespace
		self.namespace.register(entry.name, entry)

	# add a dummy entry to create a short id
	def append_dummy(self, type, shortid):
		self.entry_at(type, shortid, 0)

	# list of types of entries
	def types(self):
		return self.entries.keys()

	# list of short-ids of entries
	def shortids(self, type):
		try:
			r = self.entries[type].keys()
		except KeyError:
			r = []
		return r

	# list of byte offsets of entries:
	def byteoffsets(self, type, shortid):
		try:
			r = self.entries[type][shortid].keys()
		except KeyError:
			r = []
		return r

	# length
	def length(self, type, shortid):
		try:
			r = mitsubishi.name_and_total[shortid][type]
		except KeyError:
			bs = self.byteoffsets(type, shortid)
			if len(bs) < 1:
				r = 0
			else:
				bs.sort()
				maxb = bs[-1]
				if self.has_normaldata(type, shortid, maxb):
					r = maxb + self.entry_at(type, shortid, maxb)['normaldata'].bytes()
				else:
					r = maxb + 1
		return r

	# list of bitfields
	def bits(self, type, shortid, byteoffset):
		try:
			r = self.entries[type][shortid][byteoffset].keys()
			try:
				r.remove('normaldata')
			except ValueError:
				pass
		except KeyError:
			r = []
		return r

	# walk through entries
	def to_a(self):
		r = []
		ts = self.types()
		for type in ts:
			ids = self.shortids(type)
			for id in ids:
				bs = self.byteoffsets(type, id)
				for byte in bs:
					try:
						r.append(self.entry_at(type, id, byte)['normaldata'])
					except KeyError:
						pass
					bis = self.bits(type, id, byte)
					for bit in bis:
						r.append(self.entry_at(type, id, byte)[bit])
		r.sort(lambda x,y:cmp(x.id,y.id))
		return r

	# reader functions
	def reader_function_names(self):
		r = {}
		for e in self.to_a():
			for d in 'd', 'i', 's', 't':
				fn = e.reader_function_name(d)
				if fn:
					if not r.has_key(fn):
						r[fn] = []
					r[fn].append(e)
		return r

# unwanted reader functions: format:[destination-types]
#   'd': converting to double, 'i': converting to long long
#   's': coverting to char*,   't': converting to struct time val
no_reader_function = {
	'bit':['d', 't'],
	'str':['d', 'i', 't'],
	'BCD1':['t'],
	'BCD2':['t'],
	'BCD6':['t'],
	'BCD6_DEC':['i', 't'],
	'BCD6_Epoch':['i', 't'],
	'BCD6_RA':['i', 't'],
	'BINARY1_1':['t'],
	'BINARY1_0p01':['t'],
	'BINARY1_0p1':['t'],
	'BINARY2_0p01':['t'],
	'BINARY2_0p1':['t'],
	'BINARY2_1':['t'],
	'BINARY4_0p000000000001':['t'],
	'BINARY4_0p000001':['t'],
	'BINARY4_0p00001':['t'],
	'BINARY4_0p0001':['t'],
	'BINARY4_0p001':['t'],
	'BINARY4_0p01':['t'],
	'BINARY4_0p1':['t'],
	'BINARY4_1':['t'],
	'BINARY4_100':['t'],
	'REAL8':['i', 't'],
}


# Status Entry
class TelStatusEntry(mitsubishi.TelStatusEntry):
	# keyword
	def keyword(self):
		try:
			return self.name_wo_collision
		except AttributeError:
			return self.name

	# bit field mask (0 if not)
	def mask_value(self):
		if self.is_bitfield():
			return '1<<%d' % self.bitoffset
		else:
			return '0'

	# data format: bit, str, ASCx_TIME, BINARY_..., BCD_...
	def data_format(self):
		if self.is_bitfield():
			r = 'bit'
		elif self.format == '': # char
			r = 'str'
		elif self.format == 'ASCII':
			m = re.compile('TIME', re.I)
			if m.search(self.name):
				r = 'ASC%d_TIME' % self.bytes()
			elif self.name == 'LST(TSC Calculate)':
				r = 'ASC%d_TIME' % self.bytes()
			else:
				r = 'str'
		else: # usual member
			r = self.format + str(self.bytes())
			if self.format == 'BINARY':
				s = re.sub('\.', 'p', self.unit)
				s = re.sub('p0$', '', s )
				r += '_' + s
			elif self.format == 'BCD' and self.bytes() == 6:
					if self.name.find( 'Right Ascention' ) >= 0 and self.name.find( '1000' ) < 0:
						r += '_RA'
					elif self.name.find( 'Declination' ) >= 0 and self.name.find( '1000' ) < 0:
						r += '_DEC'
					elif self.name == 'Epoch':
						r += '_Epoch'
		return r

	def reader_function_name(self, dstformat):
		format = self.data_format()
		try:
			no_reader_function[format].index(dstformat)
			return None
		except ValueError:
			pass
		except KeyError:
			pass
		return '%s_to_%s' % (self.data_format(), dstformat)

	def array_entry(self, number):
		name = self.keyword()
		ext = extype(self.type, self.shortid)
		shortid = realshortid(self.type, self.shortid)
		id = self.id
		offset = self.byteoffset
		bytes = self.bytes()
		mask = self.mask_value()
		tod = self.reader_function_name('d')
		if not tod:
			tod = 'NULL'
		toi = self.reader_function_name('i')
		if not toi:
			toi = 'NULL'
		tos = self.reader_function_name('s')
		if not tos:
			tos = 'NULL'
			tol = 'NULL'
		else:
			tol = self.reader_function_name('s_length')
		tot = self.reader_function_name('t')
		if not tot:
			tot = 'NULL'
		if self.isalarm:
			isalarm = 1
		else:
			isalarm = 0
		if self.iswarning:
			iswarning = 1
		else:
			iswarning = 0
		if self.isfault:
			isfault = 1
		else:
			isfault = 0
		return '{"%s", "%s", 0x%s, 0x%s, %d, %d, %s, %d, %d, %d, %s, %s, %s, %s, %s},\t/* %d */\n' % (name, ext, shortid, id, offset, bytes, mask, isalarm, iswarning, isfault, tod, toi, tos, tol, tot, number)

	def hash_entry(self, number, entry_info_array_name):
		return '"%s", &%s[%d]\n' % (self.keyword(), entry_info_array_name, number)

	def sort_string(self):
		t = self.type
		s = self.shortid
		r = "%s %-2.2s %s %05d %d" % (realshortid(t, s), extype(t, s), realshortid(t, s), self.byteoffset, self.bitoffset)
		return r

#class TelStatusCsv(mitsubishi.TelStatusCsv):
	# parse
#	def parse(self):
#		r = []
#		for col in csv.reader(file(self.path)):
#			x = self.corrected(col)
#			for c in self.corrected(col):
#				e = TelStatusEntry(c[1], c[2], c[6], c[7], c[8], c[9], c[12], c[13], c[14], c[15], self.basename, {'ja':c[5]})
#				r.append(e)
#		return r

# a CSV file from Mitsubishi
class TelStatusCsv:
	def __init__(self, path):
		self.path = path
		self.basename = os.path.basename(self.path).split('.')[0]

	# Zernike polynominal terms
	zernike_terms = [
		'00', '11', '1m1', '20', '31', '3m1', '22', '2m2', '33', '3m3',
		'40', '42', '4m2', '44', '4m4', '51', '5m1', '53', '5m3', '55',
		'5m5', '60', '62', '6m2', '64', '6m4', '66', '6m6', '71', '7m1',
		'73', '7m3', '75', '7m5', '77', '7m7', '80', '82', '8m2', '84',
		'8m4', '86', '8m6', '88', '8m8', '91', '9m1', '93', '9m3', '95',
		'9m5', '97', '9m7', '99', '9m9', '100', '102', '10m2', '104', '10m4',
		'106', '10m6', '108', '10m8', '1010', '10m10', '', '', '', ''
	]

	# correction for typos in Mitsubishi CSV
	# input:output
	# col[0]	: entry number within the file
	# col[1]	: `name' of the status (string)
	# col[2]	: number of bits (int)
	# col[3]	: positive/negative logic
	# col[4]	: source interface
	# col[5]	: description in Japanese (string, encoding depends upon input)
	#	         Python >= 2.4 is needed to convert strings to UTF-8
	#	         with a standard Python library.
	# col[6]	: whether this is a source of alarm
	# col[7]	: whether this is a source of warning
	# col[8]	: whether this is a source of failure
	# col[9]	: format of numeric data (upper-cased string)
	# col[10] : exclusive signlals
	# col[11] : remarks
	# col[12] : status ID (zero-filled string)
	# col[13] : E:event driven/variable, L:long period, S:short period (string)
	# col[14] : byte offset (int)
	# col[15] : bit offset or decimal position (string)
	# returns a list of normalized and type-converted status entries

	def corrected(self, col):
		e = col[:]
		# number of columns
		if len(col) < 15:
			return []
		# correct O (Capital-O) to 0 (zero)
		e[12] = re.sub('O', '0', e[12]) # MCP1MON.csv
		e[14] = re.sub('O', '0', e[14]) # TLSCPCT2.csv and TLSCPENV.csv
		e[15] = re.sub('O', '0', e[15]) # DOMECT2.csv and DOMEENVI.csv
		# byte offset
		try:
			e[14] = int(e[14])
		except ValueError:	# we can ignore invalid entries
			return []
		# ID
		if len(e[12]) == 0:	# we can ignore entries with an empty ID
			return []
		# fill ID with zeros
		e[12] = e[12].zfill(7)
		# skipped numbers: "ketsu-ban" in Shift-JIS
		if e[1] == '\x8c\x87\x94\xd4':
			return []
		# for future use
		elif e[12] == '0006049':
			return []
		# some names have a trailing space
		e[1] = e[1].strip()
		# some decimal positions have a trailing space
		e[15] = e[15].strip()
		# number of bits
		e[2] = int(e[2])
		# decription
		# e[5] = some way to convert encoding
		# format
		e[9] = string.upper(e[9])

		# ID specfic typos and expansions
		idnum = string.atoi(e[12],16)
		# id:002B07F seems not to be a BCD but a bit field 
		if e[12] == '002B07F':
			e[9] = ''
		# id:000E02C in CALMON.csv has a typo
		elif e[12] == '000E02C':
			e[1] = 'Cal Source TurretD Holder3 Selected'
		# id:002A0FC in DOMEENVI.csv has a typo
		elif e[12] == '002A0FC':
			e[1] = 'Ns FL Rear V WVEL Data'
		# id:002A174 in DOMEENVI.csv has a typo
		elif e[12] == '002A174':
			e[1] = 'Top FL Rear V WVEL Data'
		# id:002C295-002C2A0 in TLSCPENV.csv have typos
		elif 0x002C295 <= idnum and idnum < 0x002C2A1:
			e[1] = re.sub('IR-B', 'Opt-B', e[1])
		# id:002D100 in TLSCPCT2.csv has a typo
		elif e[12] == '002D100':
			e[1] = 'CSCT(F-IR) CT2 OVER RNG'
		# id:00B218B has to be expanded to many of Zernike polynominals
		# source: MLP2-MLP2_1.h by Morino
		elif e[12] == '00B218B':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 64
			template[9] = 'REAL'
			for i in range(0,len(TelStatusCsv.zernike_terms)):
				if len(TelStatusCsv.zernike_terms[i]) > 0:
					template[1] = "%s %02d A%s" % (e[1],i,TelStatusCsv.zernike_terms[i])
				else:
					template[1] = "%s %02d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040001 has to be expanded to many of Zernike polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040001':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 16
			template[9] = 'BINARY'
			template[15] = '0.01'
			for i in range(1,66):
				template[1] = "%s %02d A%s" % (e[1],i,TelStatusCsv.zernike_terms[i])
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040004 has to be expanded to many polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040004':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 64
			template[9] = 'REAL'
			for i in range(1,33):
				template[1] = "%s q%02d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040015 has to be expanded to many polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040015':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 16
			template[9] = 'BINARY'
			template[15] = '0.01'
			for i in range(0,192):
				template[1] = "%s %03d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# entries in PMAMON.csv has to be expanded to those of #000-260
		# also for name_and_total
		# http://www1.naoj.org/telgroup/TSC_Status_html/sig/Pmamon.htm
		# http://www1.naoj.org/telgroup/TSC_Command.html/Individual/TWS/TWS_2_3.htm
		elif e[12][0:4] == '0101':
			r = []
			template = e[:]
			for n1 in range(1,9):
				for n2 in range(0,7):
					for n3 in range(0,10):
						numstr = "%d%d%d" % (n1, n2, n3)
						template[1] = "%s%d-%d%d%s" % (e[1][0:8], n1, n2, n3, e[1][12:])
						template[12] = e[12][0] + numstr + e[12][4:7]
						r.append(template[:])
						mitsubishi.name_and_total[template[12][0:4]] = {'name':'PMA-'+numstr}
			return r
		return [e]

	# parse
	def parse(self):
		r = []
		for col in csv.reader(file(self.path)):
			for c in self.corrected(col):
				if c[6] == "1":
					isalarm = True
				else:
					isalarm = False
				if c[7] == "1":
					iswarning = True
				else:
					iswarning = False
				if c[8] == "1":
					isfault = True
				else:
					isfault = False
				e = TelStatusEntry(c[1], c[2], isalarm, iswarning, isfault, c[9], c[12], c[13], c[14], c[15], self.basename, {'ja':c[5]})
				r.append(e)
		return r
