/* shm-interface.c: interface between shared memories and InstrumentStatus */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <assert.h>

#include <melco/tsc-message.h>
#include <melco/shm-interface.h>
#include <melco/snprintf-compat.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

/* private functions */
typedef int (file_open_callback_t)(const char *name, int flags, mode_t mode);
typedef int (file_unlink_callback_t)(const char *pathname);
struct _scanner_args_s {
	char *rootdir;
	file_open_callback_t *openfunc;
};
static InstrumentStatus* _InstrumentStatus_serialize(InstrumentStatus *self, file_open_callback_t openfunc, const char *rootdir);
static InstrumentStatus* _serialized_get_instrument(const char *type, const unsigned int shortid, file_open_callback_t openfunc, const char *rootdir);
static int _instrumentstatus_unlink_serialized(const char *type, const unsigned int shortid, file_unlink_callback_t unlinkfunc, const char *rootdir);
static int _instrumentstatus_serialized_unlink_all(file_unlink_callback_t unlinkfunc, const char *rootdir);
static char *_concat_path_malloc(const char *rootdir, const char *basename);
static StatusEntry* _serialized_get_status_entry(const char *keyword, file_open_callback_t openfunc, const char *rootdir);
#define _CONCAT_PATH_FORMAT "%s/%s"

/* remember mapped addresses per process per rootdir */
typedef struct _mapped_inststat_list_s {
	char *rootdir;
	InstrumentStatus **mapped_stat;
	int ninst;
	struct _mapped_inststat_list_s *next;
} mapped_inststat_list_s; 
static mapped_inststat_list_s mapped_inststat_list = {NULL, NULL, 0, NULL};
/* the first element will hold the array for shared memory */

static mapped_inststat_list_s *_mapped_instrument_status_find_or_alloc(const char *rootdir);
static InstrumentStatus *_get_mapped_instrument_status(const char *rootdir, int inst_index);
static InstrumentStatus *_set_mapped_instrument_status(const char *rootdir, int inst_index, InstrumentStatus *stat);
static int _instrumentstatus_serialized_unmap_all(const char *rootdir);
/* deregister and free self */
static void _mapped_instrument_status_free(void *self);

/* callback from scanner */
static int _MonitorDataMessage_scan_to_serialized(MonitorDataMessage* parent, void* arg __attribute__ ((unused)), struct melco_tree_s* info __attribute__ ((unused)), instrument_status_header_s *inst);

static void
_mapped_instrument_status_free(void *ptr)
{
	mapped_inststat_list_s *cur, *prev;
	struct melco_tree_s *info;
	InstrumentStatus *self;
	char key[MELCO_TREE_MAX_WORD_LENGTH + 1];

	self = (InstrumentStatus *) ptr;

	treekey(key, self->type, self->shortid);
	info = melco_tree_lookup(key, strlen(key));
	assert(info);

	prev = NULL;
	cur = &mapped_inststat_list;
	while(cur)
		{
			if (cur->mapped_stat && cur->mapped_stat[info->index] == self)
				{
					cur->mapped_stat[info->index] = NULL;
					munmap((void*) self->d, instrument_status_total_size(self->type, self->shortid));
					cur->ninst--;
					if (cur->ninst == 0)
						{
							if (cur->rootdir) free(cur->rootdir);
							cur->rootdir = NULL;
							free(cur->mapped_stat);
							cur->mapped_stat = NULL;
							if (prev)
								{
									prev->next = cur->next;
									free(cur);
								}
						}
				}
			prev = cur;
			cur = cur->next;
		}
	free(self);
}

InstrumentStatus*
InstrumentStatus_to_shm(InstrumentStatus *self)
{
	return _InstrumentStatus_serialize(self, shm_open, NULL);
}

static InstrumentStatus*
_InstrumentStatus_serialize(InstrumentStatus *self, file_open_callback_t openfunc, const char *rootdir)
{
	int fd;
	char name[MELCO_TREE_MAX_WORD_LENGTH + 2];
	char *path;
	void *dst;
	InstrumentStatus* r;

	r = self;
	instrumentstatus_shm_name(name, self->type, self->shortid);
	path = _concat_path_malloc(rootdir, name);
	if (!path) return NULL;
	fd = openfunc(path, O_RDWR | O_CREAT, 00644);
	free(path);
	if (fd == -1) return NULL;
	if (ftruncate(fd, self->length) == -1) return NULL;
	dst = mmap(0, self->length, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if (close(fd) != 0) return NULL;
	if (dst == MAP_FAILED) return NULL;
	memcpy(dst, self->d, self->length);
	if (munmap(dst, self->length) != 0) r = NULL;
	return r;
}

char *
instrumentstatus_shm_name(char *dst, const char *type, const unsigned int shortid)
{
	dst[0] = '/';
	treekey(dst + 1, type, shortid);
	dst[MELCO_TREE_MAX_WORD_LENGTH + 1] = '\0';
	return dst;
}

int
instrumentstatus_shm_unlink(const char *type, const unsigned int shortid)
{
	return _instrumentstatus_unlink_serialized(type, shortid, shm_unlink, NULL);
}

static int
_instrumentstatus_unlink_serialized(const char *type, const unsigned int shortid, file_unlink_callback_t unlinkfunc, const char *rootdir)
{
	int r;
	char name[MELCO_TREE_MAX_WORD_LENGTH + 2];
	char *path;

	instrumentstatus_shm_name(name, type, shortid);
	path = _concat_path_malloc(rootdir, name);
	if (!path) return 1;
	r = unlinkfunc(path);
	free(path);
	if (r == -1)
		return 1;	/* error */
	else
		return 0;	/* success */
}

int
instrumentstatus_shm_unlink_all(void)
{
	return _instrumentstatus_serialized_unlink_all(shm_unlink, NULL);
}

int
instrumentstatus_file_unlink_all(const char *rootdir)
{
	return _instrumentstatus_serialized_unlink_all(shm_unlink, rootdir);
}

static int
_instrumentstatus_serialized_unlink_all(file_unlink_callback_t unlinkfunc, const char *rootdir)
{
	int i, r = 0;
	char name[MELCO_TREE_MAX_WORD_LENGTH + 2];
	char *path;

	for(i = 0; i < MELCO_TREE_LENGTH; i++)
		{
			instrumentstatus_shm_name(name, melco_tree_type[i], melco_tree_shortid[i]);
			path = _concat_path_malloc(rootdir, name);
			if (!path) return -1;
			if (unlinkfunc(path) == 0) r++;
			free(path);
		}
	return r;
}

static int
_MonitorDataMessage_scan_to_serialized(MonitorDataMessage* parent, void* args, struct melco_tree_s* info, instrument_status_header_s *inst)
{
	InstrumentStatus *t, *r;

	/* prepare InstrumentStatus */
	t = InstrumentStatus_alloc(parent);
	if (!t) return -1;
	InstrumentStatus_set(t, inst, NULL, info);

	/* register */
	r = _InstrumentStatus_serialize(t, ((struct _scanner_args_s *) args)->openfunc, ((struct _scanner_args_s *) args)->rootdir);

	/* clean up */
	InstrumentStatus_decref(t);
	return r ? 1 : -1;
}

int
MonitorDataMessage_scan_to_shm(MonitorDataMessage* self, scan_option_t opt)
{
	struct _scanner_args_s args = {NULL, shm_open};
	return MonitorDataMessage_scanner(self, &args, opt, _MonitorDataMessage_scan_to_serialized);
}

int
MonitorDataMessage_scan_to_file(MonitorDataMessage* self, scan_option_t opt, const char *rootdir)
{
	int r;
	struct _scanner_args_s args = {rootdir, open};
	r = MonitorDataMessage_scanner(self, &args, opt, _MonitorDataMessage_scan_to_serialized);
	return r;
}

InstrumentStatus*
shm_get_instrument(const char *type, const unsigned int shortid)
{
	return _serialized_get_instrument(type, shortid, shm_open, NULL);
}

static InstrumentStatus*
_serialized_get_instrument(const char *type, const unsigned int shortid, file_open_callback_t openfunc, const char *rootdir)
{
	InstrumentStatus* r;
	struct melco_tree_s *info;
	char key[MELCO_TREE_MAX_WORD_LENGTH + 1];

	treekey(key, type, shortid);
	info = melco_tree_lookup(key, strlen(key));
	if (!info) return NULL;

	r = _get_mapped_instrument_status(rootdir, info->index);
	if (r)
		{
			return r;
		}
	else
		{
			int fd;
			int length;
			char name[MELCO_TREE_MAX_WORD_LENGTH + 2];
			char *path;
			void *src;

			length = instrument_status_total_size(type, shortid);
			instrumentstatus_shm_name(name, type, shortid);
			path = _concat_path_malloc(rootdir, name);
			if (!path) return NULL;
			fd = openfunc(path, O_RDONLY, 0);
			free(path);
			if (fd == -1) return NULL;
			src = mmap(0, length, PROT_READ, MAP_SHARED, fd, 0);
			if (src == MAP_FAILED)
				{
					close(fd);
					return NULL;
				}

			r = InstrumentStatus_alloc(NULL);
			if (!r) return NULL;
			InstrumentStatus_set(r, (instrument_status_header_s *) src, NULL, info);
			/* we will free (actually munmap) the data segment in _mapped_instrument_status_free */
			_set_mapped_instrument_status(rootdir, info->index, r);
			status_content_type_set(type, r->type);
			r->self_free = _mapped_instrument_status_free;
			close(fd);

			return r;
		}
}

StatusEntry*
shm_get_status_entry(const char *keyword)
{
	return _serialized_get_status_entry(keyword, shm_open, NULL);
}

StatusEntry*
file_get_status_entry(const char *keyword, const char *rootdir)
{
	return _serialized_get_status_entry(keyword, open, rootdir);
}

static StatusEntry*
_serialized_get_status_entry(const char *keyword, file_open_callback_t openfunc, const char *rootdir)
{
	InstrumentStatus* inst;
	StatusEntry *entry;
	struct melco_status_s* info;
	info = melco_status_lookup(keyword, strlen(keyword));
	if (!info) return NULL;
	inst = _serialized_get_instrument(info->type, info->shortid, openfunc, rootdir);
	if (!inst) return NULL;
	entry = StatusEntry_alloc(inst);
	if (!entry) return NULL;
	entry->d = (char*) inst->d + sizeof(instrument_status_header_s) + info->offset;
	entry->d_free = NULL;
	entry->info = info;
	return entry;
}

int
instrumentstatus_shm_unmap_all(void)
{
	return _instrumentstatus_serialized_unmap_all(NULL);
}

int
instrumentstatus_file_unmap_all(const char *rootdir)
{
	return _instrumentstatus_serialized_unmap_all(rootdir);
}

static int
_instrumentstatus_serialized_unmap_all(const char *rootdir)
{
	int i, r = 0;
	mapped_inststat_list_s *cur;

	cur = _mapped_instrument_status_find_or_alloc(rootdir);
	if (!cur) return 0;

	if (cur->ninst == 0)
		{
			if (cur->mapped_stat)
				{
					free(cur->mapped_stat);
					cur->mapped_stat = NULL;
				}
			if (cur->rootdir) free(cur->rootdir);
			if (&mapped_inststat_list != cur) free(cur);
			return 0;
		}

	for(i = 0; i < MELCO_TREE_LENGTH; i++)
		{
			if (cur->mapped_stat[i])
				{
					InstrumentStatus_decref(cur->mapped_stat[i]);
					r++;
				}
			if (!cur->mapped_stat) break;
		}
	return r;
}

static char *
_concat_path_malloc(const char *rootdir, const char *basename)
{
	char *r;
	int l;

	if (!rootdir) return strdup(basename);

	l = snprintf_length(_CONCAT_PATH_FORMAT, rootdir, basename) + 1;
	if (l < 1) return NULL;

	r = (char *) malloc(l);
	if (!r) return NULL;

	snprintf(r, l, _CONCAT_PATH_FORMAT, rootdir, basename);
	return r;
}

static mapped_inststat_list_s *
_mapped_instrument_status_find_or_alloc(const char *rootdir)
{
	mapped_inststat_list_s *cur;

	cur = &mapped_inststat_list;
	if (rootdir)	/* find the appropriate cache from the linked list */
		{
			while(cur->rootdir && strcmp(cur->rootdir, rootdir) && cur->next)
				{
					cur = cur->next;
				}
			if (!cur->next && (!cur->rootdir || strcmp(cur->rootdir, rootdir)))	/* not found */
				{
					cur->next = (mapped_inststat_list_s *) malloc(sizeof(mapped_inststat_list_s));
					if (!cur->next) return NULL;
					cur->next->rootdir = strdup(rootdir);
					if (!cur->next->rootdir)
						{
							free(cur->next);
							cur->next = NULL;
							return NULL;
						}
					cur = cur->next;
					cur->mapped_stat = NULL;
					cur->ninst = 0;
					cur->next = NULL;
				}
		}

	if (!cur->mapped_stat)
		{
			cur->mapped_stat = (InstrumentStatus **) malloc(sizeof(InstrumentStatus *)*MELCO_TREE_LENGTH);
			if (!cur->mapped_stat) return NULL;
			memset(cur->mapped_stat, 0, sizeof(InstrumentStatus *)*MELCO_TREE_LENGTH);
			cur->ninst = 0;
		}

	return cur;
}

static InstrumentStatus *
_get_mapped_instrument_status(const char *rootdir, int inst_index)
{
	mapped_inststat_list_s *cur;
	cur = _mapped_instrument_status_find_or_alloc(rootdir);
	if (!cur) return NULL;
	return cur->mapped_stat[inst_index];
}

static InstrumentStatus *
_set_mapped_instrument_status(const char *rootdir, int inst_index, InstrumentStatus *stat)
{
	mapped_inststat_list_s *cur;
	cur = _mapped_instrument_status_find_or_alloc(rootdir);
	if (!cur) return NULL;
	if (!cur->mapped_stat[inst_index])
		{
			cur->mapped_stat[inst_index] = stat;
			cur->ninst++;
		}
	return cur->mapped_stat[inst_index];
}
