/* statusbank.c: tscstatus.statusbank */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

 */

#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#include <Python.h>

#include <string.h>
#include <assert.h>

#include <tscstatus.h>
#include <melco/tsc-message.h>
#include <melco/status-hash.h>

void
P_StatusBank_dealloc(P_StatusBank *self)
{
	StatusBank_discard_all_inst(self->cdata);
	free(self->cdata);
	self->ob_type->tp_free((PyObject*)self);
}

PyObject*
P_StatusBank_new(PyTypeObject *type, PyObject *args __attribute__ ((unused)))
{
	P_StatusBank *self;

	self = (P_StatusBank *)type->tp_alloc(type, 0);
	if (!self) return (PyObject*) self;

	self->cdata = (StatusBank *) malloc(sizeof(StatusBank));
	if (!self->cdata)
		{
			Py_XDECREF(self);
			PyErr_NoMemory();
			return NULL;
		}
	StatusBank_init(self->cdata);

	return (PyObject*) self;
}

PyObject *
P_StatusBank_scan(P_StatusBank *self, PyObject *args)
{
	PyObject *p_file, *p_block, *p_scan;
	FILE *stream;
	scan_option_t scan_opt;
	message_blocking_t block_opt;
	long blocksize;
	MonitorDataMessage *m;
	long result;

	/* parse arguments - stream, blocking method, all/strict */
	p_block = NULL;
	p_scan = NULL;
	if (!PyArg_ParseTuple(args, "O|OO", &p_file, &p_block, &p_scan)) return NULL;

	/* first argument - stream */
	Py_XINCREF(p_file);
	if (!PyFile_Check(p_file))
		{
			Py_XDECREF(p_file);
			PyErr_Format(PyExc_ValueError, "first argument must be a file");
			return NULL;
		}
	stream = PyFile_AsFile(p_file);

	/* second argument - blocking */
	block_opt = block_none;
	blocksize = 0;
	if (p_block)
		{
			Py_XINCREF(p_block);
			if (PyInt_Check(p_block))
				{
					block_opt = block_exact;
					blocksize = PyInt_AsLong(p_block);	/* exact block size given */
				}
			else
				{
					if (PyString_Check(p_block))
						{
							char *method;
							method = PyString_AS_STRING(p_block);
							if (!strcmp("none", method))
								{
									block_opt = block_none;
									goto block_done;
								}
							if (!strcmp("fujitsu", method))
								{
									block_opt = block_fujitsu;
									goto block_done;
								}
							if (!strcmp("tws4", method))
								{
									block_opt = block_tws4;
									goto block_done;
								}
							/* otherwise */
							PyErr_Format(PyExc_ValueError, "block size for `%s' not known", method);
							Py_XDECREF(p_file);
							Py_XDECREF(p_block);
							return NULL;
						}
					else
						{
							PyErr_Format(PyExc_ValueError, "second optional argument must be an int (block size) or string (\"fujitsu\" or \"tws4\")");
							Py_XDECREF(p_file);
							Py_XDECREF(p_block);
							return NULL;
						}
				}
			block_done:
			Py_XDECREF(p_block);
		}

	/* third argument - scanning option */
	scan_opt = scan_all;
	if (p_scan)
		{
			Py_XINCREF(p_scan);
			if (p_scan == Py_True)
				{
					scan_opt = scan_all;
					goto scan_done;
				}
			if (p_scan == Py_False)
				{
					scan_opt = scan_strict;
					goto scan_done;
				}
			if (PyString_Check(p_scan))
				{
					char *scanning;
					scanning = PyString_AS_STRING(p_scan);
					if (!strcmp("all", scanning))
						{
							scan_opt = scan_all;
							goto scan_done;
						}
					if (!strcmp("strict", scanning))
						{
							scan_opt = scan_strict;
							goto scan_done;
						}
					/* otherwise */
					PyErr_Format(PyExc_ValueError, "scanning option `%s' not known", scanning);
					Py_XDECREF(p_file);
					Py_XDECREF(p_block);
					Py_XDECREF(p_scan);
					return NULL;
				}
			else
				{
					PyErr_Format(PyExc_ValueError, "third optional argument must be a True/\"all\" (default) or False/\"strict\"");
					Py_XDECREF(p_file);
					Py_XDECREF(p_block);
					return NULL;
				}
			scan_done:
			Py_XDECREF(p_scan);
		}


	/* read the file */
	{
		tsc_error_t r;

		m = MonitorDataMessage_alloc();
		if (!m)
			{
				Py_XDECREF(p_file);
				PyErr_NoMemory();
				return NULL;
			}
		r = fread_MonitorDataMessage(m, stream, block_opt, blocksize);

		if (r != tsc_success)
			{
				PyObject *p_filename;
				char *filename;

				p_filename = PyFile_Name(p_file);
				Py_XINCREF(p_filename);
				filename = PyString_AsString(p_filename);
				
				switch(r)
					{
						case tsc_error_nostream:
							PyErr_SetFromErrnoWithFilename(PyExc_EOFError, filename);
							Py_XDECREF(p_filename);
							Py_XDECREF(p_file);
							return NULL;
						case tsc_error_stream:
							PyErr_SetFromErrnoWithFilename(PyExc_IOError,	filename);
							Py_XDECREF(p_filename);
							Py_XDECREF(p_file);
							return NULL;
						case tsc_error_malloc:
							PyErr_NoMemory();
							Py_XDECREF(p_filename);
							Py_XDECREF(p_file);
							return NULL;
						case tsc_error_malformed:
							PyErr_Format(P_FileFormatError, "malformed header in %s", filename);
							Py_XDECREF(p_filename);
							Py_XDECREF(p_file);
							return NULL;
						default:	/* should not reach here */
							assert(0);
					}
				Py_XDECREF(p_filename);
			}
	}
	Py_XDECREF(p_file);

	/* scan the message */
	result = MonitorDataMessage_scan(m, self->cdata, scan_opt);

	return PyInt_FromLong(result);
}

PyObject *
P_StatusBank_get_entry(P_StatusBank *self, PyObject *args)
{
	PyObject *p_name, *r;
	StatusEntry *e;
	char *name;

	/* parse arguments - name of the requested entry */
	if (!PyArg_ParseTuple(args, "S", &p_name)) return NULL;
	Py_XINCREF(p_name);
	name = PyString_AsString(p_name);

	/* check name */
	if (!melco_status_lookup(name, strlen(name)))
		{
			PyErr_Format(PyExc_KeyError, "no status entry named \"%s\"", name);
			Py_XDECREF(p_name);
			return NULL;
		}

	/* look up */
	e = StatusBank_get_status_entry(self->cdata, name);
	if (!e)
		{
			PyErr_Format(PyExc_KeyError, "no status entry named \"%s\"", name);
			Py_XDECREF(p_name);
			return NULL;
		}
	Py_XDECREF(p_name);

	r = (PyObject*) P_StatusEntry_new(&P_StatusEntryType, Py_None);
	P_StatusEntry_set((P_StatusEntry*) r, e);
	Py_XINCREF(r);

	return r;
}


PyMemberDef P_StatusBank_members[] = {
	{NULL, 0, 0, 0, NULL}	/* Sentinel */
};

PyMethodDef P_StatusBank_methods[] = {
	{"scan", (PyCFunction)P_StatusBank_scan, METH_VARARGS,
	 "scans the file"},
	{"entry", (PyCFunction)P_StatusBank_get_entry, METH_VARARGS,
	 "retrieves a statusentry"},
	{"e", (PyCFunction)P_StatusBank_get_entry, METH_VARARGS,
	 "retrieves a statusentry"},
	{NULL, NULL, 0, NULL}	/* Sentinel */
};

PyTypeObject P_StatusBankType = {
	PyObject_HEAD_INIT(NULL)
	0,                                   /* ob_size */
	"tscstatus.statusbank",              /* tp_name */
	sizeof(P_StatusBank),                /* tp_basicsize */
	0,                                   /* tp_itemsize */
	(destructor)P_StatusBank_dealloc,    /* tp_dealloc */
	0,                                   /* tp_print */
	0,                                   /* tp_getattr */
	0,                                   /* tp_setattr */
	0,                                   /* tp_compare */
	0,                                   /* tp_repr */
	0,                                   /* tp_as_number */
	0,                                   /* tp_as_sequence */
	0,                                   /* tp_as_mapping */
	0,                                   /* tp_hash */
	0,                                   /* tp_call */
	0,                                   /* tp_str */
	0,                                   /* tp_getattro */
	0,                                   /* tp_setattro */
	0,                                   /* tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,
	                                     /* tp_flags */
	"statns entries from a stream",      /* tp_doc */
	0,                                   /* tp_traverse */
	0,                                   /* tp_clear */
	0,                                   /* tp_richcompare */
	0,                                   /* tp_weaklistoffset */
	0,                                   /* tp_iter */
	0,                                   /* tp_iternext */
	P_StatusBank_methods,                /* tp_methods */
	P_StatusBank_members,                /* tp_members */
	0,                                   /* tp_getset */
	0,                                   /* tp_base */
	0,                                   /* tp_dict */
	0,                                   /* tp_descr_get */
	0,                                   /* tp_descr_set */
	0,                                   /* tp_dictoffset */
	0,                                   /* tp_init */
	0,                                   /* tp_alloc */
	(newfunc)P_StatusBank_new,           /* tp_new */
	0,                                   /* tp_free */
	0,                                   /* tp_is_gc */
	0,                                   /* tp_bases */
	0,                                   /* tp_mro */
	0,                                   /* tp_cache */
	0,                                   /* tp_subclasses */
	0,                                   /* tp_weaklist */
	0,                                   /* tp_del */
};

