/* entryinfo.c: tscstatus.entryinfo */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

 */

#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#include <Python.h>

#include <tscstatus.h>
#include <melco/status-hash.h>

void
P_EntryInfo_dealloc(P_EntryInfo *self)
{
	self->ob_type->tp_free((PyObject*)self);
}

PyObject*
P_EntryInfo_new(PyTypeObject *type, PyObject *args __attribute__ ((unused)))
{
	P_EntryInfo *self;

	self = (P_EntryInfo *)type->tp_alloc(type, 0);
	if (!self) return (PyObject *)self;

	self->cdata = NULL;	/* will point to a static address from status-hash.h */
	return (PyObject *)self;
}

int
P_EntryInfo_init(P_EntryInfo *self, PyObject *args)
{
	PyObject *arg = NULL;

	if (!PyArg_ParseTuple(args, "O", &arg))
		return -1;

	Py_XINCREF(arg);
	if (PyString_Check(arg))	/* argument as entry name */
		{
			self->cdata = melco_status_lookup(PyString_AsString(arg), PyString_Size(arg));
			Py_XDECREF(arg);
			if (!self->cdata)
				{
					PyErr_Format(PyExc_KeyError, "no status entry named \"%s\"", PyString_AsString(arg));
					return -1;
				}
			return 0;
		}
	if (PyInt_Check(arg))	/* argument as entry number */
		{
			long i;
			i = PyInt_AsLong(arg);
			Py_XDECREF(arg);
			if (i > INT_MAX)
				{
					PyErr_Format(PyExc_ValueError, "entry number too big (should be <= %d)", INT_MAX);
					return -1;
				}
			if (i < INT_MIN)
				{
					PyErr_Format(PyExc_ValueError, "entry number too small (should be >= %d)", INT_MIN);
					return -1;
				}
			self->cdata = melco_status_ith((int) i);
			if (!self->cdata)
				{
					PyErr_Format(PyExc_KeyError, "entry number out of range (should be >= 0 and < %d)", melco_status_total());
					return -1;
				}
			return 0;
		}
	PyErr_Format(PyExc_TypeError, "argument should be a string or an integer");
	return -1;
}

PyObject *
P_EntryInfo_name(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyString_FromString(self->cdata->name);
}

PyObject *
P_EntryInfo_type(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyString_FromString(self->cdata->type);
}

PyObject *
P_EntryInfo_shortid(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyInt_FromLong(self->cdata->shortid);
}

PyObject *
P_EntryInfo_id(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyInt_FromLong(self->cdata->id);
}

PyObject *
P_EntryInfo_offset(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyInt_FromLong(self->cdata->offset);
}

PyObject *
P_EntryInfo_length(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyInt_FromLong(self->cdata->bytes);
}

PyObject *
P_EntryInfo_mask(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	return PyInt_FromLong(self->cdata->mask);
}

PyObject *
P_EntryInfo_is_flag(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->mask != 0)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_is_numeric(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->mask == 0 && !self->cdata->to_t && (self->cdata->to_i || self->cdata->to_d))
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_is_string(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->mask == 0 && !self->cdata->to_t && !self->cdata->to_i && !self->cdata->to_d && self->cdata->to_s)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_is_time(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->mask == 0 && self->cdata->to_t)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_parentid(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	char buf[MELCO_TREE_MAX_WORD_LENGTH + 1];
	treekey(buf, self->cdata->type, self->cdata->shortid);
	return PyString_FromString(buf);
}

PyObject *
P_EntryInfo_is_alarm(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->isalarm != 0)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_is_warning(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->iswarning != 0)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

PyObject *
P_EntryInfo_is_fault(P_EntryInfo *self, PyObject *args __attribute__ ((unused)))
{
	if (self->cdata->isfault != 0)
		{
			Py_XINCREF(Py_True);
			return Py_True;
		}
	Py_XINCREF(Py_False);
	return Py_False;
}

void
P_EntryInfo_set(P_EntryInfo *self, const struct melco_status_s *info)
{
	self->cdata = info;
}

PyMemberDef P_EntryInfo_members[] = {
	{NULL, 0, 0, 0, NULL}	/* Sentinel */
};

PyMethodDef P_EntryInfo_methods[] = {
	{"name", (PyCFunction)P_EntryInfo_name, METH_NOARGS,
	 "return name of status entry"},
	{"type", (PyCFunction)P_EntryInfo_type, METH_NOARGS,
	 "return type of status entry"},
	{"shortid", (PyCFunction)P_EntryInfo_shortid, METH_NOARGS,
	 "return shortid (instrument ID) of status entry"},
	{"id", (PyCFunction)P_EntryInfo_id, METH_NOARGS,
	 "return status entry ID"},
	{"offset", (PyCFunction)P_EntryInfo_offset, METH_NOARGS,
	 "return byte offset of status entry"},
	{"length", (PyCFunction)P_EntryInfo_length, METH_NOARGS,
	 "return number of bytes in status entry"},
	{"mask", (PyCFunction)P_EntryInfo_mask, METH_NOARGS,
	 "return bit field mask of status entry"},
	{"is_flag", (PyCFunction)P_EntryInfo_is_flag, METH_NOARGS,
	 "true if data is a flag"},
	{"is_numeric", (PyCFunction)P_EntryInfo_is_numeric, METH_NOARGS,
	 "true if data is numeric"},
	{"is_string", (PyCFunction)P_EntryInfo_is_string, METH_NOARGS,
	 "true if data is a string"},
	{"is_time", (PyCFunction)P_EntryInfo_is_time, METH_NOARGS,
	 "true if data is a time"},
	{"parentid", (PyCFunction)P_EntryInfo_parentid, METH_NOARGS,
	 "return instrument status key (shortid + type)"},
	{"is_alarm", (PyCFunction)P_EntryInfo_is_alarm, METH_NOARGS,
	 "true if data is a source of alarm"},
	{"is_warning", (PyCFunction)P_EntryInfo_is_warning, METH_NOARGS,
	 "true if data is a source of warning"},
	{"is_fault", (PyCFunction)P_EntryInfo_is_fault, METH_NOARGS,
	 "true if data is causes a fault"},
	{NULL, NULL, 0, NULL}	/* Sentinel */
};

PyTypeObject P_EntryInfoType = {
	PyObject_HEAD_INIT(NULL)
	0,                                   /* ob_size */
	"tscstatus.entryinfo",               /* tp_name */
	sizeof(P_EntryInfo),                 /* tp_basicsize */
	0,                                   /* tp_itemsize */
	(destructor)P_EntryInfo_dealloc,     /* tp_dealloc */
	0,                                   /* tp_print */
	0,                                   /* tp_getattr */
	0,                                   /* tp_setattr */
	0,                                   /* tp_compare */
	0,                                   /* tp_repr */
	0,                                   /* tp_as_number */
	0,                                   /* tp_as_sequence */
	0,                                   /* tp_as_mapping */
	0,                                   /* tp_hash */
	0,                                   /* tp_call */
	0,                                   /* tp_str */
	0,                                   /* tp_getattro */
	0,                                   /* tp_setattro */
	0,                                   /* tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,
	                                     /* tp_flags */
	"Instrument Status Entry Info",      /* tp_doc */
	0,                                   /* tp_traverse */
	0,                                   /* tp_clear */
	0,                                   /* tp_richcompare */
	0,                                   /* tp_weaklistoffset */
	0,                                   /* tp_iter */
	0,                                   /* tp_iternext */
	P_EntryInfo_methods,                 /* tp_methods */
	P_EntryInfo_members,                 /* tp_members */
	0,                                   /* tp_getset */
	0,                                   /* tp_base */
	0,                                   /* tp_dict */
	0,                                   /* tp_descr_get */
	0,                                   /* tp_descr_set */
	0,                                   /* tp_dictoffset */
	(initproc)P_EntryInfo_init,          /* tp_init */
	0,                                   /* tp_alloc */
	(newfunc)P_EntryInfo_new,            /* tp_new */
	0,                                   /* tp_free */
	0,                                   /* tp_is_gc */
	0,                                   /* tp_bases */
	0,                                   /* tp_mro */
	0,                                   /* tp_cache */
	0,                                   /* tp_subclasses */
	0,                                   /* tp_weaklist */
	0,                                   /* tp_del */
};

