#
# a python module to handle status data documentations from Mitsubishi
#
# Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>
#
# Permission is granted for use, copying, modification, distribution,
# and distribution of modified versions of this work under the terms of
# GPL version 2 or later.
#
# vim: set ts=2 sw=2 ai noet:
#

import csv
import re
import string
import os

# shortid:{'name':part of C++ class name, type:total bytes}
# source: test5.pl by Morino
name_and_total = {
	# Could not find a point why we use special short ID for this.
	# Seems to be better to use PMA_PMFXS, PMA_PMFXS2, and PMA_PMFXS3.
	#'0858/00859/00860':{'name': 'PMFXS',               'E':4   },
	'00A1':{'name':'TSC',             'S':150, 'L':260, 'E':1557},
	'00B1':{'name':'MLP1',            'S':118, 'L':134, 'E':159 },
	'0001':{'name':'DRDR',            'S':18,  'L':84,  'E':49  },
	'0002':{'name':'MTDR',            'S':60,  'L':264, 'E':240 },
	'0004':{'name':'FRAD',            'S':13,  'L':108, 'E':42  },
	'0006':{'name':'AG',                       'L':86,  'E':120 },
	'0007':{'name':'SV',                       'L':64,  'E':162 },
	'0030':{'name':'THRM',                     'L':222, 'E':78  },
	'000D':{'name':'FPCI',                     'L':120, 'E':30  },
	'0027':{'name':'BLCU',                     'L':72,  'E':54  },
	'0008':{'name':'AO_OBE',                   'L':26,  'E':10  },
	#'0051':{'name':'AO_OBE2',                 'L':26,  'E':10  },
	'0051':{'name':'OBE',                      'L':26,  'E':10  },
	# could not find a CSV file for 0051 nor in S0501902A00
	'002E':{'name':'CLOCK',                    'L':10,  'E':1   },
	# devide into L2 and L3
	# http://siroan.naoj.org/Tsc/FY2003_TSC_status.pdf
	'00B2-1':{'name':'MLP2_1',                 'L':1160,'E':1100},	# L2
	'00B2-2':{'name':'MLP2_2',                 'L':1100,        },	# L3
	'0009':{'name':'SH',                       'L':7,   'E':100 },
	'0003':{'name':'SMCU',                     'L':198, 'E':96  },
	'00B3':{'name':'MLP3',                     'L':192, 'E':74  },
	'0024':{'name':'CVCU',                              'E':96  },
	'0025':{'name':'TMCU',                              'E':36  },
	'002A':{'name':'DOME_TEMP',                'L':480, 'E':160 },
	'002B':{'name':'DOME_CT2',                 'L':192, 'E':64  },
	'002C':{'name':'TLSCP_TEMP',               'L':540, 'E':180 },
	'002D':{'name':'TLSCP_CT2',                'L':132, 'E':44  },
	'0029':{'name':'HSBC',                     'L':18,  'E':6   },
	'000C':{'name':'WMON',                     'L':48,          },
	'000E':{'name':'CAL',                      'L':60,  'E':76  },
	'000A':{'name':'MIRROR_HT_EXH',                     'E':11  },
	'0031':{'name':'HT_EXH',                   'L':72,  'E':10  },
	'000B':{'name':'MCP1',                              'E':50  },
	'0010':{'name':'MCP2',                     'L':111, 'E':33  },
	'0858':{'name':'PMA_PMFXS1',                        'E':2   },
	'0859':{'name':'PMA_PMFXS2',                        'E':2   },
	'0860':{'name':'PMA_PMFXS3',                        'E':2   },
	'0032':{'name':'BOLT',                     'L':120, 'E':33  },
	'0033':{'name':'SPU4',                              'E':53  },
	'0034':{'name':'SPU5',                              'E':23  },
	'0035':{'name':'SPU6',                              'E':22  },
	'0028':{'name':'TTCU',            'S':60,  'L':24,  'E':53  },
	'0036':{'name':'FRAD_PF',         'S':13,  'L':90,  'E':40  },
	'0037':{'name':'ASCU_PF',                  'L':156, 'E':31  },
	'0061':{'name':'DOME_FLAT',                'L':192, 'E':27  },
	'0040':{'name':'SH_TEST',                           'E':1000},
	'0038':{'name':'CRYOGEN_HEAT_EXH',         'L':36,  'E':9   },
	#'0039':{'name':'Redundant Exh',           'L':78,  'E':25  },
	'0039':{'name':'PUMP_D',                   'L':78,  'E':25  },
	'003A':{'name':'CIAX',                     'L':132, 'E':68  },
	'003B':{'name':'OBCP',                     'L':236, 'E':192 },
	'003C':{'name':'HYDRST EXH',                        'E':35  },
	# total bytes from FY2003_TSC_Status.pdf
	# updated with S0501902A00
}

# a status field
class TelStatusEntry:
	# name (string) : col[1] (string)
	# bits (int) : col[2] (int)
	# isalarm (flag) : col[6] (bool)
	# iswarning (flag) : col[7] (bool)
	# isfault (flag) : col[8] (bool)
	# format (string) : col[9] (string)
	# id (string) : col[12] (string)
	# type (string) : col[13] (string)
	# byteoffset (int) : col[14] (int)
	# bitoffset (int): col[15] (string)
	# descriptions (dictionary): {'ja':col[5] (string)}
	# basename (string): base name of csv file (without .csv)
	def __init__(self, name, bits, isalarm, iswarning, isfault, format, id, type, byteoffset, bitoffset, basename, descriptions):
		self.name = name	# name as string: e.g. AG Exposure Time
		self.bits = bits	# number of bits as integer
		self.isalarm = isalarm	# needs alarm dialog
		self.iswarning = iswarning	# needs warning dialog
		self.isfault = isfault	# causes fault
		self.format = format	# format as string: e.g. BINARY
		self.id = id	# id as string: e.g. 0006001
		self.type = type	# type as string: E, L, or S
		self.byteoffset = byteoffset	# byte offset as integer
		self.basename = basename	# name of source csv file
		if self.basename.find('MLP2_1') >= 0:
			self.shortid = '00B2-1'
		elif self.basename.find('MLP2_2') >= 0:
			self.shortid = '00B2-2'
		else:
			self.shortid = self.id[0:4]
		self.descriptions = descriptions	# descriptions

		# bitoffset might have unit
		self.unit = '1'
		if self.format == 'BINARY':
			self.bitoffset = 0
			self.unit = bitoffset
		elif self.format == 'TIME':
			self.bitoffset = 0
		elif self.format == 'REAL':
			self.bitoffset = 0
		elif self.format == 'BCD':
			self.bitoffset = 0 
			# might be a problem on "MA Current El Angle" in TSCMON.xls
		else:
			self.bitoffset = int(bitoffset)	# bit offset as integer

		# check types
		types = {
			'name':str,
			'bits':int,
			'isalarm':bool,
			'iswarning':bool,
			'isfault':bool,
			'format':str,
			'id':str,
			'type':str,
			'byteoffset':int,
			'bitoffset':int,
			'descriptions':dict,
			'unit':str,
		}
		for attr in types.keys():
			if eval( 'self.%s.__class__' % attr ) != types[attr]:
				raise TypeError, '%s (id:%s): %s is not a %s but a %s' % (self.name, self.id, attr, types[attr], eval( 'self.%s.__class__' % attr ))

	# number of bytes occupied with this entry
	def bytes(self):
		if self.bits >= 8:
			if self.bits % 8 != 0:
				raise ValueError, '%s (id:%s): bitfield of more than 1 byte' % (self.name, self.id)
			r = self.bits / 8
		else:
			r = 1 # bit field
		return r
	
	# bit field?
	def is_bitfield(self):
		return (self.bits < 8)

	# identity
	def same(self, other):
		if self.__class__ != other.__class__:
			return False
		if self.name != other.name:
			return False
		if self.bits != other.bits:
			return False
		if self.format != other.format:
			return False
		if self.id != other.id:
			return False
		return True
