/* tsc-message.h: handling of messages between TSC and other hosts */
/*

See JNLT Requirement Specification for F-M Interface
  http://siroan.naoj.org/Tsc/Fujitsu-Mitsubishi-Interface-Specifications.pdf
for a detailed description.

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

#ifndef TSC_MESSAGE_H
#define TSC_MESSAGE_H
#include <stdio.h>
#include <melco/status-types.h>
#include <melco/tree-hash.h>
#include <melco/status-hash.h>

/* return code of some of the functions */
typedef enum {
	tsc_success = 0,
	tsc_error_stream,	/* error in fread() etc */
	tsc_error_malloc,	/* NULL from malloc() or realloc() */
	tsc_error_malformed,	/* error in parsing the input */
	tsc_error_nostream,	/* EOF or EBADF */
} tsc_error_t;

/* message header */
typedef union {
	char buf[33];
	struct
		{
			char type[2];	/* message type */
			char dst[4];	/* destination host */
			char src[4];	/* source host */
			char pid[4];	/* ID of the control process */
			ASC15_TIME txtime;	/* transmission time */
			char length[4];	/* length of the command in bytes */
		} s;
} message_header_s;

/* copies message type into a 3-byte buffer, returns char* to buffer */
char* message_header_type(const message_header_s* ptr, char *buf_three_bytes);
/* copies destination host into a 5-byte buffer, returns char* to buffer */
char* message_header_dst(const message_header_s* ptr, char *buf_five_bytes);
/* copies source host into a 5-byte buffer, returns char* to buffer */
char* message_header_src(const message_header_s* ptr, char *buf_five_bytes);
/* returns ID of the control process */
int message_header_pid(const message_header_s* ptr);
/* returns transmisision time */
const ASC15_TIME* message_header_txtime(const message_header_s* ptr);
struct timeval* message_header_txtime_tv(const message_header_s* ptr, struct timeval* dst);
/* returns length of the (remaining) message according to the header */
int message_header_length(const message_header_s* ptr);
/* returns 1 if the header is valid (have regitimate values( */
int message_header_valid(const message_header_s* ptr);

/* sets message header */
const char* message_header_set_type(message_header_s* ptr, const char* type);
const char* message_header_set_dst(message_header_s* ptr, const char* dst);
const char* message_header_set_src(message_header_s* ptr, const char* src);
int message_header_set_pid(message_header_s* ptr, const int pid);
const struct timeval* message_header_set_txtime(message_header_s* ptr, const struct timeval *txtime);
int message_header_set_length(message_header_s* ptr, const int length);

/* copies status message type (S, E, L1-L7) into a 3-byte buffer */
char* status_content_type_set(const char* type, char *buf_three_bytes);
char *message_header_status_content_type(const message_header_s* ptr, char *buf_three_bytes);

/* instrument status header */
typedef union
{
	char buf[19];
	struct
		{
			const char shortid[4];	/* instrument ID */
			const ASC15_TIME rxtime;	/* reception time */
		} s;
} instrument_status_header_s;

/* instrument ID (shortid) */
unsigned int instrument_status_header_shortid(const instrument_status_header_s* ptr);
/* reception time */
const ASC15_TIME* instrument_status_header_rxtime(const instrument_status_header_s* ptr);

/* Monitor Data Message */
#define MONITOR_DATA_MESSAGE_TYPE "MO"

typedef struct
{
	message_header_s* d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	int length;	/* real content length sometimes is not same as the header */
	char type[3];	/* S, E, L1 ... L7, NUL terminated */
	int typelen;	/* 1 or 2 */
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
} MonitorDataMessage;

/* Files from SOSS has fixed size for each monitor data message */
#define FUJITSU_TSCL_BLOCK_SIZE 1292
#define FUJITSU_TSCS_BLOCK_SIZE 1292
#define FUJITSU_TSCV_BLOCK_SIZE 8192
/* SVR4 shared memories (by Morino) on TWS4 */
#define TWS4_TSCL_BLOCK_SIZE 1292
#define TWS4_TSCS_BLOCK_SIZE 1292
#define TWS4_TSCV_BLOCK_SIZE 12931

/* allocate or initialize a MonitorDataMessage */
MonitorDataMessage *MonitorDataMessage_alloc(void);
void MonitorDataMessage_init(MonitorDataMessage *self);

/* free everything that is no longer needed */
MonitorDataMessage *MonitorDataMessage_free(MonitorDataMessage *self);

/* free everything if no longer needed */
/* This should be called after fread_MonitorDataMessage() and
   MonitorDataMessage_scan_to_*() are called to avoid memory leak */
MonitorDataMessage *MonitorDataMessage_free_if_noref(MonitorDataMessage *self);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
MonitorDataMessage *MonitorDataMessage_incref(MonitorDataMessage *self);
MonitorDataMessage *MonitorDataMessage_decref(MonitorDataMessage *self);

/* reads a monitor data message from a stream into newly malloc'ed memory */
/* if blocksize == 1, one Message is assumed to have blocksize according to
   MonitorDataContent.type and FUJITSU_TSCL/S/V_BLOCK_SIZE */
/* if blocksize > 1, one Message is assumed to have blocksize with padded bytes
   and the stream is seaked after reading the content */
/* if blocksize < 0, the length in the message_header_s is ignored and
     (-blocksize) - length of the message header
   is used as the length instead */
typedef enum
{
	block_exact,	/* each monitor data messsage padded to BLOCKSIZE */
	/* BLOCKSIZE is ignored for the following option */
	block_none,	/* no padding in the stream */
	block_fujitsu,	/* stream is from SOSS */
	block_none_or_fujitsu,	/* first try as no padding then in SOSS way */
	block_tws4,	/*  stream is from the SRV4 shared memory on TWS4 */
} message_blocking_t;
tsc_error_t fread_MonitorDataMessage_header(MonitorDataMessage* dst, FILE* stream, int rewind_after_try);	/* reads and checks only the message header */
tsc_error_t fread_MonitorDataMessage(MonitorDataMessage* dst, FILE* stream, message_blocking_t blocking, const long blocksize);
tsc_error_t read_MonitorDataMessage(MonitorDataMessage* dst, int fd);
/* reads a monitor data message packet including the header and sets the DST */
/* src_free: call back to free src, free(2) or NULL */
tsc_error_t strto_MonitorDataMessage(MonitorDataMessage* dst, char *src, void (*src_free)(void* ptr));

/* Instrument Status Data */
typedef struct
{
	instrument_status_header_s *d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	unsigned int shortid;
	char type[3];	/* S, E, L1 ... L7, NUL terminated */
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
	MonitorDataMessage *parent;
	int length;	/* length of the status data including header */
} InstrumentStatus;

/* total bytes including header and data */
int instrument_status_total_size(const char *type, const unsigned int shortid);

/* allocate or initialize a InstrumentStatus */
InstrumentStatus *InstrumentStatus_alloc(MonitorDataMessage *parent);
void InstrumentStatus_init(InstrumentStatus *self, MonitorDataMessage *parent);

/* sets members according to the parent and d */
void InstrumentStatus_set(InstrumentStatus *self, instrument_status_header_s* d, void (*d_free)(void *ptr), struct melco_tree_s* info);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
InstrumentStatus *InstrumentStatus_incref(InstrumentStatus *self);
InstrumentStatus *InstrumentStatus_decref(InstrumentStatus *self);

/* free everything that is no longer needed */
InstrumentStatus *InstrumentStatus_free(InstrumentStatus *self);

/* Instrument Status Entry */
typedef struct
{
	char *d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	const struct melco_status_s *info;
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
	InstrumentStatus *parent;
} StatusEntry;

/* look up a status entry */
StatusEntry* InstrumentStatus_get_status_entry(InstrumentStatus *self, const char *keyword);

/* allocate or initialize a StatusEntry */
StatusEntry *StatusEntry_alloc(InstrumentStatus *parent);
void StatusEntry_init(StatusEntry *self, InstrumentStatus *parent);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
StatusEntry *StatusEntry_incref(StatusEntry *self);
StatusEntry *StatusEntry_decref(StatusEntry *self);

/* free everything that is no longer needed */
StatusEntry *StatusEntry_free(StatusEntry *self);

/* Instrument Status Data Registry */
typedef struct
{
	InstrumentStatus *data[MELCO_TREE_LENGTH];
} StatusBank;

/* initialize */
StatusBank* StatusBank_init(StatusBank *self);
/* register an InstrumentStatus */
InstrumentStatus *StatusBank_register_inst(StatusBank *self, InstrumentStatus *status);
/* get an InstrumentStatus increasing reference count of InstrumentStatus */
InstrumentStatus *StatusBank_get_inst(StatusBank *self, const char *type, const unsigned int shortid);
StatusEntry* StatusBank_get_status_entry(StatusBank *self, const char *keyword);
/* remove an InstrumentStatus from the registry */
InstrumentStatus *StatusBank_discard_inst(StatusBank *self, const char *type, const unsigned int shortid);
/* remove all InstrumentStatus from the registry */
int StatusBank_discard_all_inst(StatusBank *self);

/* key for the status tree */
/* length of the buffer has to be >= MELCO_TREE_MAX_WORD_LENGTH + 1*/
char *treekey(char *seven_byte_buf, const char *type, const unsigned int shortid);
/* type and shortid do not have to be terminated by NUL */
char *treekey_str(char *seven_byte_buf, const char *type, const char *shortid);

/* scanner */
/* callback function:
   arguments:
	 - PARENT: monitor data message
	 - ARG: directly passed through from scanner
	 - INFO: info of coming instrument status
	 - INST: pointer to the data
   return value: 1:normal 0:nothihng to register -1:error
	*/
typedef int (*MonitorDataMessage_scanner_callback_t)(MonitorDataMessage* parent, void* arg, struct melco_tree_s* info, instrument_status_header_s* inst);
/* returns total number of status or -1 if error */
typedef enum {
	scan_strict,	/* belive preregistered lengths of isntrument status */
	scan_all,	/* scan every bytes */
	scan_guess_and_all,	/* first preregistered lengths than every bytes */
} scan_option_t;
int MonitorDataMessage_scanner(MonitorDataMessage* self, void* arg, scan_option_t opt, MonitorDataMessage_scanner_callback_t callback);

/* scan through the message and register InstrumentStatus's to the tree */
/* returns -1 for an error, number of statuses othewise */
int MonitorDataMessage_scan(MonitorDataMessage* self, StatusBank* dst, scan_option_t opt);

/* Instrument Status Data */
char* InstrumentStatus_treekey(const InstrumentStatus* ptr, char *seven_byte_buf);
/* to see if self has the same type and shortid */
int InstrumentStatus_is(const InstrumentStatus* self, const char *type, const unsigned int shortid);

/* Status Entry */
/* get values, entry can be NULL, returns NULL if error */
double* StatusEntry_to_d(const StatusEntry* entry, double* dst);
long long* StatusEntry_to_i(const StatusEntry* entry, long long* dst);
char* StatusEntry_to_s(const StatusEntry* entry, char* dst);
int StatusEntry_to_s_length(const StatusEntry* entry);
struct timeval* StatusEntry_to_tv(const StatusEntry* entry, struct timeval* dst);
time_t* StatusEntry_to_t(const StatusEntry* entry, time_t* dst);
char* StatusEntry_to_hex(const StatusEntry* entry, char* dst);
int StatusEntry_to_hex_length(const StatusEntry* entry);
/* reception time from the instrument header */
struct timeval* StatusEntry_rxtime_tv(const StatusEntry* entry, struct timeval* dst);
time_t* StatusEntry_rxtime(const StatusEntry* entry, time_t* dst);
double* StatusEntry_rxtime_d(const StatusEntry* entry, double* dst);
struct timeval* InstrumentStatus_rxtime_tv(const InstrumentStatus* inst, struct timeval* dst);
time_t* InstrumentStatus_rxtime(const InstrumentStatus* inst, time_t* dst);
double* InstrumentStatus_rxtime_d(const InstrumentStatus* inst, double* dst);

/* Prior Control Message and Command Demand Message */
#define PRIORITY_MESSAGE_TYPE "CC"
#define COMMAND_DEMAND_MESSAGE_TYPE "CD"

#define MAX_SEQNUM 0xFFFFFF

typedef struct
{
	message_header_s* d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	long seqnum;	/* command request number */
	long id;	/* command id */
	char *parameters;	/* pointer to the parameters */
	void (*par_free)(void* ptr);	/* call back for free(2) */
	int parlen;	/* length of the parameters in bytes */
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
} CommandMessage;

#define COMMANDMESSAGE_SEQ_SIZE 6
#define COMMANDMESSAGE_ID_SIZE 6

/* allocate or initialize a CommandMessage */
CommandMessage *CommandMessage_alloc(void);
void CommandMessage_init(CommandMessage *self);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
CommandMessage *CommandMessage_incref(CommandMessage *self);
CommandMessage *CommandMessage_decref(CommandMessage *self);

/* free everything that is no longer needed */
CommandMessage *CommandMessage_free(CommandMessage *self);

/* reads a command message packet including the header and sets the DST */
/* src_free: call back to free src, free(2) or NULL */
tsc_error_t strto_CommandMessage(CommandMessage* dst, char *src, void (*src_free)(void* ptr));
tsc_error_t read_CommandMessage(CommandMessage* dst, int fd);

/* sets up a packet from parameters */
tsc_error_t CommandMessage_set(CommandMessage *self,
	const char *type,	/* "CC" or "CD" */
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	const char *parameters,	/* parameters */
	int parlen	/* length of the parameters in bytes */
);
tsc_error_t CommandMessage_set_seqnum(CommandMessage *self, long seqnum);
/* the packet string and its length */
char* CommandMessage_tostr(CommandMessage *self);
size_t CommandMessage_strlen(CommandMessage *self);

/* iterator through arguments in paramters */
/* supply *startptr = NULL to find frist argument, *startptr = *endptr afterwards */
/* startptr is set to point the first byte of next argument or NULL after last argument */
/* length is set as the length of the argument (ignoring %s) */
/* returns new *startptr */
/* NOTE: this may return invalid results when arguments include blank spaces */
const char* CommandMessage_arg(CommandMessage *self, const char **startptr, size_t *length);

/* Reception Response Message */
#define RECEPTION_RESPONSE_MESSAGE_TYPE "CA"

typedef enum {
	response_ok,	/* OK */
	response_no_priority,	/* NG because no priority in Rx machine */
	response_locked,	/* NG because the machine is locked */
	response_ng,	/* NG because of other reason */
	response_complete,	/* completed */
	response_cancel,	/* canceled */
	response_timeout,	/* time out */
	response_fail,	/* interlocked */
	response_err_format,	/* format error */
	response_err_comm,	/* communication error */
	response_nil
} response_message_t;

typedef struct
{
	message_header_s* d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	long seqnum;	/* command request number */
	long id;	/* command id */
	long rxnum;	/* rx sequence number */
	response_message_t result;	/* result of acceptance test */
	time_t endtime;	/* estimated end time */
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
} ReceptionResponseMessage;

#define COMMANDMESSAGE_RXSEQ_SIZE 6
#define COMMANDMESSAGE_RESULT_SIZE 2	/* OK or NG */
#define COMMANDMESSAGE_INFO_SIZE 6	/* hhmmss */

/* allocate or initialize a ReceptionResponseMessage */
ReceptionResponseMessage *ReceptionResponseMessage_alloc(void);
void ReceptionResponseMessage_init(ReceptionResponseMessage *self);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
ReceptionResponseMessage *ReceptionResponseMessage_incref(ReceptionResponseMessage *self);
ReceptionResponseMessage *ReceptionResponseMessage_decref(ReceptionResponseMessage *self);

/* free everything that is no longer needed */
ReceptionResponseMessage *ReceptionResponseMessage_free(ReceptionResponseMessage *self);

/* reads a reception response message packet including the header and sets the DST */
/* src_free: call back to free src, free(2) or NULL */
tsc_error_t strto_ReceptionResponseMessage(ReceptionResponseMessage* dst, char *src, void (*src_free)(void* ptr));
tsc_error_t read_ReceptionResponseMessage(ReceptionResponseMessage* dst, int fd);

/* sets up a packet from parameters */
tsc_error_t ReceptionResponseMessage_set(ReceptionResponseMessage *self,
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	long rxnum,	/* rx sequence number */
	response_message_t result,	/* result of acceptance test */
	time_t endtime	/* estimated end time, 0 if not applicable */
);

/* the packet string and its length */
char* ReceptionResponseMessage_tostr(ReceptionResponseMessage *self);
size_t ReceptionResponseMessage_strlen(ReceptionResponseMessage *self);

/* error codes on Reception Message */
const char *ReceptionResponseMessage_strerr(ReceptionResponseMessage *self);

/* Completion Response Message */
#define COMPLETION_RESPONSE_MESSAGE_TYPE "CE"

typedef struct
{
	message_header_s* d;
	void (*d_free)(void* ptr);	/* call back for free(2) */
	long seqnum;	/* command request number */
	long id;	/* command id */
	long rxnum;	/* rx sequence number */
	response_message_t result;	/* result of execution */
	long errorcode;	/* 0 when success, otherwise as defined in Table 2.5-1 in "F-M interface" documentation */
	char *data;	/* status data, command, or communication data as defined in Table 2.5-1 in "F-M interface" documentation, excluding error code */
	int datalen;	/* length of data */
	void (*data_free)(void* ptr);	/* call back for free(2) */
	int refcount;	/* reference count */
	void (*self_free)(void* ptr);	/* call back for free(2) */
} CompletionResponseMessage;

#define COMMANDMESSAGE_COMPLETION_RESULT_SIZE 10	/* COMPLETE%%, etc. */
#define COMMANDMESSAGE_COMPLETION_ERRORCODE_SIZE 6

#define COMMANDMESSAGE_COMPLETION_TIMEOUT 5	/* seconds after endtime before timeout */

/* allocate or initialize a CompletionResponseMessage */
CompletionResponseMessage *CompletionResponseMessage_alloc(void);
void CompletionResponseMessage_init(CompletionResponseMessage *self);

/* reference count - free's memories that is no longer needed */
/* returns self or NULL when free'ed */
CompletionResponseMessage *CompletionResponseMessage_incref(CompletionResponseMessage *self);
CompletionResponseMessage *CompletionResponseMessage_decref(CompletionResponseMessage *self);

/* free everything that is no longer needed */
CompletionResponseMessage *CompletionResponseMessage_free(CompletionResponseMessage *self);

/* reads a completion response message packet including the header and sets the DST */
/* src_free: call back to free src, free(2) or NULL */
tsc_error_t strto_CompletionResponseMessage(CompletionResponseMessage* dst, char *src, void (*src_free)(void* ptr));
tsc_error_t read_CompletionResponseMessage(CompletionResponseMessage* dst, int fd);

/* sets up a packet from parameters */
tsc_error_t CompletionResponseMessage_set(CompletionResponseMessage *self,
	const char *dst,	/* "TSC%" etc */
	const char *src,	/* "TWS4" etc */
	int pid,	/* 0 etc */
	const struct timeval *txtime,	/* time stamp */
	long seqnum,	/* command request number */
	long id,	/* command id */
	long rxnum,	/* rx sequence number */
	response_message_t result,	/* result of execution */
	long errorcode,	/* error code */
	char *data,	/* status data, command, or communication data as defined in Table 2.5-1 in "F-M interface" documentation, excluding error code */
	int datalen	/* length of data */
);

/* the packet string and its length */
char* CompletionResponseMessage_tostr(CompletionResponseMessage *self);
size_t CompletionResponseMessage_strlen(CompletionResponseMessage *self);

/* iterator through arguments in data */
/* supply *startptr = NULL to find frist argument, *startptr = *endptr afterwards */
/* startptr is set to point the first byte of next argument or NULL after last argument */
/* length is set as the length of the argument (ignoring %s) */
/* returns new *startptr */
/* NOTE: this may return invalid results when arguments include blank spaces */
const char* CompletionResponseMessage_arg(CompletionResponseMessage *self, const char **startptr, size_t *length);

/* generic message receiver - returns a malloced memory filled with message read from fd */
/* error codes: NULL:malloc 1:stream 2:malformed header 3:EBADF */
char* read_Message(int fd);

/* error codes on Completion Message */
const char *CompletionResponseMessage_strerr(CompletionResponseMessage *self);
const char *completion_error_message(response_message_t response, long errorcode);	/* supply error code 0 for reception error message */

/* message types (partially) supported by this library: char** */
#define SUPPORTED_MESSAGE_TYPES {\
	MONITOR_DATA_MESSAGE_TYPE,\
	PRIORITY_MESSAGE_TYPE,\
	COMMAND_DEMAND_MESSAGE_TYPE,\
	RECEPTION_RESPONSE_MESSAGE_TYPE,\
	COMPLETION_RESPONSE_MESSAGE_TYPE,\
	NULL,\
}

#endif	/* TSC_MESSAGE_H */
