/* status-types.h: defnitions of type used in TSC status handling */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/
/*

  reader functions (... _to_d, _to_i, _to_s, and _to_t) set the value into
	DST, with the following format:
	  _to_d: double*
		_to_i: long long*
		_to_s: char* with (int) _to_s_length returning length required for
		                                     buffer including '\0'
		_to_t: struct timeval*
  with appropriate conversions.

  to_hex: char* of hexiadecimal presentation like XX-XX-...
	to_hex_length: int of length required for buffer inlcuding '\0'

	BCD6_RA_to_d sets RA in radians.

*/
#ifndef	STATUS_TYPES_H
#define	STATUS_TYPES_H

#include <time.h>
#include <sys/time.h>

#ifndef HAVE_TIMEGM
time_t timegm(struct tm *tm);
#endif

/* valid range of time stamp of status packets - needed in scanning status */
#define MIN_MESSAGE_TIME 788918400	/* 1995-01-01 00:00:00 UTC */
#define MAX_MESSAGE_TIME 2147483647	/* 2038-01-19 03:14:07 UTC */

void* to_hex(const void *addr, const unsigned char mask, const size_t length, void *dst);
int to_hex_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

void* bit_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* bit_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int bit_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* str_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int str_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

/* 15-byte ASCII expression of UT */
typedef union
{
	char buf[15];
	struct
		{
			const char year[4];
			const char month[2];
			const char day[2];
			const char hour[2];
			const char min[2];
			const char sec[3];
		} sep;
} ASC15_TIME;

void* ASC15_TIME_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t + sub-seconds */
void* ASC15_TIME_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* ASC15_TIME_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* 16 bytes, exact copy */
int ASC15_TIME_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* 16 bytes, exact copy */
void* ASC15_TIME_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

typedef union
{
	char buf[7];
	struct
		{
			const char hour[2];
			const char min[2];
			const char sec[3];
		} sep;
} ASC7_TIME;

void* ASC7_TIME_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t + sub-seconds */
void* ASC7_TIME_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* ASC7_TIME_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* 8 bytes, exact copy */
int ASC7_TIME_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* 8 bytes, exact copy */
void* ASC7_TIME_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

typedef union
{
	char buf[1];
} BCD1;

void* BCD1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BCD1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[2];
} BCD2;

void* BCD2_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD2_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD2_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BCD2_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[6];
} BCD6;

void* BCD6_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD6_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCD6_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BCD6_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[6];
} BCD6_DEC;

void* BCD6_DEC_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* radian */
void* BCD6_DEC_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* +DD:MM:SS.SSSS */
int BCD6_DEC_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[6];
} BCD6_Epoch;

void* BCD6_Epoch_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* year */
void* BCD6_Epoch_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BCD6_Epoch_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[6];
} BCD6_RA;

void* BCD6_RA_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* radian */
void* BCD6_RA_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* HH:MM:SS.SSSSSS */
int BCD6_RA_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[1];
} BINARY1_0p01;

void* BINARY1_0p01_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY1_0p01_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY1_0p01_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY1_0p01_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[1];
} BINARY1_1;

void* BINARY1_1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY1_1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY1_1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY1_1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[2];
} BINARY2_0p01;

void* BINARY2_0p01_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY2_0p01_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY2_0p01_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY2_0p01_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[2];
} BINARY2_0p1;

void* BINARY2_0p1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY2_0p1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY2_0p1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY2_0p1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[2];
} BINARY2_1;

void* BINARY2_1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY2_1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY2_1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY2_1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p000000000001;

void* BINARY4_0p000000000001_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p000000000001_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p000000000001_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p000000000001_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p000001;

void* BINARY4_0p000001_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p000001_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p000001_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p000001_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p00001;

void* BINARY4_0p00001_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p00001_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p00001_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p00001_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p0001;

void* BINARY4_0p0001_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p0001_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p0001_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p0001_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p001;

void* BINARY4_0p001_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p001_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p001_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p001_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p01;

void* BINARY4_0p01_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p01_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p01_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p01_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_0p1;

void* BINARY4_0p1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_0p1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_0p1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_0p1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_1;

void* BINARY4_1_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_1_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_1_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_1_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[4];
} BINARY4_100;

void* BINARY4_100_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BINARY4_100_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* before scaling */
void* BINARY4_100_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int BINARY4_100_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[8];
} REAL8;

void* REAL8_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* REAL8_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* 9 bytes, exact copy */
int REAL8_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);

typedef union
{
	char buf[3];
	struct	/* UT */
		{
			unsigned char hh;
			unsigned char mm;
			unsigned char ss;
		} sep;
} TIME3;

void* TIME3_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME3_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME3_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* HH:MM:SS in UT */
int TIME3_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* TIME3_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

typedef union
{
	char buf[4];
	struct	/* UT */
		{
			unsigned char day;
			unsigned char hh;
			unsigned char mm;
			unsigned char ss;
		} sep;
} TIME4;

void* TIME4_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t + sub-seconds */
void* TIME4_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME4_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* DD-HH:MM:SS */
int TIME4_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* TIME4_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

typedef union
{
	char buf[6];
	struct	/* UT */
		{
			char year;		/* year - 2000 */
			unsigned char month;	
			unsigned char day;	
			unsigned char hh;
			unsigned char mm;
			unsigned char ss;
		} sep;
} TIME6;

void* TIME6_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME6_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME6_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* yyyy/dd/mm HH:MM:SS */
int TIME6_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* TIME6_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

typedef union
{
	char buf[8];
	struct {
		TIME6 time6;
		unsigned char byte6;	/* do not know what these bytes mean */
		unsigned char byte7;
	} sep;
} TIME8;

void* TIME8_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t + sub-seconds */
void* TIME8_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* time_t */
void* TIME8_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
int TIME8_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* TIME8_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst);	/* struct timeval */

#endif	/* STATUS_TYPES_H */
