/* tscl-fujitsu-offset.c: looks for TSCL entries from offsets */
/*

Copyright (C) 2007 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <melco/tsc-message.h>
#include <melco/tree-hash.h>
#include <melco/status-hash.h>
#include <melco/status-ids.h>

/* prototypes */
struct _shortid_offsets {
	struct melco_tree_s *instrument_status;
	unsigned long offset_from;	/* where the status starts */
	unsigned long offset_next;	/* next from the end of the status */
};

struct melco_status_s_list {
	struct melco_status_s *status[9]; /* max 8 entries per byte plus sentinel */
};

void usage(FILE *stream);
int build_offset_table(void);	/* returns 1 for a successful build */
unsigned int tscl_from_offset(struct melco_status_s_list *dst, unsigned long offset);

/* locations of the status - from S0501902A00 */
char *L[] = {
	"00A1L", "00B1L", "0001L", "0002L", "0004L", "0006L", "0007L", "0028L", "002EL", NULL,	/* L1 */
	"00B2L2", NULL,	/* L2 */
	"00B2L3", "0009L", "0039L", NULL, /* L3 */
	"0003L", "00B3L", "002CL", "002DL", "0038L", NULL, 	/* L4 */
	"0030L", "002AL", "002BL", "0008L", "0051L", "003AL", NULL,	/* L5 */
	"000CL", "0029L", "0031L", "0032L", "000EL", "0010L", "0028L", "0036L", "0037L", "000DL", NULL, /* L6 */
	"003BL", NULL, /* L7 */
	NULL,	/* sentinel */
};

unsigned long tscl_block_size = 1292;
unsigned long short_to_id = 0x1000;

struct _shortid_offsets instrument_offsets[sizeof(L) + 1];

int
build_offset_table(void)
{
	unsigned int s, d;
	unsigned long block, offset;

	s = d = 0;
	block = offset = 0;
	while(L[s])
		{
			if (d >= sizeof(L))
				{
					fprintf(stderr, "%s: too many instrument statuses: %d\n", __func__, d);
					return 0;
				}

			offset += sizeof(message_header_s);	/* skip message header */
			offset += 2;	/* L1, L2, ..., or L7 */

			while(L[s])
				{
					struct melco_tree_s *instrument_status;

					instrument_status = melco_tree_lookup(L[s], strlen(L[s]));
					if (!instrument_status)
						{
							fprintf(stderr, "%s: could not find instrument status: %s\n", __func__, L[s]);
							return 0;
						}

					instrument_offsets[d].instrument_status = instrument_status;
					offset += sizeof(instrument_status_header_s);	/* skip instrument header */
					instrument_offsets[d].offset_from = block + offset;
					offset += instrument_status->bytes;
					instrument_offsets[d].offset_next = block + offset;
					d++;

					s++;
				}
			if (offset > tscl_block_size)
				{
					fprintf(stderr, "%s: over filled block ending with %s\n", __func__, L[s-1]);
					return 0;
				}

			/* moving to next message packet */
			block += tscl_block_size;
			offset = 0;
			s++;
		}

	instrument_offsets[d].instrument_status = NULL;
	instrument_offsets[d].offset_from = 0;
	instrument_offsets[d].offset_next = 0;
	return 1;
}

unsigned int
tscl_from_offset(struct melco_status_s_list *dst, unsigned long offset)
{
	unsigned int i, n;
	unsigned long id_from, id_to, id;
	long offset_inside;

	/* find short id */
	i = 0;
	while(instrument_offsets[i].instrument_status)
		{
			if (instrument_offsets[i].offset_from <= offset && offset < instrument_offsets[i].offset_next) break;
			i++;
		}
	if (!instrument_offsets[i].instrument_status)
		{
			dst->status[0] = NULL;
			return 0;
		}
	id_from = melco_tree_shortid[instrument_offsets[i].instrument_status->index]*short_to_id;
	id_to = id_from + short_to_id;

	/* find status entries */
	offset_inside = offset - instrument_offsets[i].offset_from;
	for(n = 0, id = id_from; n < 8 && id < id_to; id++)
		{
			struct melco_status_s* s;

			s = melco_status_from_id(id);
			if (s && 'L' == s->type[0] && offset_inside == s->offset)
				{
					dst->status[n] = s;
					n++;
				}
		}

	dst->status[n] = NULL;
	return n;
}

void
usage(FILE *stream)
{
	fputs("usage: tscl-fujitsu-offset offset, ...\n", stream);
	fputs("\tsearches TSCL entries defined by offset in Fujitsu nomenclature\n", stream);
}

int
main(int argc, char *argv[])
{
	int i;
	int r;

	if (argc < 2)
		{
			usage(stderr);
			exit(EXIT_FAILURE);
		}

	if (!build_offset_table()) exit(EXIT_FAILURE);

	r = EXIT_SUCCESS;
	for(i = 1; i < argc; i++)
		{
			struct melco_status_s_list entries;
			unsigned long offset;
			unsigned int n, j;
			char *eptr;

			offset = strtoul(argv[i], &eptr, 10);
			if (*eptr != '\0')
				{
					fprintf(stdout, "%d\tNA     \tNA\n", 0);
					fprintf(stderr, "tscl-fujitsu-offset: unrecognized decimal offset: %s\n", argv[i]);
					if (EXIT_SUCCESS == r) usage(stderr);
					r = EXIT_FAILURE;
					continue;
				}

			n = tscl_from_offset(&entries, offset);
			if (n < 1)
				{
					fprintf(stdout, "%lu\tNA     \tNA\n", offset);
					fprintf(stderr, "tscl-fujitsu-offset: no entry found for offset: %lu\n", offset);
					r = EXIT_FAILURE;
					continue;
				}
			for(j = 0; j < n; j++)
				{
					fprintf(stdout, "%lu\t%07X\t%s\n", offset, entries.status[j]->id, entries.status[j]->name);
				}
		}

	return r;
}
