#!/usr/local/bin/python

import sys
import math
import catWalkWx

#### CONSTANTS
DEBUG = False

#### GLOBAL VARIABLES
M1CellT = []             # List of [temp. location, temperature, RH]

# eAldl : Calculates the dew point using the Alduchov Enhanced August-Roche-Magnus(dry air) water vapor
#         pressure over water for Tc > 0.00C and over ice for Tc <= 0.00C
#
#         RH = Relative Humidity Percentage
#         Tc = Temperature in Celsius
def enAldARMag(RH, Tc):

    el = 0.61094
    al = 17.625
    bl = 243.04
    ei = 0.61121
    ai = 22.587
    bi = 273.86

    RH = RH / 100.0

    # Saturated water vapor calculated over a plane of water
    if Tc > 0.00 :
        alpha = math.log(RH) + (al * Tc) / (bl + Tc)
        ans = (bl * alpha) / (al - alpha)
    # Saturated water vapor calculated over a plane of ice
    else :
        alpha = math.log(RH) + (ai * Tc) / (bi + Tc)
        ans = (bi * alpha) / (ai - alpha)
                
    return ans



def calLowerDewPtDiff(RH, tAir, sideT, botmT) :

    
    dewPt = enAldARMag(RH, tAir)

    diffSide = sideT - dewPt
    diffBotm = botmT - dewPt

    if diffSide <= diffBotm :
        ans = sideT
    else :
        ans = botmT

    if DEBUG :
        print "RH = ", RH, " tAir = ", tAir, " sideT = ", sideT, " botmT = ", botmT
        print "dewPt = ", dewPt, " diffSide = ", diffSide, " diffBotm = ", diffBotm, " ans = ", ans
        
    return ans, dewPt


# maxFloat: Finds the maximum of two floats and returns the maximum and its name.
def maxFloat(name1, flt1, name2, flt2) :

    if flt1 >= flt2 :
        maxFlt = flt1
        maxName = name1
    else :
        maxFlt = flt2
        maxName = name2
        
    return maxFlt, maxName



# minFloat: Finds the minimum of two floats and returns the minimum and its name.
def minFloat(name1, flt1, name2, flt2) :

    if flt1 >= flt2 :
        minFlt = flt1
        minName = name1
    else :
        minFlt = flt2
        minName = name2
        
    return minFlt, minName



# getHiMirTemp: Finds the warmest mirror temperature and the warmest DP
# Input: tscl -- tws4 shared memory for long variables.
# Returns: hiTpName -- three letter code for temperature location on M1
#          hiTp -- highest M1 temperature
#          hiDPTpName -- three letter code for dew point location on M1
#          hiDPTp -- highest dew point temperature
def getHiMirTempDP(tscl) :

    hiTp = -273.06
    hiTpName = "BAD"
    hiDPTp = -273.06
    hiDPTpName = "BAD"

    # M1 IR Side & Bot. Temp and humidity
    hmd = tscl.Cell_IR_HMD_Data.val()
    M1CellT = [["MIS", tscl.M1_IR_S_Temp_Data.val(), hmd]]      # M1 IR Side Temp.
    M1CellT.append(["MIB", tscl.M1_IR_B_Temp_Data.val(), hmd])  # M1 IR Bot. Temp.
    # M1 Frt. Side & Bot. Temp and humidity
    hmd = tscl.Cell_F_HMD_Data.val()    
    M1CellT.append(["MFS", tscl.M1_F_S_Temp_Data.val(), hmd])   # M1 Frt. Side Temp.
    M1CellT.append(["MFB", tscl.M1_F_B_Temp_Data.val(), hmd])   # M1 Frt. Bot. Temp.
    # M1 Rear Side & Bot. Temp and humidity
    hmd = tscl.Cell_R_HMD_Data.val()
    M1CellT.append(["MRS", tscl.M1_R_S_Temp_Data.val(), hmd])      # M1 Rear Side Temp.
    M1CellT.append(["MRB", tscl.M1_R_B_Temp_Data.val(), hmd])      # M1 Rear Bot. Temp.
    # M1 Opt Side & Bot. Temp and humidity
    hmd = tscl.Cell_Opt_HMD_Data.val()
    M1CellT.append(["MOS", tscl.M1_Opt_S_Temp_Data.val(), hmd])    # M1 Opt. Side Temp.
    M1CellT.append(["MOB", tscl.M1_Opt_B_Temp_Data.val(), hmd])    # M1 Opt. Bot. Temp.
    ### M1 Cell 8-61,8-62,8-63: set humidity to -0.1
    ###M1CellT.append(["C61", tscl.Cell_8_61_Temp_Data.val(), -1.0])   # M1 Cell 8-61 Temp.
    ###M1CellT.append(["C62", tscl.Cell_8_62_Temp_Data.val(), -1.0])   # M1 Cell 8-62 Temp. 
    ###M1CellT.append(["C63", tscl.Cell_8_63_Temp_Data.val(), -1.0])   # M1 Cell 8-63 Temp.
    ###print "Heat_Exh_Mirror_Temp_Real.val() = ", tscl.Heat_Exh_Mirror_Temp_Real.val()
    ###print "Heat_Exh_Open_Air_HMD_Mirror.val() = ",tscl.Heat_Exh_Open_Air_HMD_Mirror.val()
    ###print "Heat_Exh_Open_Air_Temp_Mirror.val() = ",tscl.Heat_Exh_Open_Air_Temp_Mirror.val()
    

    if DEBUG :
        print "M1CellT = ", M1CellT
        print "M1CellT[0][1] = ", M1CellT[0][1]

    for tmpPair in M1CellT :
        if tmpPair[1] > hiTp :
            hiTp = tmpPair[1]
            hiTpName = tmpPair[0]
        if tmpPair[2] > 0.01 :
           ###dewPt = enAldARMag(tmpPair[2], tmpPair[1])
           dewPt = -273.00
           if DEBUG :
               print "dewPt = ", dewPt, " dewPtName = ", tmpPair[0]
           if dewPt > hiDPTp :
               hiDPTp = dewPt
               hiDPTpName = tmpPair[0]


    return  M1CellT, hiDPTpName, hiDPTp   # end of getHiMirTempDP



# getHiDPExt: Calculates the dew point temp. for all the exterior (control bldg. & catwalk)
#             sensor humidities and finds the highest dew point temp.
# Input: tscl -- tws4 shared memory for long variables.
# Returns: dewPt -- highest dew point temp.
#          dewPtLoc -- three letter code of the location of the highest dew pt. temp.
def getHiDPExt(tscl) :
    
    dewPt = -273.0
    dewPtLoc = "BAD"
    
    # Get the Cat Walk Weather Information
    catWkData = catWalkWx.getWx()
    ###print "Catwalk Data = ", catWkData

    hiHmd = -1.0
    loopEnd = len(catWkData)
    for i in range(1, loopEnd):
        ###print "catWkData[",i,"] = ",catWkData[i][1]
        if catWkData[i][1] != "N/A" :
            curHmd = float(catWkData[i][1])
            if curHmd > hiHmd :
                hiHmd = curHmd
                dewPtLoc = catWkData[i][0]
                
    if hiHmd > 0.01 : 
        dewPt = enAldARMag(hiHmd, float(catWkData[0]))

    # Get the humidity and temp. from the control bldg. sensors
    CBHmd = tscl.Weather_Humidity.val()
    CBTemp = tscl.Weather_Temperature.val()
    CBDewPt = enAldARMag(CBHmd,CBTemp)

    retDP, retLoc = maxFloat(dewPtLoc, dewPt, "WxT", CBDewPt)
    
    if DEBUG :
        print dewPtLoc, ":  RH% = ", hiHmd, "  temp = ", catWkData[0],"  dewPt = ", dewPt
        print "WxT", ":  RH% = ", CBHmd, "  temp = ", CBTemp,"  dewPt = ", CBDewPt
        print "retLoc = ", retLoc, " retDP = ", retDP

    return retLoc, retDP   # end of getHiDPExt



# getHiDPExtAll: Calculates the dew point temp. for all the exterior (control bldg. & catwalk)
#             sensor humidities and finds the highest catwalk dew point temp and the Wx dew
#             point.
# Input: tscl -- tws4 shared memory for long variables.
# Returns: CBDewPt -- dew point temp. for Wx Tower
#          "WxT" -- three letter code of the location for WXT
#          dewPtCat -- highest dew point temp.
#          dewPtCatLoc -- three letter code of the location of the highest dew pt. temp.

def getHiDPExtAll(tscl) :
    
    dewPt = -273.0
    dewPtLoc = "BAD"
    
    # Get the Cat Walk Weather Information
    catWkData = catWalkWx.getWx()
    ###print "Catwalk Data = ", catWkData

    hiHmd = -1.0
    loopEnd = len(catWkData)
    for i in range(1, loopEnd):
        ###print "catWkData[",i,"] = ",catWkData[i][1]
        if catWkData[i][1] != "N/A" and catWkData[i][1] != "" :
            curHmd = float(catWkData[i][1])
            if curHmd > hiHmd :
                hiHmd = curHmd
                dewPtLoc = catWkData[i][0]
                
    if hiHmd > 0.01 : 
        dewPt = enAldARMag(hiHmd, float(catWkData[0]))

    # Get the humidity and temp. from the control bldg. sensors
    CBHmd = tscl.Weather_Humidity.val()
    CBTemp = tscl.Weather_Temperature.val()
    CBDewPt = enAldARMag(CBHmd,CBTemp)
    
    if DEBUG :
        print dewPtLoc, ":  RH% = ", hiHmd, "  temp = ", catWkData[0],"  dewPt = ", dewPt
        print "WxT", ":  RH% = ", CBHmd, "  temp = ", CBTemp,"  dewPt = ", CBDewPt

    return "WxT", CBDewPt, dewPtLoc, dewPt    # end of getHiDPExtAll



# getExtDPMT: Finds the highest dew point temperatures from internal and external sources.
#          Also, finds mirror temps. that are already below dew point or close to the
#          dew point temp.
# Input: tscl -- tws4 shared memory for long variables. 
# Returns: dewPt -- highest dew point temp.
#          dewPtLoc -- three letter code of the location of the highest dew pt. temp.
#          M1Temp -- mirror temp that's closest to the dew point or the lowest below
#                    dew point.
#          M1TempLoc -- mirror location of above
def getHiDPMT(tscl) :
    
    dewPt = -273.0
    dewPtLoc = "BAD"
    MirTemp = -273.0
    MirLoc = "BAD"
    
    # Get the Cat Walk temp. and highest internal dew point temp..
    M1CellT, mirHiDPTpName, mirHiDPTp = getHiMirTempDP(tscl)
    
    # Get the highest external dew point temp.
    extDPTmpLoc, extDPTmp = getHiDPExt(tscl)

    # Determine the highest dew point temp. between internal and external
    maxDP, maxDPLoc = maxFloat(mirHiDPTpName, mirHiDPTp, extDPTmpLoc, extDPTmp)
    if DEBUG :
        print "maxDP = ", maxDP, " maxDPLoc = ", maxDPLoc

    maxTmp = -273.0
    maxTmpLoc = "BAD"
    minTmp = 9999.0
    minTmpLoc = "BAD"
    hiLoTmp = -273
    for tmpTrip in M1CellT :
        curTemp = tmpTrip[1]
        if curTemp < minTmp :
            minTmp = curTemp
            minTmpLoc = tmpTrip[0]

    ###if maxTmp <= maxDP :
        # Calc.dew pt. is above all mirror temps. or equal to the max temp.  Return max. temp.
    ###    retTmp = maxTmp
    ###    retTmpLoc = maxTmpLoc
    ###else :
        # Calc.dew pt. is below max. temp. and either above, below or equal to min. temp.
        # Return min.temp.
        ###retTmp = minTmp
        ###retTmpLoc = minTmpLoc
        
    # Always return the minimum M1 temperature and location.  If dew point is below minimum then
    # minimum will be closest to the dew point.  If dew point is above the minimum then the
    # minimum represents the worst case condensation risk.
    retTmp = minTmp
    retTmpLoc = minTmpLoc

    return maxDPLoc, maxDP, retTmpLoc,retTmp   # end of getHiDPMT

