/* status-receiver.c : status reception server on TWS4 */
/*

Copyright (C) 2010 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <sys/time.h>
#include <math.h>
#include <sys/mman.h>

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifndef HAVE_PSELECT
#include "pselect.h"
#endif

#include <melco/tsc-message.h>
#include <melco/shm-interface.h>

/* include path to be specified in Makefile */
#include "rpc-hosts.h"
#include "tws_tsc_short.h"
#include "tws_tsc_long.h"
#include "tws_tsc_stat.h"

#include "legacy/tws_tsc_shm.h"

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

/* pipe to write[1]/read[0] monitor data message - global for each process */
int pipefd[2];
size_t packet_size;

/* put everything to be done here */
void read_and_broadcast(int input_fd, ssize_t max_size, char *type, unsigned int timeout, int v, copy_to_shm_t copier);

void usage(FILE *stream);
void register_tscl_receiver(int v);
void long_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp);
void *long_transmit_1_svc(tscl_broadcastForm *form, struct svc_req *svcdata);

void register_tscs_receiver(int v);
void short_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp);
void *short_transmit_1_svc(tscs_broadcastForm *form, struct svc_req *svcdata);

void register_tscv_receiver(int v);
void stat_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp);
bool_t *stat_transmit_1_svc(TSCV_NetForm *form, struct svc_req *svcdata);

void
usage(FILE *stream)
{
	fputs("usage: status-receiver [options]\n", stream);
	fputs("\treads status data from the RPC client (TSC) and\n", stream);
	fputs("\tcopies them to shared memories.\n", stream);
	fputs("OPTIONS:\n", stream);
	fputs("-v: increases verbosity\n", stream);
	fputs("-h: shows this\n", stream);
}

void
read_and_broadcast(int input_fd, ssize_t max_size, char *type, unsigned int timeout, int v, copy_to_shm_t copier)
{
	int n_instmsg;
	static size_t r_read;
	static char *buf;
	static MonitorDataMessage msg;
	tsc_error_t r_msg;
	int timedout;
	struct timespec timeout_ts, *timeout_ptr;
	int r_select;
	sigset_t sigmask, orig_sigmask;
	fd_set readfds;

	MonitorDataMessage_init(&msg);

	buf = malloc(max_size);
	if (!buf)
		{
			perror("malloc()");
			exit(EXIT_FAILURE);
		}

	timedout = 0;
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGALRM);
	sigprocmask(SIG_BLOCK, &sigmask, &orig_sigmask);

	while(1)
		{
			if (!timedout)
				{
					timeout_ts.tv_sec = timeout;
					timeout_ts.tv_nsec = 0;
					timeout_ptr = &timeout_ts;
				}
			else
				{
					timeout_ptr = NULL;
				}
			FD_ZERO(&readfds);
			FD_SET(input_fd, &readfds);
			r_select = pselect(input_fd + 1, &readfds, NULL, NULL, timeout_ptr, &orig_sigmask);
			if (r_select == 0)	/* timed out */
				{
					if (!timedout)
						{
							fprintf(stderr, "Status packets for %s not received for %d sec. Still Waiting.\n", type, timeout);
						}
					timedout = 1;
					continue;
				}
			if (r_select < 0)
				{
					perror("pselect()");
					exit(EXIT_FAILURE);
				}

			r_read = read(input_fd, buf, max_size);
			if (r_read < 1)
				{
					perror("read()");
					exit(EXIT_FAILURE);
				}

			if (copier) copier(buf, r_read);

			errno = 0;
			r_msg = strto_MonitorDataMessage(&msg, buf, NULL);
			if (tsc_success != r_msg)
				{
					fprintf(stderr, "%35.35s ", buf);
					switch(r_msg)
						{
							case tsc_error_stream:
							case tsc_error_malloc:
							case tsc_error_nostream:
								perror("strto_MonitorDataMessage()");
								break;
							case tsc_error_malformed:
								fputs("strto_MonitorDataMessage(): malformed header\n", stderr);
								break;
						}
					continue;
				}

			if (timedout)
				{
					fprintf(stderr, "Status packets for %s began coming as type %s\n", type, msg.type);
					timedout = 0;
				}

			n_instmsg = MonitorDataMessage_scan_to_shm(&msg, scan_all);
			if (n_instmsg < 1 && v > 0)	/* L7 does not have content when FMOS is off */
				{
					fprintf(stderr, "%35.35s got no instrument messages\n", buf);
					continue;
				}

			if (v > 0) fprintf(stderr, "%35.35s %4d bytes\n", buf, r_read);
		}

	free(buf);
	exit(EXIT_FAILURE);
}

void *
long_transmit_1_svc(tscl_broadcastForm *form, struct svc_req *svcdata)
{
	static bool_t result = TRUE;

	if (!form || !svcdata)
		{
			result = FALSE;
			return &result;
		}

	write(pipefd[1], form->contents, tscl_packet_size);
	return &result;
}

void *
short_transmit_1_svc(tscs_broadcastForm *form, struct svc_req *svcdata)
{
	static bool_t result = TRUE;

	if (!form || !svcdata)
		{
			result = FALSE;
			return &result;
		}

	write(pipefd[1], form->contents, tscs_packet_size);
	return &result;
}

bool_t *
stat_transmit_1_svc(TSCV_NetForm *form, struct svc_req *svcdata)
{
	static bool_t result = TRUE;

	if (!form || !svcdata)
		{
			result = FALSE;
			return &result;
		}

	write(pipefd[1], form->TSCV_NetForm_val, form->TSCV_NetForm_len);
	return &result;
}

/* copied from test-rpc/tws_tsc_long_svc.c */
void
long_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp)
{
	union {
		tscl_broadcastForm long_transmit_1_arg;
	} argument;
	char *result;
	xdrproc_t _xdr_argument, _xdr_result;
	char *(*local)(char *, struct svc_req *);

	switch(rqstp->rq_proc)
		{
			case NULLPROC:
				(void) svc_sendreply(transp, (xdrproc_t) xdr_void, (char *)NULL);
				return;
			case LONG_TRANSMIT:
				_xdr_argument = (xdrproc_t) xdr_tscl_broadcastForm;
				_xdr_result = (xdrproc_t) xdr_void;
				local = (char *(*)(char *, struct svc_req *)) long_transmit_1_svc;
				break;
			default:
				svcerr_noproc(transp);
				return;
		}
	memset((char *)&argument, 0, sizeof (argument));
	if (!svc_getargs(transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument))
		{
			svcerr_decode(transp);
			return;
		}
	result = (*local)((char *)&argument, rqstp);
	if (result != NULL && !svc_sendreply(transp, (xdrproc_t) _xdr_result, result))
		{
			svcerr_systemerr(transp);
		}
	if (!svc_freeargs(transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument))
		{
			fprintf(stderr, "%s", "unable to free arguments");
			exit(1);
		}
	return;
}

void
short_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp)
{
	union {
		tscl_broadcastForm short_transmit_1_arg;
	} argument;
	char *result;
	xdrproc_t _xdr_argument, _xdr_result;
	char *(*local)(char *, struct svc_req *);

	switch(rqstp->rq_proc)
		{
			case NULLPROC:
				(void) svc_sendreply(transp, (xdrproc_t) xdr_void, (char *)NULL);
				return;
			case SHORT_TRANSMIT:
				_xdr_argument = (xdrproc_t) xdr_tscl_broadcastForm;
				_xdr_result = (xdrproc_t) xdr_void;
				local = (char *(*)(char *, struct svc_req *)) short_transmit_1_svc;
				break;
			default:
				svcerr_noproc(transp);
				return;
		}
	memset((char *)&argument, 0, sizeof (argument));
	if (!svc_getargs(transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument))
		{
			svcerr_decode(transp);
			return;
		}
	result = (*local)((char *)&argument, rqstp);
	if (result != NULL && !svc_sendreply(transp, (xdrproc_t) _xdr_result, result))
		{
			svcerr_systemerr(transp);
		}
	if (!svc_freeargs(transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument))
		{
			fprintf(stderr, "%s", "unable to free arguments");
			exit(1);
		}
	return;
}

/* copied from test-rpc/tws_tsc_stat_svc.c */
void
stat_transmit_prog_1(struct svc_req *rqstp, register SVCXPRT *transp)
{
	union {
		TSCV_NetForm stat_transmit_1_arg;
	} argument;
	char *result;
	xdrproc_t _xdr_argument, _xdr_result;
	char *(*local)(char *, struct svc_req *);

	switch (rqstp->rq_proc) {
	case NULLPROC:
		(void) svc_sendreply (transp, (xdrproc_t) xdr_void, (char *)NULL);
		return;

	case STAT_TRANSMIT:
		_xdr_argument = (xdrproc_t) xdr_TSCV_NetForm;
		_xdr_result = (xdrproc_t) xdr_bool;
		local = (char *(*)(char *, struct svc_req *)) stat_transmit_1_svc;
		break;

	default:
		svcerr_noproc (transp);
		return;
	}
	memset ((char *)&argument, 0, sizeof (argument));
	if (!svc_getargs (transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument)) {
		svcerr_decode (transp);
		return;
	}
	result = (*local)((char *)&argument, rqstp);
	if (result != NULL && !svc_sendreply(transp, (xdrproc_t) _xdr_result, result)) {
		svcerr_systemerr (transp);
	}
	if (!svc_freeargs (transp, (xdrproc_t) _xdr_argument, (caddr_t) &argument)) {
		fprintf (stderr, "%s", "unable to free arguments");
		exit (1);
	}
	return;
}

void
register_tscl_receiver(int v)
{
	register SVCXPRT *transp;

	pmap_unset(LONG_TRANSMIT_PROG, LONG_TRANSMIT_VERS);

	transp = svcudp_create(RPC_ANYSOCK);
	if (!transp)
		{
			perror("svcudp_create() for TSCL");
			exit(EXIT_FAILURE);
		}

	if (!svc_register(transp, LONG_TRANSMIT_PROG, LONG_TRANSMIT_VERS, long_transmit_prog_1, IPPROTO_UDP))
		{
			perror("svc_register() for TSCL");
			exit(EXIT_FAILURE);
		}

	/* need this to allow udp client to connect */
	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (!transp)
		{
			perror("svctcp_create() for TSCL");
			exit(EXIT_FAILURE);
		}

	if (!svc_register(transp, LONG_TRANSMIT_PROG, LONG_TRANSMIT_VERS, long_transmit_prog_1, IPPROTO_TCP))
		{
			perror("svc_register() for TSCL");
			exit(EXIT_FAILURE);
		}

	if (pipe(pipefd))
		{
			perror("pipe()");
			exit(EXIT_FAILURE);
		}
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCL broadcaster at pid:%d\n", getpid());
							init_tscl_shm();
							read_and_broadcast(pipefd[0], tscl_packet_size, "TSCL", 10, v, copy_tscl_shm);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCL broadcaster at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	svc_run();
	perror("svc_run() for TSCL");
	exit(EXIT_FAILURE);
}

void
register_tscs_receiver(int v)
{
	register SVCXPRT *transp;

	pmap_unset(SHORT_TRANSMIT_PROG, SHORT_TRANSMIT_VERS);

	transp = svcudp_create(RPC_ANYSOCK);
	if (!transp)
		{
			perror("svcudp_create() for TSCS");
			exit(EXIT_FAILURE);
		}

	if (!svc_register(transp, SHORT_TRANSMIT_PROG, SHORT_TRANSMIT_VERS, short_transmit_prog_1, IPPROTO_UDP))
		{
			perror("svc_register() for TSCS");
			exit(EXIT_FAILURE);
		}

	/* need this to allow udp client to connect */
	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (!transp)
		{
			perror("svctcp_create() for TSCS");
			exit(EXIT_FAILURE);
		}

	if (!svc_register(transp, SHORT_TRANSMIT_PROG, SHORT_TRANSMIT_VERS, short_transmit_prog_1, IPPROTO_TCP))
		{
			perror("svc_register() for TSCS");
			exit(EXIT_FAILURE);
		}

	pipe(pipefd);
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCS broadcaster at pid:%d\n", getpid());
							init_tscs_shm();
							read_and_broadcast(pipefd[0], tscs_packet_size, "TSCS", 1, v, copy_tscs_shm);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCS broadcaster at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	svc_run();
	perror("svc_run() for TSCS");
	exit(EXIT_FAILURE);
}

void
register_tscv_receiver(int v)
{
	register SVCXPRT *transp;

	pmap_unset(STAT_TRANSMIT_PROG, STAT_TRANSMIT_VERS);

	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (!transp)
		{
			perror("svctcp_create() for TSCV");
			exit(EXIT_FAILURE);
		}

	if (!svc_register(transp, STAT_TRANSMIT_PROG, STAT_TRANSMIT_VERS, stat_transmit_prog_1, IPPROTO_TCP))
		{
			perror("svc_register() for TSCV");
			exit(EXIT_FAILURE);
		}

	pipe(pipefd);
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCV broadcaster at pid:%d\n", getpid());
							init_tscv_shm();
							read_and_broadcast(pipefd[0], tscv_max_packet_size, "TSCV", 600, v, copy_tscv_shm);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCV broadcaster at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	svc_run();
	perror("svc_run() for TSCV");
	exit(EXIT_FAILURE);
}

int
main(int argc, char *argv[])
{
	int v;

	/* command line options */
	{
		int o;
		v = 0;
		while((o = getopt(argc, argv, "hv")) != -1)
			{
				switch(o)
					{
					case 'h':	/* help */
						usage(stdout);
						return EXIT_SUCCESS;
					case 'v':	/* increase verbosity */
						v += 1;
						break;
					default:
						usage(stderr);
						exit(EXIT_FAILURE);
					}
			}
	}

	/* TSCL */
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCL receiver at pid:%d\n", getpid());
							register_tscl_receiver(v);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCL receiver at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	/* TSCS */
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCS receiver at pid:%d\n", getpid());
							register_tscs_receiver(v);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCS receiver at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	/* TSCV */
	if (fork() == 0)
		{
			pid_t pid;
			while(1)
				{
					pid = fork();
					if (pid == 0)
						{
							fprintf(stderr, "Starting TSCV receiver at pid:%d\n", getpid());
							register_tscv_receiver(v);
						}
					else
						{
							waitpid(pid, NULL, 0);
							fprintf(stderr, "TSCV receiver at pid:%d exited. Restarting in 1 sec.\n", pid);
							sleep(1);
						}
				}
		}

	while(wait(NULL) > 0);
	return EXIT_SUCCESS;
}

