#!/usr/bin/python
import tempfile
import os
import sys
import re
import calendar
import time

def strToTime(str):
	m = re.match('(\d{4,4})(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d)', str)
	t = calendar.timegm(map(int, m.group(1, 2, 3, 4, 5, 6)))
	t += int(m.group(7))*0.1
	return t

def timeParse(str, now = time.time()):	# Parses 'yyyy-mm-dd HH:MM', 'HH:MM', or 'MM min ago'
	m = re.match('^(\d\d):(\d\d)\Z', str.strip())
	if m:
		return time.mktime(time.localtime(now)[0:3] + (int(m.group(1)), int(m.group(2)), 0, 0, 0, -1))
	m = re.match('^(\d{4,4})-(\d\d)-(\d\d).(\d\d):(\d\d)\Z', str)
	if m:
		return time.mktime(map(int, m.group(1, 2, 3, 4, 5)) + [0, 0, 0, -1])
	m = re.match('^(\d+)\s+min.*\sago\Z', str)
	if m:
		return now - int(m.group(1))*60
	m = re.match('^(\d+)\s+hour.*\sago\Z', str)
	if m:
		return now - int(m.group(1))*60*60
	return None

# History
class History:
	def __init__(self):
		self.status = {}
		self.range_from = None
		self.range_to = None
		self.file_range_from = None
		self.file_range_to = None
	def parse(self, str):
		try:
			t_tsc, t_mlp, updown, f_alm, id, name = str.strip().split(' ', 5)
		except ValueError:
			return False
		ts = strToTime(t_mlp)
		if self.range_from != None and ts < self.range_from:
			return ts
		if self.range_to != None and self.range_to < ts:
			return ts
		id = int(id, 16)
		if not self.status.has_key(id):
			self.status[id] = Status(id)
		stat = self.status[id]
		stat.name = name
		if '1' == f_alm:
			stat.isAlarm = True
		else:
			stat.isAlarm = False
		if 'OCR' == updown:
			stat.setRaisedAt(ts)
		else:
			stat.setClearedAt(ts)
		return ts
	def setTimeRange(self, start, end):	# UNIX time
		self.range_from = start
		self.range_to = end
	def read(self, path):
		f = open(path)
		ts = None
		try:
			for line in f:
				ts = self.parse(line)
				if not self.file_range_from:
					self.file_range_from = ts
					if self.range_from and self.range_from < self.file_range_from:
						raise RuntimeError, "History file %s is too new" % path
				if ts:
					self.file_range_to = ts
					if self.range_to and self.range_to < self.file_range_to:
						break
		finally:
			f.close()
		if self.range_to:
			if (not self.file_range_to) or self.file_range_to < self.range_to:
				raise RuntimeError, "History file %s is too old" % path

# An entry of status
class Status:
	def __init__(self, id):
		self.status_id = id	# integer
		self.hasRaised = False
		self.beingRaised = False
		self.raisedAt = None
		self.clearedAt = None
		self.name = None
		self.isAlarm = False

	def setRaisedAt(self, t):	# t as a UNIX time
		self.raisedAt = t
		self.hasRaised = True
		if self.clearedAt < self.raisedAt:
			self.beingRaised = True

	def setClearedAt(self, t):	# t as a UNIX time
		self.clearedAt = t
		if self.raisedAt < self.clearedAt:
			self.beingRaised = False

# Fetch the history from TSC
def fetchHistory():
	status_file = tempfile.mkstemp()
	status_file_path = status_file[1]
	os.close(status_file[0])

	fetch_history_cmd = [
		#'/nowhere test on error',
		'exec-tsc -q 1A1901naoj NAOJ % % %',
		'exec-tsc -q 1A1034STAT%%%',
		'exec-tsc -q 1A1011',
		'scp naoj@tsc:/JNLT/DATA/TSC/USER/RECORD/APPARATUS_STATUS_ALL.LOG %s' % status_file_path,
	]
	fetch_history_cmd_ensure = [
		'exec-tsc -q 1A1902',
	]

	result = -1
	print 'Fetching status history from TSC'
	for cmd in fetch_history_cmd:
		print cmd
		result = os.system(cmd)
		if result != 0:
			break
	for cmd in fetch_history_cmd_ensure:
		print cmd
		os.system(cmd)
	if result != 0:
		sys.stderr.write("Problem in fetching status history. Stopping\n")
		os.unlink(status_file_path)
		return None
	return status_file_path

def isAlarm(status):
	return status.isAlarm
def isNotAlarm(status):
	return not status.isAlarm
def beingRaised(status):
	return status.beingRaised

def print_stats(stats):
	if len(stats) > 0:
		stats.sort(key = lambda alarm: alarm.raisedAt)
		for stat in stats:
			timestamp = time.strftime("%H:%M:%S", time.localtime(stat.raisedAt))
			status_id = "%07.7X" % stat.status_id
			name = stat.name
			print "%s %s %s" % (timestamp, status_id, name)
	else:
		print "None"

if __name__ == "__main__":
	sys.stderr.write("= Fetching latest alarm history from TSC\n")
	status_file_path = fetchHistory()
	time_from = timeParse('30 min ago', time.time())
	time_to = None

	sys.stderr.write("= Analyzing alarm history\n")
	history = History()
	history.setTimeRange(time_from, time_to)
	history.read(status_file_path)
	status = history.status.values()
	active_status = filter(beingRaised, status)

	print time.strftime("= Active flags raised after %H:%M:%S %Z", time.localtime(time_from))
	print_stats(filter(isNotAlarm, active_status))

	print time.strftime("= Active alarms raised after %H:%M:%S %Z", time.localtime(time_from))
	print_stats(filter(isAlarm, active_status))

	os.unlink(status_file_path)
