/* dummy-periodic-status-sender.c : dummy tsc sending TSCL, TSCS, and TSCV  */
/*

Copyright (C) 2010 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <sys/time.h>
#include <math.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <melco/tsc-message.h>

/* include path to be specified in Makefile */
#include "rpc-hosts.h"
#include "tws_tsc_short.h"
#include "tws_tsc_long.h"
#include "tws_tsc_stat.h"

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

typedef void (*rpc_send_callback_t)(void *argp, CLIENT *clnt);

typedef struct _path_list {
	struct _path_list *next;
	const char *path;
} path_list_s;

typedef struct _data_list {
	struct _data_list *next;
	char *addr;
	int type_code;	/* S and V:0 L1-7: 1-7 */
	size_t length;
} data_list_s;

typedef void (sender_t)(data_list_s *data_list, rpc_send_callback_t callback, CLIENT *cl);

void usage(FILE *stream);
void scan_status(path_list_s *paths, double interval, int repeat, rpc_send_callback_t callback, CLIENT *cl, sender_t sender, int block_size);
void send_status(data_list_s *data_list, rpc_send_callback_t callback, CLIENT *cl);
void send_tscv(data_list_s *data_list, rpc_send_callback_t callback, CLIENT *cl);
void paths_free(path_list_s *start);
void data_free(data_list_s *start);
void sigalrm(int x);
int get_type_code(char *mo_header_ptr);

int timedout;

void
usage(FILE *stream)
{
	fputs("usage: dummy-periodic-status-sender [options]\n", stream);
	fputs("\treads status data from files and send them to udp RPC.\n", stream);
	fputs("\tFiles will be repeateadly read in sequence to continue sending data\n", stream);
	fputs("OPTIONS - at least one -l or -s option required:\n", stream);
	fputs("-o            : Stop once files are scanned.\n", stream);
	fputs("-l file       : specifies a TSCL file to be sent every 1 sec.\n", stream);
	fputs("-s file       : specifies a TSCS file to be sent every 0.1 sec.\n", stream);
	fputs("-x multiplier : sends status at a higher rate.\n", stream);
	fputs("-h            : shows this\n", stream);
}

void
scan_status(path_list_s *paths, double interval, int repeat, rpc_send_callback_t callback, CLIENT *cl, sender_t sender, int block_size)
{
	path_list_s *cur_path;
	data_list_s out_data, *cur_data, *new_data;;
	struct itimerval timer;
	message_header_s header;
	int prev_type_code, cur_type_code;
	int data_length;
	int repeatable;

	/* start interval timer */
	timer.it_interval.tv_sec = (long) floor(interval);
	timer.it_interval.tv_usec = (long) ((interval - (double) timer.it_interval.tv_sec) * 1e6);
	timer.it_value = timer.it_interval;
	timedout = 0;
	signal(SIGALRM, sigalrm);
	if (setitimer(ITIMER_REAL, &timer, NULL))
		{
			perror("setitimer()");
			exit(EXIT_FAILURE);
		}

	/* read and send files */
	repeatable = 0;
	do
		{
			int fd;
			char *addr;
			size_t length;
			size_t pos;

			cur_path = paths;
			prev_type_code = 0;
			while(cur_path)
				{
					if (!cur_path->path)
						{
							cur_path = cur_path->next;
							continue;
						}

					out_data.next = NULL;
					cur_data = &out_data;
					/* map the file */
					{
						struct stat sb;
						fd = open(cur_path->path, O_RDONLY);
						addr = NULL;
						length = 0;
						pos = 0;
						if (fd == -1)
							{
								perror(cur_path->path);
								cur_path->path = NULL;
								cur_path = cur_path->next;
								continue;
							}
						else
							{
								if (fstat(fd, &sb) == -1)
									{
										perror(cur_path->path);
										close(fd);
										cur_path->path = NULL;
										fd = -1;
										goto next_file;
									}
								else
									{
										length = sb.st_size;
										addr = mmap(NULL, length, PROT_READ, MAP_PRIVATE, fd, 0);
										if (addr == MAP_FAILED)
											{
												perror(cur_path->path);
												close(fd);
												fd = -1;
												length = 0;
												addr = NULL;
												goto next_file;
											}
									}
							}
					}

					pos = 0;
					memcpy(&header, addr + pos, sizeof(header));
					if (!message_header_valid(&header))
						{
							fprintf(stderr, "%s: invalid format as status message. Ignoring\n", cur_path->path);
							if (munmap(addr, length)) perror(cur_path->path);
							if (close(fd)) perror(cur_path->path);
							addr = NULL;
							fd = -1;
							cur_path = cur_path->next;
							continue;
						}
					cur_type_code = get_type_code(addr + pos);
					data_length = message_header_length(&header) + sizeof(header);

					while(pos < length)
						{
							/* memo */
							new_data = (data_list_s *) malloc(sizeof(data_list_s));
							if (!new_data)
								{
									perror("malloc()");
									exit(EXIT_FAILURE);
								}
							new_data->addr = addr + pos;
							new_data->length = data_length;
							new_data->next = NULL;
							new_data->type_code = cur_type_code;

							/* flush if needed */
							if (cur_type_code > prev_type_code)
								{
									/* pass */
								}
							else
								{
									/* flush */
									if (out_data.next)
										{
											sender(out_data.next, callback, cl);
											repeatable = 1;
											data_free(out_data.next);
											out_data.next = NULL;
											cur_data = &out_data;
										}
								}

							/* record */
							cur_data->next = new_data;
							cur_data = new_data;

							prev_type_code = cur_type_code;

							/* check next */
							pos += cur_data->length;
							memcpy(&header, addr + pos, sizeof(header));
							if (!message_header_valid(&header))
								{
									/* maybe padded */
									pos -= cur_data->length;
									pos += block_size;
									while(pos < length)
										{
											memcpy(&header, addr + pos, sizeof(header));
											if (message_header_valid(&header)) break;
											pos += block_size;
										}
								}
							if (pos >= length) goto next_file;

							cur_type_code = get_type_code(addr + pos);
							data_length = message_header_length(&header) + sizeof(header);
						}

					next_file:
					if (out_data.next)
						{
							sender(out_data.next, callback, cl);
							repeatable = 1;
							data_free(out_data.next);
						}

					/* unmap the file */
					if (munmap(addr, length)) perror(cur_path->path);
					if (close(fd)) perror(cur_path->path);

					cur_path = cur_path->next;
				}
		}
	while(repeatable && repeat);

	/* stop interval timer */
	timer.it_interval.tv_sec = 0;
	timer.it_interval.tv_usec = 0;
	timer.it_value = timer.it_interval;
	if (setitimer(ITIMER_REAL, &timer, NULL)) perror("setitimer()");
	signal(SIGALRM, SIG_DFL);

	if (setitimer(ITIMER_REAL, &timer, NULL))
		{
			perror("setitimer()");
			exit(EXIT_FAILURE);
		}
	
	paths_free(paths);
	exit(EXIT_SUCCESS);
}

void
paths_free(path_list_s *start)
{
	path_list_s *cur, *next;
	cur = start;
	while(cur)
		{
			next = cur->next;
			free(cur);
			cur = next;
		}
}

void
send_status(data_list_s *data_list, rpc_send_callback_t callback, CLIENT *cl)
{
	data_list_s *cur;

	if (timedout) fputs("Could not keep up with the speed.\n", stderr);
	errno = 0;
	pause();
	assert(errno == EINTR);
	timedout = 0;

	cur = data_list;
	while(cur)
		{
			fprintf(stderr, "%35.35s %4d bytes\n", cur->addr, cur->length);
			callback(cur->addr, cl);
			cur = cur->next;
		}
}

void
send_tscv(data_list_s *data_list, rpc_send_callback_t callback, CLIENT *cl)
{
	data_list_s *cur;
	TSCV_NetForm form;

	if (timedout) fputs("Could not keep up with the speed.\n", stderr);
	errno = 0;
	pause();
	assert(errno == EINTR);
	timedout = 0;

	cur = data_list;
	while(cur)
		{
			fprintf(stderr, "%35.35s %4d bytes\n", cur->addr, cur->length);
			form.TSCV_NetForm_val = cur->addr;
			form.TSCV_NetForm_len = cur->length;
			callback(&form, cl);
			cur = cur->next;
		}
}

void
data_free(data_list_s *start)
{
	data_list_s *cur, *next;
	cur = start;
	while(cur)
		{
			next = cur->next;
			free(cur);
			cur = next;
		}
}

void sigalrm(int x __attribute__((unused)))
{
	timedout = 1;
}

int
get_type_code(char *mo_header_ptr)
{
	switch(*(mo_header_ptr + sizeof(message_header_s)))
		{
		case 'S': return 0;
		case 'V': return 0;
		case 'L': return *(mo_header_ptr + sizeof(message_header_s) + 1) - '0';
		default: return 0;
		}
}

int
main(int argc, char *argv[])
{
	int repeat;
	double speed;
	path_list_s tscl_paths;
	path_list_s tscs_paths;
	path_list_s tscv_paths;
	char *dst = TWS4_HOST_NAME;

	tscl_paths.next = NULL;
	tscs_paths.next = NULL;
	tscv_paths.next = NULL;

	/* command line options */
	{
		int o;
		path_list_s *tscl_cur, *tscs_cur, *tscv_cur;
		repeat = 1;
		tscl_cur = &tscl_paths;
		tscs_cur = &tscs_paths;
		tscv_cur = &tscv_paths;
		path_list_s *new_path;
		speed = 1.0;
		while((o = getopt(argc, argv, "hx:l:s:v:o")) != -1)
			{
				switch(o)
					{
					case 'h':	/* help */
						usage(stdout);
						return EXIT_SUCCESS;
					case 'o':
						repeat = 0;
						break;
					case 'x':
						speed = atof(optarg);
						if (speed <= 0)
							{
								fputs("-x: multiplier to speed must be more than zero\n", stderr);
								exit(EXIT_FAILURE);
							}
						break;
					case 'l':	/* TSCL file */
					case 's':	/* TSCS file */
					case 'v':	/* TSCV file */
						new_path = (path_list_s *) malloc(sizeof(path_list_s));
						assert(new_path);
						new_path->path = optarg;
						new_path->next = NULL;
						switch(o)
							{
							case 'l':
								tscl_cur->next = new_path;
								tscl_cur = new_path;
								break;
							case 's':
								tscs_cur->next = new_path;
								tscs_cur = new_path;
								break;
							case 'v':
								tscv_cur->next = new_path;
								tscv_cur = new_path;
								break;
							}
						break;
					default:
						usage(stderr);
						exit(EXIT_FAILURE);
					}
			}
	}
	if (!tscs_paths.next && !tscl_paths.next && !tscv_paths.next)
		{
			fputs("At least one file must be specified with -l, -s, or -v\n", stderr);
			exit(EXIT_FAILURE);
		}

	if (tscl_paths.next && fork() == 0)
		{
			CLIENT *cl;
			cl = clnt_create(dst,  LONG_TRANSMIT_PROG, LONG_TRANSMIT_VERS, "udp");
			if (!cl)
				{
					clnt_pcreateerror("clnt_create() for TSCL");
					exit(EXIT_FAILURE);
				}
			scan_status(tscl_paths.next, 1 / speed, repeat, (rpc_send_callback_t) long_transmit_1, cl, send_status, FUJITSU_TSCL_BLOCK_SIZE);
		}

	if (tscs_paths.next && fork() == 0)
		{
			CLIENT *cl;
			cl = clnt_create(dst,  SHORT_TRANSMIT_PROG, SHORT_TRANSMIT_VERS, "udp");
			if (!cl)
				{
					clnt_pcreateerror("clnt_create() for TSCS");
					exit(EXIT_FAILURE);
				}
			scan_status(tscs_paths.next, 0.1 / speed, repeat, (rpc_send_callback_t) short_transmit_1, cl, send_status, FUJITSU_TSCS_BLOCK_SIZE);
		}

	if (tscv_paths.next && fork() == 0)
		{
			CLIENT *cl;
			cl = clnt_create(dst,  STAT_TRANSMIT_PROG, STAT_TRANSMIT_VERS, "tcp");
			if (!cl)
				{
					clnt_pcreateerror("clnt_create() for TSCV");
					exit(EXIT_FAILURE);
				}
			scan_status(tscv_paths.next, 5 / speed, repeat, (rpc_send_callback_t) stat_transmit_1, cl, send_tscv, FUJITSU_TSCV_BLOCK_SIZE);	/* 5 sec is arbitrary */
		}

	/* free the list */
	while(wait(NULL) > 0);
	paths_free(tscs_paths.next);
	paths_free(tscl_paths.next);
	paths_free(tscv_paths.next);

	return EXIT_SUCCESS;
}

