/* command-dispatcher.c : pretends command interface of TSC */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <signal.h>

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifndef HAVE_PSELECT
#include "pselect.h"
#endif

#include <melco/tsc-message.h>
#include <test-tsc/command-dispatcher.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define DATA_SIZE 128

/* dummy login information */
tsc_login_s tsc_login[] = {
	{"naoj", "naojpassword", 1, 15, 1, 0},
	{"soss", "sosspassword", 8, 3, 8, 0},
	{NULL, NULL, 0, 0, 0, 0},	/* centinel */
};

/* private functions */
static void show_log(FILE* stream, char *action, char *string, int length);
static void sigchld(int x);

/* sequential number managment */
static long next_rxnum(void);

/* execution */
static void _fork_and_execute(CommandMessage *command, ReceptionResponseMessage *reception, response_message_t completion_result, tsc_completion_callback_t callback, int waitsec, int fd);

/* par process variable */
int got_sigchld;
long tsc_rxnum = -1;
long tsc_seqnum = -1;

static void
sigchld(int x __attribute__ ((unused)))
{
	while(waitpid(-1, NULL, WNOHANG) > 0);
	got_sigchld = 1;
	signal(SIGCHLD, sigchld);
}

static void
show_log(FILE* stream, char *action, char *string, int length)
{
	int i;
	fprintf(stream, TSC_COMM_FORMAT, action);
	for(i = 0; i < length; i++)
		{
			if (isprint((int) string[i]))
				fputc(string[i], stream);
			else
				fputc('.', stream);
		}
	fputc('\n', stream);
}

static long
next_rxnum(void)
{
	tsc_rxnum++;
	if (tsc_rxnum > MAX_SEQNUM) tsc_rxnum = 0;
	return tsc_rxnum;
}

static void
_fork_and_execute(CommandMessage *command, ReceptionResponseMessage *reception, response_message_t completion_result, tsc_completion_callback_t callback, int waitsec, int fd)
{
	if (fork() == 0)
		{
			/* child */
			int success = 0;

			/* sleep instead execute :) */
			if (waitsec > 0) sleep(waitsec);

			/* make up completion response */
			{
				tsc_error_t r;
				long rxnum;
				int orig_pid;
				char orig_src[5], orig_dst[5];
				struct timeval txtime;
				CompletionResponseMessage *completion;

				rxnum = reception->rxnum;
				message_header_src(command->d, orig_src);
				message_header_dst(command->d, orig_dst);
				orig_pid = message_header_pid(command->d);
				memset(&txtime, 0, sizeof(txtime));
				gettimeofday(&txtime, NULL);

				completion = CompletionResponseMessage_alloc();
				if (!completion)
					{
						fputs("memory exhausted\n", stderr);
						exit(EXIT_FAILURE);
					}
				if (0x1A1901 == command->id)	/* returns uid and gid */
					{
						int i, len;
						char buf[DATA_SIZE];
						tsc_login_s *login;
						const char *username, *password;
						int lusername, lpassword;

						login = NULL;
						do {
							username = NULL;
							CommandMessage_arg(command, &username, &lusername);
							if (!username || lusername < 1) break;

							password = username + lusername;
							CommandMessage_arg(command, &password, &lpassword);
							if (!password || lpassword < 1) break;

							for(i = 0; tsc_login[i].username; i++)
								{
									if (strlen(tsc_login[i].username) == lusername &&
										!strncmp(username, tsc_login[i].username, lusername) &&
										strlen(tsc_login[i].password) == lpassword &&
										!strncmp(password, tsc_login[i].password, lpassword))
										{
											login = &tsc_login[i];
											break;
										}
								}
						} while(0);

						if (login)
							{
								len = snprintf(buf, sizeof(buf), "%02d %04d %d %d", login->uid, login->gid, login->level, login->lang);
								r = CompletionResponseMessage_set(completion, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, response_complete, 0, buf, len);
							}
						else
							{
								r = CompletionResponseMessage_set(completion, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, response_fail, 0, NULL, 0);
							}
					}
				else	/* returns nothing */
					{
						if (completion_result != response_fail)
							{
								r = CompletionResponseMessage_set(completion, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, completion_result, 0, NULL, 0);
							}
						else
							{
								char parameter[] = "parameter";
								r = CompletionResponseMessage_set(completion, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, completion_result, 0, parameter, sizeof(parameter));
							}
					}
				if (fd == -1)
					show_log(stdout, "complete", CompletionResponseMessage_tostr(completion), CompletionResponseMessage_strlen(completion));
				else
					write(fd, CompletionResponseMessage_tostr(completion), CompletionResponseMessage_strlen(completion));
				if (completion->result == response_complete) success = 1;
				if (callback) callback(completion);
				CompletionResponseMessage_decref(completion);
			}
			CommandMessage_decref(command);
			ReceptionResponseMessage_decref(reception);
			exit(success ? EXIT_SUCCESS : EXIT_FAILURE);
		}
}

ReceptionResponseMessage*
tsc_execute(char *commandstr, void (*command_free)(void* ptr), tsc_completion_callback_t callback, response_message_t reception_result, response_message_t completion_result, int headerwait, int realwait, int responsewait, int fd)
{
	tsc_error_t command_parse;
	CommandMessage *command;
	ReceptionResponseMessage *reception;
	long rxnum;
	int orig_pid;
	char orig_src[5], orig_dst[5];
	struct timeval txtime;

	/* parse the command */
	assert(command = CommandMessage_alloc());
	command_parse = strto_CommandMessage(command, commandstr, command_free);
	rxnum = next_rxnum();
	message_header_src(command->d, orig_src);
	message_header_dst(command->d, orig_dst);
	orig_pid = message_header_pid(command->d);
	if (fd == -1) show_log(stdout, "command", CommandMessage_tostr(command), CommandMessage_strlen(command));

	/* wait if needed */
	sleep(responsewait);

	/* check result of parsing */
	reception = ReceptionResponseMessage_alloc();
	if(!reception)
		{
			fputs("memory exhausted\n", stderr);
			exit(EXIT_FAILURE);
		}
	memset(&txtime, 0, sizeof(txtime));
	gettimeofday(&txtime, NULL);
	if (command_parse != tsc_success)
		{
			ReceptionResponseMessage_set(reception, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, response_ng, 0);
			if (fd == -1)
				show_log(stdout, "received", ReceptionResponseMessage_tostr(reception), ReceptionResponseMessage_strlen(reception));
			else
				write(fd, ReceptionResponseMessage_tostr(reception), ReceptionResponseMessage_strlen(reception));
			CommandMessage_decref(command);
			return reception;
		}

	/* make up the response */
	{
		time_t endtime;
		tsc_error_t r;
		endtime = txtime.tv_sec + headerwait;
		r = ReceptionResponseMessage_set(reception, orig_src, orig_dst, orig_pid, &txtime, command->seqnum, command->id, rxnum, reception_result, endtime);
		if (r == tsc_error_malloc)
			{
				fputs("memory exhausted\n", stderr);
				exit(EXIT_FAILURE);
			}
		assert(r == tsc_success);
		if (fd == -1)
			show_log(stdout, "received", ReceptionResponseMessage_tostr(reception), ReceptionResponseMessage_strlen(reception));
		else
			write(fd, ReceptionResponseMessage_tostr(reception), ReceptionResponseMessage_strlen(reception));
	}

	/* execute the command */
	if (reception_result == response_ok)
		_fork_and_execute(command, reception, completion_result, callback, realwait, fd);

	CommandMessage_decref(command);
	return reception;
}

void
tsc_server(int input_fd, int output_fd, int waitsec)
{
	response_message_t rec_ress[] = REC_RES;
	response_message_t com_ress[] = COM_RES;
	int response_waits[] = {0, 0, 0, 0, 0, 0, 0, 0, 2};
	sigset_t sigmask, orig_sigmask;

	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGALRM);
	sigaddset(&sigmask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &sigmask, &orig_sigmask);
	signal(SIGCHLD, sigchld);

	if (waitsec < 0) srandom(-waitsec);
	got_sigchld = 0;
	while(1)
		{
			char *command_str;
			fd_set readfds;
			FD_ZERO(&readfds);
			FD_SET(input_fd, &readfds);
			if (pselect(input_fd + 1, &readfds, NULL, NULL, NULL, &orig_sigmask) < 0)
				{
					if (got_sigchld)
						{
							got_sigchld = 0;
							continue;
						}
				}
			if ((command_str = read_Message(input_fd)))
				{
					ReceptionResponseMessage *r;
					int header_wait, real_wait, response_wait;
					response_message_t rec_res, com_res;
					if (waitsec < 0)
						{
							rec_res = rec_ress[random() % (sizeof(rec_ress)/sizeof(char*))];
							com_res = com_ress[random() % (sizeof(com_ress)/sizeof(char*))];
							header_wait = random() % RANDOM_WAIT_MAX;
							real_wait = random() % (2*RANDOM_WAIT_MAX);
							response_wait = response_waits[random() % (sizeof(response_waits)/sizeof(int))];
						}
					else
						{
							rec_res = response_ok;
							com_res = response_complete;
							header_wait = waitsec;
							real_wait = waitsec;
							response_wait = 0;
						}
					r = tsc_execute(command_str, free, NULL, rec_res, com_res, header_wait, real_wait, response_wait, output_fd);
					ReceptionResponseMessage_decref(r);
				}
		}
}
