/* test-status-types.c: test cases for status-types.c */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <melco/status-types.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

typedef void* (*_to_s)(const void *addr, const unsigned char mask, const size_t length, void *dst);
typedef void* (*_to_d)(const void *addr, const unsigned char mask, const size_t length, void *dst);
typedef int (*_to_l)(const void *addr, const unsigned char mask, const size_t length, void *dst);

void test_ASC15_TIME(void);
void test_ASC7_TIME(void);
void test_str_to_s(void);
void test_radec_to_s(void);
void test_radec_to_d(void);
void test_numeric_to_s(void);
void test_to_hex(void);
void test_invalid_bcd(void);

void
test_ASC15_TIME(void)
{
	ASC15_TIME data;
	char testdate[] = "200502220252415", testbuf[16];
	struct timeval tv, cal;
	double d;
	long long i;

	fprintf(stderr, "%s\n", __func__);

	strncpy(data.buf, testdate, sizeof(data.buf));

	{
		struct tm tmp_tm;
		memset(&tmp_tm, 0, sizeof(tmp_tm));
		tmp_tm.tm_year = 2005 - 1900;
		tmp_tm.tm_mon = 2 - 1;
		tmp_tm.tm_mday = 22;
		tmp_tm.tm_hour = 2;
		tmp_tm.tm_min = 52;
		tmp_tm.tm_sec = 41;
		tv.tv_sec = timegm(&tmp_tm);
		tv.tv_usec = 5*100000;
	}

	assert(ASC15_TIME_to_t(&data, 0, 15, &cal));
	assert(cal.tv_sec == tv.tv_sec);
	assert(cal.tv_usec == tv.tv_usec);
	assert(ASC15_TIME_to_d(&data, 0, 15, &d));
	assert((double) tv.tv_sec + 0.5 == d);
	assert(ASC15_TIME_to_i(&data, 0, 15, &i));
	assert(tv.tv_sec + 1 == i);
	assert(ASC15_TIME_to_s(&data, 0, 15, testbuf));
	assert(!strcmp(testdate, testbuf));
}

void
test_ASC7_TIME(void)
{
	ASC7_TIME data;
	char testdate[] = "0252415", testbuf[8];
	struct timeval tv, cal;
	double d;
	long long i;

	fprintf(stderr, "%s\n", __func__);

	strncpy(data.buf, testdate, sizeof(data.buf));

	{
		struct tm tmp_tm;
		memset(&tmp_tm, 0, sizeof(tmp_tm));
		tmp_tm.tm_year = 1971;
		tmp_tm.tm_mon = 0;
		tmp_tm.tm_mday = 1;
		tmp_tm.tm_hour = 2;
		tmp_tm.tm_min = 52;
		tmp_tm.tm_sec = 41;
		tv.tv_sec = timegm(&tmp_tm);
		tv.tv_usec = 5*100000;
	}

	assert(ASC7_TIME_to_t(&data, 0, 7, &cal));
	assert(cal.tv_sec == tv.tv_sec);
	assert(cal.tv_usec == tv.tv_usec);
	assert(ASC7_TIME_to_d(&data, 0, 7, &d));
	assert((double) tv.tv_sec + 0.5 == d);
	assert(ASC7_TIME_to_i(&data, 0, 7, &i));
	assert(tv.tv_sec + 1 == i);
	assert(ASC7_TIME_to_s(&data, 0, 7, testbuf));
	assert(!strcmp(testdate, testbuf));
}

void
test_str_to_s(void)
{
  char teststring[] = "hello world";
  char dst[sizeof(teststring)];

	fprintf(stderr, "%s\n", __func__);

  assert(str_to_s((void*) teststring, 0, strlen(teststring), dst));
  assert(!strcmp(teststring, dst));
}

void
test_numeric_to_s(void)
{
	struct {
		int length;
		char *data;
		char *target;
		_to_s to_s;
		_to_l to_l;
	} t[] = {
		{6, "\x81\x31\x84\x39\x35\x00", "-131.84393500", BCD6_to_s, BCD6_to_s_length},
		{6, "\x80\x00\x76\x31\x70\x00", "-0.76317000", BCD6_to_s, BCD6_to_s_length},
		/* BINARY1_0p01_to_s; 000902E SH REF I */
		{1, "\x22", "34", BINARY1_1_to_s, BINARY1_1_to_s_length},	/* AG Binning Pattern */
		/* BINARY2_0p01_to_s; 0007128 SV Star Position1 Intensity on SV */
		{2,"\x00\x14", "2.0", BINARY2_0p1_to_s, BINARY2_0p1_to_s_length}, /* PMFXS8-60 Temp MON1 Data */
		{2, "\x0C\x75", "3189", BINARY2_1_to_s, BINARY2_1_to_s_length}, /* SV Star Position1 Intensity on MLP1 */
		{4, "\xFF\xFF\xFE\x68", "-0.000000000408", BINARY4_0p000000000001_to_s, BINARY4_0p000000000001_to_s_length},	/* Time GEN Freq S.D. */
		/* BINARY4_0p000001_to_s; 00A11E4 PA IEC A1 Offset */
		/* BINARY4_0p00001_to_s; 00B1090 Fast Guide Signal AZ */
		/* BINARY4_0p001_to_s; 0007001 SV Field Real Position */
		{4, "\xFF\xFF\xF0\x56", "-40.10", BINARY4_0p01_to_s, BINARY4_0p01_to_s_length},	/* AG CCD Temperature */
		{4, "\x00\x00\x18\x4C", "622.0",  BINARY4_0p1_to_s, BINARY4_0p1_to_s_length}, /* Weather Atmospheric Pressure */
		/* BINARY4_100_to_s; 000700E SV Exposure Time */
		{4, "\x00\x00\x0F\x8C", "3980", BINARY4_1_to_s, BINARY4_1_to_s_length},	/* AG Exposure Time */
		/* REAL8_to_s; 004001F Zernike Data RMS(without A20) */
		{8, "\x7E\xB8\xEE\xDC\x1F\x1D\x62\x3F", "0.00221115", REAL8_to_s, REAL8_to_s_length},
		{0, NULL, NULL, NULL, NULL}
	};
	char *dst = NULL;
	int i, len;

	fprintf(stderr, "%s\n", __func__);

	for(i = 0; t[i].data; i++)
		{
			len = (t[i].to_l)(t[i].data, 0, t[i].length, dst);
			dst = (char*) malloc(len);
			if (!dst)
				{
					fputs("memory exhausted\n", stderr);
					exit(EXIT_FAILURE);
				}
			(t[i].to_s)(t[i].data, 0, t[i].length, dst);

			if((int) strlen(t[i].target)+1 != len)
				fprintf(stderr, "#%d: len target:%d got:%d\n", i+1, strlen(t[i].target)+1, len);
			if(strcmp(t[i].target, dst))
				fprintf(stderr, "#%d: string target:\"%s\" got:\"%s\"\n", i+1, t[i].target, dst);
			assert((int) strlen(t[i].target)+1 == len);
			assert(!strcmp(t[i].target, dst));
			free(dst);
		}
}

void
test_radec_to_d(void)
{
	struct {
		char *data;
		double target;
		double epsilon;
		_to_d to_d;
	} t[] = {
		{"\x06\x39\x05\x60\x50\x00", 1.74137353596674, 7.27220521664304e-11, BCD6_RA_to_d},
		{"\x00\x08\x55\x05\x47\x99", 0.155651758941072, 4.84813681109536e-10, BCD6_DEC_to_d},
		{"\x03\x32\x00\x31\x00\x00", 0.925047047393166, 7.27220521664304e-11, BCD6_RA_to_d},
		{"\x80\x27\x42\x18\x00\x00", -0.483543469265029, 4.84813681109536e-10, BCD6_DEC_to_d},
		{"\x80\x00\x00\x01\x79\x80", -1.30754249795242e-06, 7.27220521664304e-11, BCD6_RA_to_d},
		{"\x80\x00\x01\x36\x86\x00", -0.000469590531522697, 4.84813681109536e-10, BCD6_DEC_to_d},
		{NULL, 0, 0, NULL}
	};
	double dst;
	int i;

	fprintf(stderr, "%s\n", __func__);

	for(i = 0; t[i].data; i++)
		{
			(t[i].to_d)(t[i].data, 0, sizeof(dst), &dst);
			if(abs(t[i].target - dst) > t[i].epsilon)
				{
					int j;
					fprintf(stderr, "#%d: data:", i+1);
					for(j = 0; j < 6; j++)
						fprintf(stderr, "%02X ", (unsigned char) t[i].data[j]);
					fprintf(stderr, "got:%g\n", dst);
				}
			assert(abs(t[i].target - dst) <= t[i].epsilon);
		}
}
void
test_radec_to_s(void)
{
	struct {
		int length;
		char *data;
		char *target;
		_to_s to_s;
		_to_l to_l;
	} t[] = {
		{6, "\x06\x39\x05\x60\x50\x00", "+06:39:05.605000", BCD6_RA_to_s, BCD6_RA_to_s_length},
		{6, "\x00\x08\x55\x05\x47\x99", "+08:55:05.4799", BCD6_DEC_to_s, BCD6_DEC_to_s_length},
		{6, "\x03\x32\x00\x31\x00\x00", "+03:32:00.310000", BCD6_RA_to_s, BCD6_RA_to_s_length},
		{6, "\x80\x27\x42\x18\x00\x00", "-27:42:18.0000", BCD6_DEC_to_s, BCD6_DEC_to_s_length},
		{6, "\x80\x00\x00\x01\x79\x80", "-00:00:00.017980", BCD6_RA_to_s, BCD6_RA_to_s_length},
		{6, "\x80\x00\x01\x36\x86\x00", "-00:01:36.8600", BCD6_DEC_to_s, BCD6_DEC_to_s_length},
		{0, NULL, NULL, NULL, NULL}
	};
	char *dst = NULL;
	int i, len;

	fprintf(stderr, "%s\n", __func__);

	for(i = 0; t[i].data; i++)
		{
			len = (t[i].to_l)(t[i].data, 0, t[i].length, dst);
			dst = (char*) malloc(len);
			if (!dst)
				{
					fputs("memory exhausted\n", stderr);
					exit(EXIT_FAILURE);
				}
			(t[i].to_s)(t[i].data, 0, t[i].length, dst);

			if((int) strlen(t[i].target)+1 != len)
				fprintf(stderr, "#%d: len target:%d got:%d\n", i+1, strlen(t[i].target)+1, len);
			if(strcmp(t[i].target, dst))
				fprintf(stderr, "#%d: string target:\"%s\" got:\"%s\"\n", i+1, t[i].target, dst);
			assert((int) strlen(t[i].target)+1 == len);
			assert(!strcmp(t[i].target, dst));
			free(dst);
		}
}

void
test_to_hex(void)
{
	char testdat[] = "\x12\x34\xab\xcd";
	char testbuf[12];

	fprintf(stderr, "%s\n", __func__);

  assert(1 == to_hex_length(NULL, 0, 0, NULL));
  assert(3 == to_hex_length(NULL, 0, 1, NULL));
  assert(sizeof(testbuf) == to_hex_length(NULL, 0, 4, NULL));

	to_hex(testdat, 0, 4, testbuf);
	assert(!strcmp("12-34-AB-CD", testbuf));
}

void
test_invalid_bcd(void)
{
	char invalid_dat[] = "\xff\xff\xff\xff\xfe\xff";
	char valid_dat[] = "\x00\x00\x12\x34\x56\x78";
	char testbuf[18];
	long long i;

	fprintf(stderr, "%s\n", __func__);

	assert(!BCD6_to_i(invalid_dat, 0, 6, &i));
	assert(sizeof(testbuf) == BCD6_to_s_length(invalid_dat, 0, 6, &testbuf));
	assert(!BCD6_to_s(invalid_dat, 0, 6, &testbuf));
	assert(!strcmp("FF-FF-FF-FF-FE-FF", testbuf));
	assert(BCD6_to_i(valid_dat, 0, 6, &i));
	assert(12345678 == i);
}

int
main(int argc __attribute__ ((unused)), char *argv[])
{ 
  fprintf(stderr, "%s running\n", argv[0]);
	test_ASC15_TIME();
	test_ASC7_TIME();
	test_str_to_s();
	test_numeric_to_s();
	test_radec_to_s();
	test_radec_to_d();
	test_to_hex();
	test_invalid_bcd();
	return EXIT_SUCCESS;
}
