/* test-shm-reading.c: test cases for message.c reading TWS4 shared memory */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <melco/tsc-message.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

/* paths to shared memory images */
#define TSCSFILE "test-data/050418-1615xx-shm-0x100.bin"
#define TSCLFILE "test-data/050418-1615xx-shm-0x101.bin"
#define TSCVFILE "test-data/050418-1615xx-shm-0x102.bin"

/* sizes of shared memories */
#define TSCSSIZE 1292
#define TSCLSIZE 9044
#define TSCVSIZE 12931

/* instrument IDs(shortids) of shared memories from FY2003_TSC_Status.pdf */
#define TSCSIDS {0x0001,0x0002,0x0004,0x00A1,\
	/* broken: 0x0036, */\
	/* broken: 0x0028, */\
	0x00B1,\
	0}
#define TSCSRXT {1113876960.4,1113876960.4,1113876960.4,1113876960.5,\
	1113876960.5}
#define TSCLIDS \
	{0x00A1,0x00B1,0x0001,0x0002,0x0004,0x0006,0x0007,0x0027,0x002E,\
	0x00B2,\
	0x00B2,0x0009,0x0039,\
	0x0003,0x00B3,0x002C,0x002D,0x0038,\
	0x0030,0x002A,0x002B,/* broken:0x0008,0x0051,0x003A,*/\
	0x000C,0x0029,0x0031,/* broken:0x0032,*/0x000E,0x0010,/* broken:0x0028,0x0036,0x0037*/0x0061,0x000D,\
	/* broken:0x003B,*/\
	0}
	/* L1 through L7 on each line */
#define TSCLRXT \
	{1113876965.3,1113876965.0,1113876964.9,1113876964.9,1113876964.9,1113876964.9,1113876964.9,1113876964.9,1113875335.8,\
	1113876965.2,\
	1113876965.2,1113876964.4,1113876964.0,\
	1113876964.6,1113876964.9,1113876964.8,1113876964.2,1113876963.8,\
	1113876964.3,1113876963.6,1113876964.3,\
	1113876964.2,1113876963.7,1113876963.7,1113876963.3,1113876963.7,1113876964.2,1113876964.9,\
	}
#define TSCVIDS \
	{0x0006,/*0x0008,0x0051,*/0x0037,0x0027,/*0x0032,*/0x000E,/*0x003A,*/\
	0x002E,0x0038,0x0024,0x002B,0x0061,0x002A,0x0001,0x000D,0x0004,0x0036,0x0029,\
	0x0031,0x000B,0x0010,0x000A,0x00B1,0x00B2,0x00B3,0x0002,0x0858,0x0859,\
	0x0860,0x0039,0x0009,0x0003,0x0033,0x0034,0x0035,0x0007,0x0030,0x002D,\
	0x002C,0x0025,0x00A1,/*0x00C2,0x0028,*/\
	0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,\
	0x0111,0x0112,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,\
	0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,\
	0x0301,0x0302,0x0303,0x0304,0x0305,0x0306,0x0307,0x0308,0x0309,0x0310,\
	0x0311,0x0312,0x0313,0x0314,0x0315,0x0316,0x0317,0x0318,0x0319,0x0320,\
	0x0321,0x0322,0x0323,0x0324,0x0401,0x0402,0x0403,0x0404,0x0405,0x0406,\
	0x0407,0x0408,0x0409,0x0410,0x0411,0x0412,0x0413,0x0414,0x0415,0x0416,\
	0x0417,0x0418,0x0419,0x0420,0x0421,0x0422,0x0423,0x0424,0x0425,0x0426,\
	0x0427,0x0428,0x0429,0x0430,0x0501,0x0502,0x0503,0x0504,0x0505,0x0506,\
	0x0507,0x0508,0x0509,0x0510,0x0511,0x0512,0x0513,0x0514,0x0515,0x0516,\
	0x0517,0x0518,0x0519,0x0520,0x0521,0x0522,0x0523,0x0524,0x0525,0x0526,\
	0x0527,0x0528,0x0529,0x0530,0x0531,0x0532,0x0533,0x0534,0x0535,0x0536,\
	0x0601,0x0602,0x0603,0x0604,0x0605,0x0606,0x0607,0x0608,0x0609,0x0610,\
	0x0611,0x0612,0x0613,0x0614,0x0615,0x0616,0x0617,0x0618,0x0619,0x0620,\
	0x0621,0x0622,0x0623,0x0624,0x0625,0x0626,0x0627,0x0628,0x0629,0x0630,\
	0x0631,0x0632,0x0633,0x0634,0x0635,0x0636,0x0637,0x0638,0x0639,0x0640,\
	0x0641,0x0642,0x0701,0x0702,0x0703,0x0704,0x0705,0x0706,0x0707,0x0708,\
	0x0709,0x0710,0x0711,0x0712,0x0713,0x0714,0x0715,0x0716,0x0717,0x0718,\
	0x0719,0x0720,0x0721,0x0722,0x0723,0x0724,0x0725,0x0726,0x0727,0x0728,\
	0x0729,0x0730,0x0731,0x0732,0x0733,0x0734,0x0735,0x0736,0x0737,0x0738,\
	0x0739,0x0740,0x0741,0x0742,0x0743,0x0744,0x0745,0x0746,0x0747,0x0748,\
	0x0801,0x0802,0x0803,0x0804,0x0805,0x0806,0x0807,0x0808,0x0809,0x0810,\
	0x0811,0x0812,0x0813,\
	0x0814,0x0815,\
	0x0816,0x0817,0x0818,0x0819,0x0820,0x0821,0x0822,0x0823,0x0824,0x0825,\
	0x0826,0x0827,0x0828,0x0829,0x0830,0x0831,0x0832,0x0833,0x0834,0x0835,\
	0x0836,0x0837,0x0838,0x0839,0x0840,0x0841,0x0842,0x0843,0x0844,0x0845,\
	0x0846,0x0847,0x0848,0x0849,0x0850,0x0851,0x0861,0x0862,0x0863,\
	/*0x003B,*/0x003C,\
	0}

void test_reading(const char *file, const unsigned int shortids[], const double rxtimes[], const char *type);
void test_reading_tscs(void);
void test_reading_tscl(void);
void test_reading_tscv(void);

void
test_reading(const char *filename, const unsigned int shortids[], const double rxtimes[], const char *type)
{
	int i, r, n;
	int li=1;
	char extype[3];
	StatusBank db;
	FILE *file;
	MonitorDataMessage *msg;

	for(n = 0; shortids[n] > 0; n++);	/* count number of statuses in the goal */
	fprintf(stderr, "instrument IDs in %s should be %d ...", filename, n);
	assert(StatusBank_init(&db));
	r = 0;
	assert(file = fopen(filename, "r"));
	while(!feof(file))
		{
			int rr;
			assert(msg = MonitorDataMessage_alloc());
			if ((rr = fread_MonitorDataMessage(msg, file, block_tws4, 0)) != tsc_success)
				{
					switch(rr)
						{
							case tsc_error_stream:
								if (!feof(file))
									{
										perror(filename);
										assert(0);
									}
								break;
							case tsc_error_malloc:
								perror(filename);
								assert(0);
								break;
							case tsc_error_malformed:
								fprintf(stderr, "%s:malformed header\n", filename);
								goto fc;
						}
				}
			else
				{
					int rrr;
					rrr = MonitorDataMessage_scan(msg, &db, scan_all);
					assert(rrr >= 0);
					r += rrr;
				}
		}
	fc:
	fclose(file);
	fprintf(stderr, " got %d.\n", r);
	extype[0] = type[0];
	extype[1] = '\0';
	extype[2] = '\0';
	for(i = 0; i < n; i++)
		{
			const InstrumentStatus *inst;
			double rxtime;
			if ('L' == extype[0] && 0x00B2 == shortids[i])
				{
					li++;
					snprintf(&(extype[1]), 2, "%d", li);
				}
			else
				{
					if ('L' == extype[0] && li >= 3)
						{
							extype[1] = '\0';
						}
				}
			/*fprintf(stderr, "inst %04X%s:", shortids[i], extype);*/
			assert(inst = StatusBank_get_inst(&db, extype, shortids[i]));
			assert(InstrumentStatus_rxtime_d(inst, &rxtime));
			/*fprintf(stderr, "rxtime %.1f\n", rxtime);*/
			if (rxtimes) assert(rxtimes[i] == rxtime);
		}
	/* check for unknown IDs */
	for(i = 0; i < MELCO_TREE_LENGTH; i++)
		{
			if (db.data[i])
				{
					int j;
					for(j = 0; j < n; j++)
						{
							if (shortids[j] == instrument_status_header_shortid(db.data[i]->d))
								break;
						}
					if (j == n)
						{
							fprintf(stderr, "inst %04X in type %s in StatusBank slot %d not registered in test case!\n", instrument_status_header_shortid(db.data[i]->d), type, i);
							assert(0);
						}
				}
		}
	assert(n == r);
	StatusBank_discard_all_inst(&db);
}

void
test_reading_tscs(void)
{
	int ids[] = TSCSIDS;
	double rxtimes[] = TSCSRXT;
	test_reading(TSCSFILE, ids, rxtimes, "S");
}

void
test_reading_tscl(void)
{
	int ids[] = TSCLIDS;
	double rxtimes[] = TSCLRXT;
	test_reading(TSCLFILE, ids, rxtimes, "L");
}

void
test_reading_tscv(void)
{
	int ids[] = TSCVIDS;
	struct stat st;
	off_t size;
	assert(!stat(TSCVFILE, &st));
	size = st.st_size;
	test_reading(TSCVFILE, ids, NULL, "E");
}

int
main(int argc __attribute__ ((unused)), char *argv[])
{
	fprintf(stderr, "%s running\n", argv[0]);
	test_reading_tscs();
	test_reading_tscl();
	test_reading_tscv();
	return EXIT_SUCCESS;
}
