/* status-types.cpp: defnitions of type used in TSC status handling */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define __USE_XOPEN
#include <time.h>

#include <melco/status-types.h>
#include <melco/snprintf-compat.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define DOUBLE_FORMAT "%g"

#define HSEC2RAD	7.27220521664304e-05	/* PI/(12*3600) */
#define DSEC2RAD	4.84813681109536e-06	/* PI/(180*3600) */

/* private functions */
inline int _i_to_s_length(const long long x, const int exp);
inline void* _i_to_s(const long long x, const int exp, void *dst);
inline void* _str_to_s(const void *addr, const size_t length, void *dst);
inline int _str_to_s_length(const size_t length);
inline void _tv_to_d(const struct timeval *tv, double* dst);
inline void _tv_to_i(const struct timeval *tv, long long* dst);
inline void* _bcd_to_i(const void *addr, const size_t length, void* dst);
inline void* _bcd_to_s(const void *addr, const size_t length, void* dst);
inline int _bcd_to_s_length(const void *addr, const size_t length);
inline void* _binary_to_i(const void *addr, const size_t length, void* dst);
inline void* _binary_to_d(const void *addr, const size_t length, void* dst, const int exp);
inline void* _binary_to_s(const void *addr, const size_t length, void *dst, const int exp);
inline int _binary_to_s_length(const void *addr, const size_t length, void *dst, const int exp);

inline void* BCDn_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst);
inline int BCDn_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst);
inline void* BCDn_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst);
void* BCDn_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst);

#ifndef HAVE_TIMEGM
time_t
timegm(struct tm *tm)
{
	tzset();
	return mktime(tm) - timezone;
}
#endif

inline int
_i_to_s_length(const long long x, const int exp)
{
	int i, r;
	long long base;

	r = 0;

	/* minus sign */
	if (x < 0) r++;

	/* digits above decimal */
	base = x >= 0 ? x : -x;
	for(i = 0; i > exp; i--) base /= 10;
	if (base == 0)
		{
			r += 1;	/* zero */
		}
	else
		{
			for(; base > 0; base /= 10)	r++;	/* digits above decimal */
			if (exp > 0) r += exp;	/* zeros above decimal */
		}

	/* digits below decimal */
	if (exp < 0) r += -exp + 1;	/* digits below decimal plus decimal */
	
	return r + 1;	/* +1 for \0 */
}

inline void*
_i_to_s(const long long x, const int exp, void *dst)
{
	char digits[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};
	char *ptr;

	ptr = (char*) dst;

	/* minus sign */
	if (x < 0)
		{
			*ptr = '-';
			ptr++;
		}

	/* digits above decimal */
	{
		int i;
		long long base;
		base = x >= 0 ? x : -x;
		for(i = 0; i > exp; i--) base /= 10;

		if (base == 0)
			{
				*ptr = '0';
				ptr++;
			}
		else
			{
				/* digits above decimal */
				{
					long long dbase;
					char *dptr;

					/* point to last non-zero digit */
					dbase = base;
					dptr = ptr;
					for(; dbase > 0; dbase /= 10)	dptr++;
					ptr = dptr;
					dptr--;

					/* fill the digits */
					for(; base > 0; base /= 10)
						{
							*dptr = digits[base % 10];
							dptr--;
						}
				}
				/* zeros above decimal */
				for(i = 0; i < exp; i++)
					{
						*ptr = '0';
						ptr++;
					}
			}
	}

	/* digits below decimal */
	if (exp < 0)
		{
			int i;
			long long frac;
			char *dptr;

			*ptr = '.';
			ptr++;

			/* point to last digit */
			dptr = ptr - exp - 1;
			ptr = dptr + 1;

			/* fill the digits */
			frac = x >= 0 ? x : -x;
			for(i = 0; i > exp; i--)
				{
					*dptr = digits[frac % 10];
					frac /= 10;
					dptr--;
				}
		}
	
	*ptr = '\0';
	return dst;
}

inline void*
_str_to_s(const void *addr, const size_t length, void *dst)
{
  memcpy(dst, addr, length);
  *(((char*) dst) + length) = '\0';
  return dst;
}

int
_str_to_s_length(const size_t length)
{
  return length + 1;
}

inline void
_tv_to_d(const struct timeval *tv, double* dst)
{
	*dst = (double) tv->tv_sec + (double) tv->tv_usec * 1e-6;
}

inline void
_tv_to_i(const struct timeval *tv, long long* dst)
{
	*dst = (long long) tv->tv_sec;
	if (tv->tv_usec >= 500000) *dst += 1;
}

inline void*
_bcd_to_s(const void *addr, const size_t length, void* dst)
{
	void *r;
	long long x;
	r = _bcd_to_i(addr, length, &x);
	if (r)
		_i_to_s(x, 0, dst);
	else
		to_hex(addr, 0, length, dst);
	return r;
}

int
_bcd_to_s_length(const void *addr, const size_t length)
{
	void *r;
	long long x;
	r = _bcd_to_i(addr, length, &x);
	return r ? _i_to_s_length(x, 10) : to_hex_length(addr, 0, length, NULL);
}

/* convert `BCD' to long long */
/*
  tsc_extract.cpp and tsc_bcd*.cpp shows that:
   - first 4bit is only used for sign: 0x8 means minus, otherwise plus
   - for BCD6, value has to bemultiplied with 1e-8

   Most significant digit of BCD seems to be stored in the biggest
   address.

  returns NULL if data is invalid as a BCD
*/
inline void*
_bcd_to_i(const void *addr, const size_t length, void* dst)
{
	size_t i;
	long long *idst;
	void *r;
	unsigned char x;
	r = dst;
	idst = (long long*) dst;
	*idst = 0;
	for(i = 0; r && i < length; i++)
		{
			if(length < 6 || i > 0)
				{
					*idst *= 10;
					x = (((unsigned char*) addr)[i] >> 4);
					if (x < 10)
						*idst += x;
					else
						r = NULL;
				}
			*idst *= 10;
			x = (((unsigned char*) addr)[i] & 0x0F);
			if (x <10)
				*idst += x;
			else
				r = NULL;
		}
	if (length >= 6 && ((unsigned char *) addr)[0] & 0x80) *idst *= -1;
	return r;
}

inline void*
_binary_to_i(const void *addr, const size_t length, void* dst)
{
	if (*(char*) addr >= 0) /* most significant byte from TSC */
		*((long long*) dst) = 0;
	else
		*((long long*) dst) = -1;
	if (length > sizeof(long long)) return NULL;
#ifndef WORDS_BIGENDIAN
	{
		size_t i;
		for(i = 0; i < length; i++)
			memcpy((char*) dst + length - i - 1, (char*) addr + i, 1);
	}
#else
	memcpy((char*) dst + sizeof(long long) - length, addr, length);
#endif
	return dst;
}

inline void*
_binary_to_d(const void *addr, const size_t length, void* dst, const int exp)
{
	long long buf, scale;
	int i;
	if (!_binary_to_i(addr, length, &buf)) return NULL;
	scale = 1;
	for(i = 0; i < (exp > 0 ? exp : -exp); i++) scale *= 10;
	*((double*)dst) = (double) buf;
	if (exp > 0)
		*((double*)dst) *= scale;
	else
		*((double*)dst) /= scale;
	return dst;
}

inline void*
_binary_to_s(const void *addr, const size_t length, void *dst, const int exp)
{
	long long x;
	_binary_to_i(addr, length, &x);
	return _i_to_s(x, exp, dst);
}

inline int
_binary_to_s_length(const void *addr, const size_t length, void *dst __attribute__ ((unused)), const int exp)
{
	long long x;
	_binary_to_i(addr, length, &x);
	return _i_to_s_length(x, exp);
}

/* public functions */
void*
str_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _str_to_s(addr, length, dst);
}

int
str_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return _str_to_s_length(length);
}

void*
ASC15_TIME_to_t(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	ASC15_TIME *data;
	struct timeval *tv;
	struct tm tm;
	char secbuf[sizeof(data->sep.sec)-1];
	char *res;

	data = (ASC15_TIME*) addr;
	tv = (struct timeval*) dst;

	memset(&tm, 0, sizeof(tm));
	res = strptime(data->buf, "%Y%m%d%H%M%S", &tm);
	if (!res) return NULL;	/* error in input time string */
	tv->tv_sec = timegm(&tm);

	strncpy(secbuf, data->sep.sec+2, sizeof(data->sep.sec)-2);
	secbuf[sizeof(data->sep.sec)-2] = '\0';
	tv->tv_usec = atoi(secbuf)*100000;	// 1e5

	return dst;
}

void*
ASC15_TIME_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	struct timeval tv;
	if (!ASC15_TIME_to_t(addr, mask, length, &tv)) return NULL;
	_tv_to_d(&tv, (double*) dst);
	return dst;
}

void*
ASC15_TIME_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	struct timeval tv;
	if (!ASC15_TIME_to_t(addr, mask, length, &tv)) return NULL;
	_tv_to_i(&tv, (long long*) dst);
	return dst;
}

void*
ASC15_TIME_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _str_to_s(addr, length, dst);
}

int
ASC15_TIME_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return _str_to_s_length(length);
}

void*
ASC7_TIME_to_t(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	ASC7_TIME *data;
	struct timeval *tv;
	struct tm tm;
	char secbuf[sizeof(data->sep.sec)-1];

	data = (ASC7_TIME*) addr;
	tv = (struct timeval*) dst;

	memset(&tm, 0, sizeof(tm));
	strptime(data->buf, "%H%M%S", &tm);
	tv->tv_sec = timegm(&tm);

	strncpy(secbuf, data->sep.sec+2, sizeof(data->sep.sec)-2);
	secbuf[sizeof(data->sep.sec)-2] = '\0';
	tv->tv_usec = atoi(secbuf)*100000;	// 1e5

	return dst;
}

void*
ASC7_TIME_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	struct timeval tv;
	if (!ASC7_TIME_to_t(addr, mask, length, &tv)) return NULL;
	_tv_to_d(&tv, (double*) dst);
	return dst;
}

void*
ASC7_TIME_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	struct timeval tv;
	if (!ASC7_TIME_to_t(addr, mask, length, &tv)) return NULL;
	_tv_to_i(&tv, (long long*) dst);
	return dst;
}

void*
ASC7_TIME_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _str_to_s(addr, length, dst);
}

int
ASC7_TIME_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return _str_to_s_length(length);
}

void*
BCDn_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _bcd_to_s(addr, length, dst);
}

int
BCDn_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return _bcd_to_s_length(addr, length);
}

void*
BCDn_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _bcd_to_i(addr, length, dst);
}

void*
BCDn_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	void *r;
	long long buf;
	r = _bcd_to_i(addr, length, &buf);
	if (r)
		{
			*((double *) dst) = (double) buf;
			return dst;
		}
	else
		{
			*((double *) dst) = 0;
			return NULL;
		}
}

void*
BCD1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_s(addr, mask, length, dst);
}

int
BCD1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return BCDn_to_s_length(addr, mask, length, dst);
}

void*
BCD1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_i(addr, mask, length, dst);
}

void*
BCD1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_d(addr, mask, length, dst);
}

void*
BCD2_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_s(addr, mask, length, dst);
}

int
BCD2_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return BCDn_to_s_length(addr, mask, length, dst);
}

void*
BCD2_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_i(addr, mask, length, dst);
}

void*
BCD2_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_d(addr, mask, length, dst);
}

void*
BCD6_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	void *r;
	long long buf;
	r = _bcd_to_i(addr, length, &buf);
	if (r)
		{
			*((double *) dst) = (double) buf * 1e-8;
		}
	else
		{
			*((double *) dst) = 0;
		}
	return r;
}

void*
BCD6_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return BCDn_to_i(addr, mask, length, dst);
}

void*
BCD6_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	void *r;
	long long x;
	r = _bcd_to_i(addr, length, &x);
	if (r)
		return _i_to_s(x, -8, dst);
	else
		return _bcd_to_s(addr, length, dst);
}

int
BCD6_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	void *r;
	long long x;
	r = _bcd_to_i(addr, length, &x);
	if (r)
		return _i_to_s_length(x, -8);
	else
		return _bcd_to_s_length(addr, length);
}

void*
BCD6_DEC_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	void *r;
	long long h, m, s;
	r = _bcd_to_i((char*) addr + 1, 1, &h);	// DD
	if (r) r = _bcd_to_i((char*) addr + 2, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 3, 3, &s);	// SS.SSSS
	if  (r)
		{
			*((double*) dst) = ((double)(h*3600) + (double)(m*60) + ((double)s*1e-6)) * DSEC2RAD;
			if (*(char*)addr & 0x80) *((double*) dst) *= -1;
			return dst;
		}
	else
		{
			*((double*) dst) = 0;
			return NULL;
		}
}

void*
BCD6_DEC_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	void *r;
	long long h, m, s, si, sl;
	r = _bcd_to_i((char*) addr + 1, 1, &h);	// DD
	if (r) r = _bcd_to_i((char*) addr + 2, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 3, 3, &s);	// SS.SSSS
	if (r)
		{
			si = s / 10000;
			sl = s % 10000;
			snprintf(dst, BCD6_DEC_to_s_length(addr, mask, length, dst), "%c%02lld:%02lld:%02lld.%04lld", ((*(char*)addr & 0x80) ? '-' : '+'), h, m, si, sl);
			return dst;
		}
	else
		{
			to_hex(addr, mask, length, dst);
			return NULL;
		}
}

int
BCD6_DEC_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	void *r;
	long long h, m, s;
	r = _bcd_to_i((char*) addr + 1, 1, &h);	// DD
	if (r) r = _bcd_to_i((char*) addr + 2, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 3, 3, &s);	// SS.SSSS
	if (r)
		return sizeof("+DD:MM:SS.SSSS");
	else
		return to_hex_length(addr, mask, length, dst);
}

void*
BCD6_Epoch_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	void *r;
	long long buf;
	r = _bcd_to_i(addr, length, &buf);
	if (r)
		*((double*)dst) = buf*1e-6;
	else
		*((double*)dst) = 0;
	return r ? dst : NULL;
}

void*
BCD6_Epoch_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	void *r;
	int buflen;
	double d;
	buflen = BCD6_Epoch_to_s_length(addr, mask, length, dst);
	r = BCD6_Epoch_to_d(addr, mask, length, &d);
	if (r)
		{
			if (buflen - 1 != snprintf((char *) dst, buflen, DOUBLE_FORMAT, d)) return NULL;
			return dst;
		} 
	else
		{
			return to_hex(addr, mask, length, dst);
		}
}

int
BCD6_Epoch_to_s_length(const void *addr, const unsigned char mask, const size_t length, void *dst __attribute__ ((unused)))
{
	void *r;
	double d;
	r = BCD6_Epoch_to_d(addr, mask, length, &d);
	if (r)
		return snprintf_length(DOUBLE_FORMAT, d) + 1;
	else
		return to_hex_length(addr, mask, length, dst);
}

void*
BCD6_RA_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	void *r;
	char buf;
	long long h, m, s;
	buf = *((char*) addr) & 0x7F;
	r = _bcd_to_i(&buf, 1, &h);	// HH
	if (r) r = _bcd_to_i((char*) addr + 1, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 2, 4, &s);	// SS.SSSSSS
	if (r)
		{
			*((double*) dst) = ((double)(h*3600) + (double)(m*60) + ((double)s*1e-6)) * HSEC2RAD;
			if (*(char*)addr & 0x80) *((double*) dst) *= -1;
			return dst;
		}
	else
		{
			*((double*) dst) = 0;
			return NULL;
		}
}

void*
BCD6_RA_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	void *r;
	char buf;
	long long h, m, s, si, sl;
	buf = *((char*) addr) & 0x7F;
	r = _bcd_to_i(&buf, 1, &h);	// HH
	if (r) r = _bcd_to_i((char*) addr + 1, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 2, 4, &s);	// SS.SSSSSS
	if (r)
		{
			si = s / 1000000;
			sl = s % 1000000;
			snprintf(dst, BCD6_RA_to_s_length(addr, mask, length, dst) + 1, "%c%02lld:%02lld:%02lld.%06lld", ((*(char*)addr & 0x80) ? '-' : '+'), h, m, si, sl);
			return dst;
		}
	else
		{
			to_hex(addr, mask, length, dst);
			return NULL;
		}
}

int
BCD6_RA_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	void *r;
	char buf;
	long long h, m, s;
	buf = *((char*) addr) & 0x7F;
	r = _bcd_to_i(&buf, 1, &h);	// HH
	if (r) r = _bcd_to_i((char*) addr + 1, 1, &m);	// MM
	if (r) r = _bcd_to_i((char*) addr + 2, 4, &s);	// SS.SSSSSS
	if (r)
		return sizeof("+HH:MM:SS.SSSSSS");
	else
		return to_hex_length(addr, mask, length, dst);
}

void*
BINARY1_0p01_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _binary_to_d(addr, length, dst, -2);
}

void*
BINARY1_0p01_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _binary_to_i(addr, length, dst);
}

void*
BINARY1_0p01_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _binary_to_s(addr, length, dst, -2);
}

int
BINARY1_0p01_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	return _binary_to_s_length(addr, length, dst, -2);
}

void*
BINARY1_1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, 0);
}

void*
BINARY1_1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY1_1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, 0);
}

int
BINARY1_1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, 0);
}

void*
BINARY2_0p01_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -2);
}

void*
BINARY2_0p01_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY2_0p01_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -2);
}

int
BINARY2_0p01_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -2);
}

void*
BINARY2_0p1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -1);
}

void*
BINARY2_0p1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY2_0p1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -1);
}
int
BINARY2_0p1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -1);
}

void*
BINARY2_1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, 0);
}

void*
BINARY2_1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY2_1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, 0);
}

int
BINARY2_1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, 0);
}

void*
BINARY4_0p000000000001_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -12);
}

void*
BINARY4_0p000000000001_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p000000000001_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -12);
}

int
BINARY4_0p000000000001_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -12);
}

void*
BINARY4_0p000001_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -6);
}

void*
BINARY4_0p000001_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p000001_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -6);
}

int
BINARY4_0p000001_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -6);
}

void*
BINARY4_0p00001_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -5);
}

void*
BINARY4_0p00001_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p00001_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -5);
}

int
BINARY4_0p00001_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -5);
}

void*
BINARY4_0p001_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -3);
}

void*
BINARY4_0p001_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p001_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -3);
}

int
BINARY4_0p001_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -3);
}

void*
BINARY4_0p01_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -2);
}

void*
BINARY4_0p01_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p01_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -2);
}

int
BINARY4_0p01_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -2);
}

void*
BINARY4_0p1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, -1);
}

void*
BINARY4_0p1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_0p1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, -1);
}

int
BINARY4_0p1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, -1);
}

void*
BINARY4_100_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, 2);
}

void*
BINARY4_100_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_100_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, 2);
}

int
BINARY4_100_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, 2);
}

void*
BINARY4_1_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_d(addr, length, dst, 0);
}

void*
BINARY4_1_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_i(addr, length, dst);
}

void*
BINARY4_1_to_s(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s(addr, length, dst, 0);
}

int
BINARY4_1_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
  return _binary_to_s_length(addr, length, dst, 0);
}

void*
REAL8_to_d(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	if (length != sizeof(double)) return NULL;
#ifndef WORDS_BIGENDIAN
	memcpy(dst, addr, length);
#else
	{
		size_t i;
		for(i = 0; i < length; i++)
			memcpy((char*) dst + length - i - 1, (char*) addr + i, 1);
	}
#endif
	return dst;
}

void*
REAL8_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	int buflen;
	double d;
	buflen = REAL8_to_s_length(addr, mask, length, &d);
	REAL8_to_d(addr, mask, length, &d);
	if (buflen - 1 != snprintf((char *) dst, buflen, DOUBLE_FORMAT, d)) return NULL;
	return dst;
}

int
REAL8_to_s_length(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	double d;
	REAL8_to_d(addr, 0, length, &d);
	return snprintf_length(DOUBLE_FORMAT, d) + 1;
}

void*
TIME3_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long buf;
	TIME3_to_i(addr, mask, length, &buf);
	*((double*) dst) = (double) buf;
	return dst;
}

void*
TIME3_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	TIME3* self;
	self = (TIME3*) addr;
	*((long long *)dst) = self->sep.hh*3600 + self->sep.mm*60 + self->sep.ss;
	return dst;
}

void*
TIME3_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	int buflen;
	TIME3* self;
	self = (TIME3*) addr;
	buflen = TIME3_to_s_length(addr, mask, length, NULL);
	if (buflen - 1 != snprintf((char*) dst, buflen, "%02u:%02u:%02u", self->sep.hh, self->sep.mm, self->sep.ss)) return NULL;
	return dst;
}

int
TIME3_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	return sizeof("HH:MM:SS");
}

void*
TIME3_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long sec;
	struct timeval *tv;
	tv = (struct timeval *) dst;
	TIME3_to_i(addr, mask, length, &sec);
	tv->tv_sec = sec;
	tv->tv_usec = 0;
	return dst;
}

void*
TIME4_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long buf;
	TIME4_to_i(addr, mask, length, &buf);
	*((double*) dst) = (double) buf;
	return dst;
}

void*
TIME4_to_i(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	TIME4* self;
	self = (TIME4*) addr;
	*((long long *)dst) = self->sep.day*3600*24 + self->sep.hh*3600 + self->sep.mm*60 + self->sep.ss;
	return dst;
}

void*
TIME4_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	int buflen;
	TIME4* self;
	self = (TIME4*) addr;
	buflen = TIME4_to_s_length(addr, mask, length, NULL);
	if (buflen - 1 != snprintf((char*) dst, buflen, "%02u-%02u:%02u:%02u", self->sep.day, self->sep.hh, self->sep.mm, self->sep.ss)) return NULL;
	return dst;
}

int
TIME4_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	return sizeof("DD-HH:MM:SS");
}

void*
TIME4_to_t(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long sec;
	struct timeval *tv;
	tv = (struct timeval *) dst;
	TIME4_to_i(addr, mask, length, &sec);
	tv->tv_sec = sec;
	tv->tv_usec = 0;
	return dst;
}

void*
TIME6_to_d(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long buf;
	TIME6_to_i(addr, mask, length, &buf);
	*((double*) dst) = (double) buf;
	return dst;
}

void*
TIME6_to_i(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	struct timeval tv;
	TIME6_to_t(addr, mask, length, &tv);
	*((long long*) dst) = tv.tv_sec;
	return dst;
}

void*
TIME6_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	int buflen;
	TIME6* self;
	self = (TIME6*) addr;
	buflen = TIME6_to_s_length(addr, mask, length, NULL);
	if (buflen - 1 != snprintf((char*) dst, buflen, "%04d/%02u/%02u %02u:%02u:%02u", self->sep.year + 2000, self->sep.month, self->sep.day, self->sep.hh, self->sep.mm, self->sep.ss)) return NULL;
	return dst;
}

int
TIME6_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	return sizeof("yyyy/mm/dd HH:MM:SS");
}

void*
TIME6_to_t(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst)
{
	struct tm tm;
	struct timeval *tv;
	TIME6* self;
	self = (TIME6*) addr;
	tv = (struct timeval *) dst;

	memset(&tm, 0, sizeof(tm));
	tm.tm_sec = self->sep.ss;
	tm.tm_min = self->sep.mm;
	tm.tm_hour = self->sep.hh;
	tm.tm_mday = self->sep.day;
	tm.tm_mon = self->sep.month - 1;
	tm.tm_year = self->sep.year + 100;
	tm.tm_wday = 0;
	tm.tm_yday = 0;
	tm.tm_isdst = 0;
	tv->tv_sec = timegm(&tm);
	tv->tv_usec = 0;
	return dst;
}

void*
TIME8_to_d(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	return TIME6_to_d(&(((TIME8*)addr)->sep.time6), mask, 6, dst);
}
void*
TIME8_to_i(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	return TIME6_to_i(&(((TIME8*)addr)->sep.time6), mask, 6, dst);
}

void*
TIME8_to_s(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	return TIME6_to_s(&(((TIME8*)addr)->sep.time6), mask, 6, dst);
}

int
TIME8_to_s_length(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	return TIME6_to_s_length(&(((TIME8*)addr)->sep.time6), mask, 6, dst);
}

void*
TIME8_to_t(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	return TIME6_to_t(&(((TIME8*)addr)->sep.time6), mask, 6, dst);
}

void*
bit_to_i(const void *addr, const unsigned char mask, const size_t length __attribute__ ((unused)), void *dst)
{
	if (*((unsigned char*) addr) & mask)
		*((long long *) dst) = 1L;
	else
		*((long long *) dst) = 0L;
	return dst;
}

void*
bit_to_s(const void *addr, const unsigned char mask, const size_t length, void *dst)
{
	long long v;
	if (!bit_to_i(addr, mask, length, &v)) return NULL;
	if (v) *((char *)dst) = '1'; else *((char *)dst) = '0';
	*((char *)dst + 1) = '\0';
	return dst;
}

int
bit_to_s_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length __attribute__ ((unused)), void *dst __attribute__ ((unused)))
{
	return 2;
}

void*
to_hex(const void *addr, const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst)
{
	size_t i;
	unsigned char *src;
	char *ptr;

	src = (unsigned char *) addr;
	ptr = (char *) dst;

	if (length > 0)
		{
			sprintf(ptr, "%02X", *src);
			ptr += 2;
			src++;
		}
	for(i = 1; i < length; i++)
		{
			sprintf(ptr, "-%02X", *src);
			ptr += 3;
			src++;
		}
	*ptr = '\0';

	return dst;
}

int
to_hex_length(const void *addr __attribute__ ((unused)), const unsigned char mask __attribute__ ((unused)), const size_t length, void *dst __attribute__ ((unused)))
{
	return (length > 0) ? 3*length : 1;	/* "XX-"*(length-1) + "XX\0" */
}

