/* status-bank.c: Instrument Status Registry */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <melco/tsc-message.h>
#include <melco/tree-hash.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

char*
treekey(char *seven_byte_buf, const char *type, const unsigned int shortid)
{
	if (*type != 'L' || shortid == 0x00B2)
		{
			snprintf(seven_byte_buf, MELCO_TREE_MAX_WORD_LENGTH+1, "%04X%s", shortid, type);
		}
	else
		{
			snprintf(seven_byte_buf, MELCO_TREE_MAX_WORD_LENGTH+1, "%04X%c", shortid, *type);
		}
	return seven_byte_buf;
}

char*
treekey_str(char *seven_byte_buf, const char *type, const char *shortid)
{
	memcpy(seven_byte_buf, shortid, 4);
	seven_byte_buf[4] = type[0];
	if (type[0] == 'L' && memcmp(shortid, "00B2", 4) == 0)
		{
			seven_byte_buf[5] = type[1];
			seven_byte_buf[6] = '\0';
		}
	else
		{
			seven_byte_buf[5] = '\0';
		}
	return seven_byte_buf;
}

StatusBank *
StatusBank_init(StatusBank *self)
{
	int i;
	for(i = 0; i < MELCO_TREE_LENGTH; i++) self->data[i] = NULL;
	return self;
}

InstrumentStatus *
StatusBank_register_inst(StatusBank *self, InstrumentStatus *status)
{
	char key[MELCO_TREE_MAX_WORD_LENGTH+1];
	struct melco_tree_s *tree_info;
	treekey(key, status->type, status->shortid);
	tree_info = melco_tree_lookup(key, strlen(key));
	if (!tree_info) return NULL;
	self->data[tree_info->index] = status;
	InstrumentStatus_incref(status);
	return status;
}

InstrumentStatus *
StatusBank_get_inst(StatusBank *self, const char *type, const unsigned int shortid)
{
	InstrumentStatus *status;
	char key[MELCO_TREE_MAX_WORD_LENGTH+1];
	struct melco_tree_s *tree_info;
	treekey(key, type, shortid);
	tree_info = melco_tree_lookup(key, strlen(key));
	if (!tree_info) return NULL;
	status = self->data[tree_info->index];
	if (status) InstrumentStatus_incref(status);
	return status;
}

InstrumentStatus *
StatusBank_discard_inst(StatusBank *self, const char *type, const unsigned int shortid)
{
	InstrumentStatus *status;
	char key[MELCO_TREE_MAX_WORD_LENGTH+1];
	struct melco_tree_s *tree_info;
	treekey(key, type, shortid);
	tree_info = melco_tree_lookup(key, strlen(key));
	if (!tree_info) return NULL;
	status = self->data[tree_info->index];
	return InstrumentStatus_decref(status);
}

int
StatusBank_discard_all_inst(StatusBank *self)
{
	int i, c;
	for(i = 0, c= 0; i < MELCO_TREE_LENGTH; i++)
		{
			if (self->data[i])
				{
					InstrumentStatus_decref(self->data[i]);
					self->data[i] = NULL;
					c++;
				}
		}
	return c;
}
