/* statusentry.c: tscstatus.statusentry */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

 */

#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#include <Python.h>

#include <tscstatus.h>
#include <melco/tsc-message.h>
#include <melco/status-types.h>

void
P_StatusEntry_dealloc(P_StatusEntry *self)
{
	if (self->cdata) StatusEntry_decref(self->cdata);
	self->ob_type->tp_free((PyObject*)self);
}

PyObject*
P_StatusEntry_new(PyTypeObject *type, PyObject *args __attribute__ ((unused)))
{
	P_StatusEntry *self;

	self = (P_StatusEntry *)type->tp_alloc(type, 0);
	if (!self) return (PyObject *)self;

	self->cdata = NULL;
	return (PyObject *)self;
}

PyObject *
P_StatusEntry_info(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	PyObject *r;

	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}

	r = (PyObject*) P_EntryInfo_new(&P_EntryInfoType, Py_None);
	if (!r)
		{
			PyErr_NoMemory();
			return NULL;
		}
	P_EntryInfo_set((P_EntryInfo*) r, self->cdata->info);
	Py_XINCREF(r);

	return r;
}

PyObject *
P_StatusEntry_float(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	double r;
	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}
	if (!StatusEntry_to_d(self->cdata, &r))
		{
			PyErr_Format(PyExc_RuntimeError, "cannot convert \"%s\" into a float", self->cdata->info->name);
			return NULL;
		}
	return PyFloat_FromDouble(r);
}

PyObject *
P_StatusEntry_long(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	long long r;
	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}
	if (!StatusEntry_to_i(self->cdata, &r))
		{
			PyErr_Format(PyExc_RuntimeError, "cannot convert \"%s\" into an integer", self->cdata->info->name);
			return NULL;
		}
	return PyLong_FromLongLong(r);
}

PyObject *
P_StatusEntry_str(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	int l;
	char *s;
	PyObject* r;
	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}
	l = StatusEntry_to_s_length(self->cdata);
	if (!(s = (char*) malloc(l + 1)))
		{
			PyErr_NoMemory();
			return NULL;
		}
	if (!StatusEntry_to_s(self->cdata, s))
		{
			PyErr_Format(PyExc_RuntimeError, "cannot convert \"%s\" into a string", self->cdata->info->name);
			free(s);
			return NULL;
		}
	r = PyString_FromString(s);
	free(s);
	return r;
}

PyObject *
P_StatusEntry_time(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	struct timeval tv;
	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}
	if (!StatusEntry_to_tv(self->cdata, &tv))
		{
			PyErr_Format(PyExc_RuntimeError, "cannot convert \"%s\" into a time", self->cdata->info->name);
			return NULL;
		}
	return PyFloat_FromDouble((double) tv.tv_sec + (double) tv.tv_usec * 1e-6);
}

PyObject *
P_StatusEntry_val(P_StatusEntry *self, PyObject *args)
{
	const struct melco_status_s* info;

	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}

	info = self->cdata->info;

	if (info->mask != 0)	/* bit field */
		{
			long long r;
			StatusEntry_to_i(self->cdata, &r);
			if (r)
				{
					Py_XINCREF(Py_True);
					return Py_True;
				}
			else
				{
					Py_XINCREF(Py_False);
					return Py_False;
				}
		}
	if (info->to_t)	/* time */
		{
			return P_StatusEntry_time(self, args);
		}
	if (info->to_i)	/* integer */
		{
			return P_StatusEntry_long(self, args);
		}
	if (info->to_d)	/* float */
		{
			return P_StatusEntry_float(self, args);
		}
	if (info->to_s)	/* string */
		{
			return P_StatusEntry_str(self, args);
		}

	PyErr_Format(PyExc_RuntimeError, "no appropriate type conversion for \"%s\"", info->name);
	return NULL;
}

PyObject *
P_StatusEntry_rxtime(P_StatusEntry *self, PyObject *args __attribute__ ((unused)))
{
	struct timeval tv;
	if (!self->cdata)
		{
			PyErr_Format(PyExc_RuntimeError, "status entry is not set");
			return NULL;
		}
	if (!StatusEntry_rxtime_tv(self->cdata, &tv))
		{
			PyErr_Format(PyExc_RuntimeError, "cannot get rx time for \"%s\"", self->cdata->info->name);
			return NULL;
		}
	return PyFloat_FromDouble((double) tv.tv_sec + (double) tv.tv_usec * 1e-6);
}

void
P_StatusEntry_set(P_StatusEntry* self, StatusEntry *data)
{
	if (self->cdata) StatusEntry_decref(self->cdata);
	self->cdata = data;
	StatusEntry_incref(self->cdata);
}

PyMemberDef P_StatusEntry_members[] = {
	{NULL, 0, 0, 0, NULL}	/* Sentinel */
};

PyMethodDef P_StatusEntry_methods[] = {
	{"info", (PyCFunction)P_StatusEntry_info, METH_NOARGS,
	 "returns the entryinfo"},
	{"float", (PyCFunction)P_StatusEntry_float, METH_NOARGS,
	 "returns the value as a float"},
	{"long", (PyCFunction)P_StatusEntry_long, METH_NOARGS,
	 "returns the value as a long"},
	{"str", (PyCFunction)P_StatusEntry_str, METH_NOARGS,
	 "returns the value as a string"},
	{"time", (PyCFunction)P_StatusEntry_time, METH_NOARGS,
	 "returns the time value as a float"},
	{"val", (PyCFunction)P_StatusEntry_val, METH_NOARGS,
	 "returns the value as an appropriate type"},
	{"rxtime", (PyCFunction)P_StatusEntry_rxtime, METH_NOARGS,
	 "returns the reception time as a float"},
	{NULL, NULL, 0, NULL}	/* Sentinel */
};

PyTypeObject P_StatusEntryType = {
	PyObject_HEAD_INIT(NULL)
	0,                                   /* ob_size */
	"tscstatus.statusentry",             /* tp_name */
	sizeof(P_StatusEntry),               /* tp_basicsize */
	0,                                   /* tp_itemsize */
	(destructor)P_StatusEntry_dealloc,   /* tp_dealloc */
	0,                                   /* tp_print */
	0,                                   /* tp_getattr */
	0,                                   /* tp_setattr */
	0,                                   /* tp_compare */
	0,                                   /* tp_repr */
	0,                                   /* tp_as_number */
	0,                                   /* tp_as_sequence */
	0,                                   /* tp_as_mapping */
	0,                                   /* tp_hash */
	0,                                   /* tp_call */
	(reprfunc) P_StatusEntry_str,        /* tp_str */
	0,                                   /* tp_getattro */
	0,                                   /* tp_setattro */
	0,                                   /* tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,
	                                     /* tp_flags */
	"Status Entry",                      /* tp_doc */
	0,                                   /* tp_traverse */
	0,                                   /* tp_clear */
	0,                                   /* tp_richcompare */
	0,                                   /* tp_weaklistoffset */
	0,                                   /* tp_iter */
	0,                                   /* tp_iternext */
	P_StatusEntry_methods,               /* tp_methods */
	P_StatusEntry_members,               /* tp_members */
	0,                                   /* tp_getset */
	0,                                   /* tp_base */
	0,                                   /* tp_dict */
	0,                                   /* tp_descr_get */
	0,                                   /* tp_descr_set */
	0,                                   /* tp_dictoffset */
	0,                                   /* tp_init */
	0,                                   /* tp_alloc */
	(newfunc)P_StatusEntry_new,          /* tp_new */
	0,                                   /* tp_free */
	0,                                   /* tp_is_gc */
	0,                                   /* tp_bases */
	0,                                   /* tp_mro */
	0,                                   /* tp_cache */
	0,                                   /* tp_subclasses */
	0,                                   /* tp_weaklist */
	0,                                   /* tp_del */
};

