#
# a python module to handle status data documentations from Mitsubishi
#
# Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>
#
# Permission is granted for use, copying, modification, distribution,
# and distribution of modified versions of this work under the terms of
# GPL version 2 or later.
#
# vim: set ts=2 sw=2 ai noet:
#

import csv
import re
import string
import os

# shortid:{'name':part of C++ class name, type:total bytes}
# source: test5.pl by Morino
name_and_total = {
	# Could not find a point why we use special short ID for this.
	# Seems to be better to use PMA_PMFXS, PMA_PMFXS2, and PMA_PMFXS3.
	#'0858/00859/00860':{'name': 'PMFXS',               'E':4   },
	'00A1':{'name':'TSC',             'S':150, 'L':260, 'E':1557},
	'00B1':{'name':'MLP1',            'S':118, 'L':134, 'E':159 },
	'0001':{'name':'DRDR',            'S':18,  'L':84,  'E':49  },
	'0002':{'name':'MTDR',            'S':60,  'L':264, 'E':240 },
	'0004':{'name':'FRAD',            'S':13,  'L':108, 'E':42  },
	'0006':{'name':'AG',                       'L':86,  'E':120 },
	'0007':{'name':'SV',                       'L':64,  'E':162 },
	'0030':{'name':'THRM',                     'L':222, 'E':78  },
	'000D':{'name':'FPCI',                     'L':120, 'E':30  },
	'0027':{'name':'BLCU',                     'L':72,  'E':54  },
	'0008':{'name':'AO_OBE',                   'L':26,  'E':10  },
	#'0051':{'name':'AO_OBE2',                 'L':26,  'E':10  },
	'0051':{'name':'OBE',                      'L':26,  'E':10  },
	# could not find a CSV file for 0051 nor in S0501902A00
	'002E':{'name':'CLOCK',                    'L':10,  'E':1   },
	# devide into L2 and L3
	# http://siroan.naoj.org/Tsc/FY2003_TSC_status.pdf
	'00B2-1':{'name':'MLP2_1',                 'L':1160,'E':1100},	# L2
	'00B2-2':{'name':'MLP2_2',                 'L':1100,        },	# L3
	'0009':{'name':'SH',                       'L':7,   'E':100 },
	'0003':{'name':'SMCU',                     'L':198, 'E':96  },
	'00B3':{'name':'MLP3',                     'L':192, 'E':74  },
	'0024':{'name':'CVCU',                              'E':96  },
	'0025':{'name':'TMCU',                              'E':36  },
	'002A':{'name':'DOME_TEMP',                'L':480, 'E':160 },
	'002B':{'name':'DOME_CT2',                 'L':192, 'E':64  },
	'002C':{'name':'TLSCP_TEMP',               'L':540, 'E':180 },
	'002D':{'name':'TLSCP_CT2',                'L':132, 'E':44  },
	'0029':{'name':'HSBC',                     'L':18,  'E':6   },
	'000C':{'name':'WMON',                     'L':48,          },
	'000E':{'name':'CAL',                      'L':60,  'E':76  },
	'000A':{'name':'MIRROR_HT_EXH',                     'E':11  },
	'0031':{'name':'HT_EXH',                   'L':72,  'E':10  },
	'000B':{'name':'MCP1',                              'E':50  },
	'0010':{'name':'MCP2',                     'L':111, 'E':33  },
	'0858':{'name':'PMA_PMFXS1',                        'E':2   },
	'0859':{'name':'PMA_PMFXS2',                        'E':2   },
	'0860':{'name':'PMA_PMFXS3',                        'E':2   },
	'0032':{'name':'BOLT',                     'L':120, 'E':33  },
	'0033':{'name':'SPU4',                              'E':53  },
	'0034':{'name':'SPU5',                              'E':23  },
	'0035':{'name':'SPU6',                              'E':22  },
	'0028':{'name':'TTCU',            'S':60,  'L':24,  'E':53  },
	'0036':{'name':'FRAD_PF',         'S':13,  'L':90,  'E':40  },
	'0037':{'name':'ASCU_PF',                  'L':156, 'E':31  },
	'0061':{'name':'DOME_FLAT',                'L':192, 'E':27  },
	'0040':{'name':'SH_TEST',                           'E':1000},
	'0038':{'name':'CRYOGEN_HEAT_EXH',         'L':36,  'E':9   },
	#'0039':{'name':'Redundant Exh',           'L':78,  'E':25  },
	'0039':{'name':'PUMP_D',                   'L':78,  'E':25  },
	'003A':{'name':'CIAX',                     'L':132, 'E':68  },
	'003B':{'name':'OBCP',                     'L':236, 'E':192 },
	'003C':{'name':'HYDRST EXH',                        'E':35  },
	# total bytes from FY2003_TSC_Status.pdf
	# updated with S0501902A00
}

# a status field
class TelStatusEntry:
	# name (string) : col[1] (string)
	# bits (int) : col[2] (int)
	# format (string) : col[9] (string)
	# id (string) : col[12] (string)
	# type (string) : col[13] (string)
	# byteoffset (int) : col[14] (int)
	# bitoffset (int): col[15] (string)
	# descriptions (dictionary): {'ja':col[5] (string)}
	# basename (string): base name of csv file (without .csv)
	def __init__(self, name, bits, format, id, type, byteoffset, bitoffset, basename, descriptions):
		self.name = name	# name as string: e.g. AG Exposure Time
		self.bits = bits	# number of bits as integer
		self.format = format	# format as string: e.g. BINARY
		self.id = id	# id as string: e.g. 0006001
		self.type = type	# type as string: E, L, or S
		self.byteoffset = byteoffset	# byte offset as integer
		self.basename = basename	# name of source csv file
		if self.basename.find('MLP2_1') >= 0:
			self.shortid = '00B2-1'
		elif self.basename.find('MLP2_2') >= 0:
			self.shortid = '00B2-2'
		else:
			self.shortid = self.id[0:4]
		self.descriptions = descriptions	# descriptions

		# bitoffset might have unit
		self.unit = '1'
		if self.format == 'BINARY':
			self.bitoffset = 0
			self.unit = bitoffset
		elif self.format == 'TIME':
			self.bitoffset = 0
		elif self.format == 'REAL':
			self.bitoffset = 0
		elif self.format == 'BCD':
			self.bitoffset = 0 
			# might be a problem on "MA Current El Angle" in TSCMON.xls
		else:
			self.bitoffset = int(bitoffset)	# bit offset as integer

		# check types
		types = {
			'name':str,
			'bits':int,
			'format':str,
			'id':str,
			'type':str,
			'byteoffset':int,
			'bitoffset':int,
			'descriptions':dict,
			'unit':str,
		}
		for attr in types.keys():
			if eval( 'self.%s.__class__' % attr ) != types[attr]:
				raise TypeError, '%s (id:%s): %s is not a %s' % (self.name, self.id, attr, types[attr])

	# number of bytes occupied with this entry
	def bytes(self):
		if self.bits >= 8:
			if self.bits % 8 != 0:
				raise ValueError, '%s (id:%s): bitfield of more than 1 byte' % (self.name, self.id)
			r = self.bits / 8
		else:
			r = 1 # bit field
		return r
	
	# bit field?
	def is_bitfield(self):
		return (self.bits < 8)

	# identity
	def same(self, other):
		if self.__class__ != other.__class__:
			return False
		if self.name != other.name:
			return False
		if self.bits != other.bits:
			return False
		if self.format != other.format:
			return False
		if self.id != other.id:
			return False
		return True


# a CSV file from Mitsubishi
class TelStatusCsv:
	def __init__(self, path):
		self.path = path
		self.basename = os.path.basename(self.path).split('.')[0]

	# Zernike polynominal terms
	zernike_terms = [
		'00', '11', '1m1', '20', '31', '3m1', '22', '2m2', '33', '3m3',
		'40', '42', '4m2', '44', '4m4', '51', '5m1', '53', '5m3', '55',
		'5m5', '60', '62', '6m2', '64', '6m4', '66', '6m6', '71', '7m1',
		'73', '7m3', '75', '7m5', '77', '7m7', '80', '82', '8m2', '84',
		'8m4', '86', '8m6', '88', '8m8', '91', '9m1', '93', '9m3', '95',
		'9m5', '97', '9m7', '99', '9m9', '100', '102', '10m2', '104', '10m4',
		'106', '10m6', '108', '10m8', '1010', '10m10', '', '', '', ''
	]

	# correction for typos in Mitsubishi CSV
	# input:output
	# col[0]	: entry number within the file
	# col[1]	: `name' of the status (string)
	# col[2]	: number of bits (int)
	# col[3]	: positive/negative logic
	# col[4]	: source interface
	# col[5]	: description in Japanese (string, encoding depends upon input)
	#	         Python >= 2.4 is needed to convert strings to UTF-8
	#	         with a standard Python library.
	# col[6]	: whether this is a source of alarm
	# col[7]	: whether this is a source of warning
	# col[8]	: whether this is a source of failure
	# col[9]	: format of numeric data (upper-cased string)
	# col[10] : exclusive signlals
	# col[11] : remarks
	# col[12] : status ID (zero-filled string)
	# col[13] : E:event driven/variable, L:long period, S:short period (string)
	# col[14] : byte offset (int)
	# col[15] : bit offset or decimal position (string)
	# returns a list of normalized and type-converted status entries

	def corrected(self, col):
		e = col[:]
		# number of columns
		if len(col) < 15:
			return []
		# correct O (Capital-O) to 0 (zero)
		e[12] = re.sub('O', '0', e[12]) # MCP1MON.csv
		e[14] = re.sub('O', '0', e[14]) # TLSCPCT2.csv and TLSCPENV.csv
		e[15] = re.sub('O', '0', e[15]) # DOMECT2.csv and DOMEENVI.csv
		# byte offset
		try:
			e[14] = int(e[14])
		except ValueError:	# we can ignore invalid entries
			return []
		# ID
		if len(e[12]) == 0:	# we can ignore entries with an empty ID
			return []
		# fill ID with zeros
		e[12] = e[12].zfill(7)
		# skipped numbers: "ketsu-ban" in Shift-JIS
		if e[1] == '\x8c\x87\x94\xd4':
			return []
		# for future use
		elif e[12] == '0006049':
			return []
		# some names have a trailing space
		e[1] = e[1].strip()
		# some decimal positions have a trailing space
		e[15] = e[15].strip()
		# number of bits
		e[2] = int(e[2])
		# decription
		# e[5] = some way to convert encoding
		# format
		e[9] = string.upper(e[9])

		# ID specfic typos and expansions
		idnum = string.atoi(e[12],16)
		# id:002B07F seems not to be a BCD but a bit field 
		if e[12] == '002B07F':
			e[9] = ''
		# id:000E02C in CALMON.csv has a typo
		elif e[12] == '000E02C':
			e[1] = 'Cal Source TurretD Holder3 Selected'
		# id:002A0FC in DOMEENVI.csv has a typo
		elif e[12] == '002A0FC':
			e[1] = 'Ns FL Rear V WVEL Data'
		# id:002A174 in DOMEENVI.csv has a typo
		elif e[12] == '002A174':
			e[1] = 'Top FL Rear V WVEL Data'
		# id:002C295-002C2A0 in TLSCPENV.csv have typos
		elif 0x002C295 <= idnum and idnum < 0x002C2A1:
			e[1] = re.sub('IR-B', 'Opt-B', e[1])
		# id:002D100 in TLSCPCT2.csv has a typo
		elif e[12] == '002D100':
			e[1] = 'CSCT(F-IR) CT2 OVER RNG'
		# id:00B218B has to be expanded to many of Zernike polynominals
		# source: MLP2-MLP2_1.h by Morino
		elif e[12] == '00B218B':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 64
			template[9] = 'REAL'
			for i in range(0,len(TelStatusCsv.zernike_terms)):
				if len(TelStatusCsv.zernike_terms[i]) > 0:
					template[1] = "%s %02d A%s" % (e[1],i,TelStatusCsv.zernike_terms[i])
				else:
					template[1] = "%s %02d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040001 has to be expanded to many of Zernike polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040001':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 16
			template[9] = 'BINARY'
			template[15] = '0.01'
			for i in range(1,66):
				template[1] = "%s %02d A%s" % (e[1],i,TelStatusCsv.zernike_terms[i])
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040004 has to be expanded to many polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040004':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 64
			template[9] = 'REAL'
			for i in range(1,33):
				template[1] = "%s q%02d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# id:0040015 has to be expanded to many polynominals
		# source: SH_TEST-SHTESTMON.h by Morino
		elif e[12] == '0040015':
			r = []
			offset = int(e[14])
			template = e[:]
			template[2] = 16
			template[9] = 'BINARY'
			template[15] = '0.01'
			for i in range(0,192):
				template[1] = "%s %03d" % (e[1],i)
				template[14] = offset
				r.append(template[:])
				offset += template[2]/8
			return r
		# entries in PMAMON.csv has to be expanded to those of #000-260
		# also for name_and_total
		# http://www1.naoj.org/telgroup/TSC_Status_html/sig/Pmamon.htm
		# http://www1.naoj.org/telgroup/TSC_Command.html/Individual/TWS/TWS_2_3.htm
		elif e[12][0:4] == '0101':
			r = []
			template = e[:]
			for n1 in range(1,9):
				for n2 in range(0,7):
					for n3 in range(0,10):
						numstr = "%d%d%d" % (n1, n2, n3)
						template[1] = "%s%d-%d%d%s" % (e[1][0:8], n1, n2, n3, e[1][12:])
						template[12] = e[12][0] + numstr + e[12][4:7]
						r.append(template[:])
						name_and_total[template[12][0:4]] = {'name':'PMA-'+numstr}
			return r
		return [e]

	# parse
	def parse(self):
		r = []
		for col in csv.reader(file(self.path)):
			if '#' == col[0][0]:
				continue
			for c in self.corrected(col):
				e = TelStatusEntry(c[1], c[2], c[9], c[12], c[13], c[14], c[15], self.basename, {'ja':c[5]})
				r.append(e)
		return r
