#
# attribute-unused.rb: mark anused paramters as unused
#
# usage:
#   gcc -c -Wunused source.c 2>&1 | ruby attribute-unused.rb
#
# Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>
#
# Permission is granted for use, copying, modification, distribution,
# and distribution of modified versions of this work under the terms of
# GPL version 2 or later.
# 

class UnusedPar
	attr_reader :file, :line, :name
	def initialize( file, line, name )
		@file, @line, @name = file, line.to_i, name
	end
	def self::parse( string )
		d = string.split( /:/ )
		if /unused parameter/ =~ d[3] then
			file = d[0]
			line = d[1]
			name = d[3].scan( /\'(.+)\'/ )[0][0]
			return self.new( file, line, name )
		else
			return nil
		end
	end
end

class UnusedPars < Array
	def files
		self.collect{ |p| p.file }.uniq
	end
	def pars( file )
		self.reject{ |p| p.file != file }
	end
	def lines
		self.collect{ |p| p.line }.sort.uniq
	end
end

class Source < String
	Mark = '__attribute__ ((unused))'
	Head = <<'_END'
/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif
_END
	# http://www.unixwiz.net/techtips/gnu-c-attributes.html
	def mark_unused( unusedpars )
		a = self.split( /\n/ )
		unusedpars.each do |par|
			a[par.line - 1].gsub!( /\b#{par.name}\b/, "#{par.name} #{Mark}" )
		end
		r = a.join( "\n" )
		r += "\n" if /\n\Z/ =~ self
		r.sub!( /^(\#ifdef HAVE_CONFIG_H)/, "#{Head}\n\\1" ) ||\
		r.sub!( /^(\#include)/, "#{Head}\n\\1" ) ||\
		r
	end
end

if __FILE__ ==  $0 then
	pars = UnusedPars.new
	ARGF.each do |l|
		par = UnusedPar.parse( l )
		pars << par if par
	end
	pars.files.each do |src|
		dst = src + '.unused-marked'
		puts "creating #{dst}"
		File.open(src) do |s|
			File.open(dst, 'w') do |d|
				d.print Source.new( s.read ).mark_unused( pars.pars( src ) )
			end
		end
	end
end
