/* message_header.c: handling of information in message header */
/*

See JNLT Requirement Specification for F-M Interface
  http://siroan.naoj.org/Tsc/Fujitsu-Mitsubishi-Interface-Specifications.pdf
for a detailed description.

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <melco/tsc-message.h>
#include <melco/status-types.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

char*
message_header_type(const message_header_s* ptr, char *buf_three_bytes)
{
	memcpy(buf_three_bytes, ptr->s.type, sizeof(ptr->s.type));
	buf_three_bytes[sizeof(ptr->s.type)] = '\0';
	return buf_three_bytes;
}

char*
message_header_dst(const message_header_s* ptr, char *buf_five_bytes)
{
	memcpy(buf_five_bytes, ptr->s.dst, sizeof(ptr->s.dst));
	buf_five_bytes[sizeof(ptr->s.dst)] = '\0';
	return buf_five_bytes;
}

char*
message_header_src(const message_header_s* ptr, char *buf_five_bytes)
{
	memcpy(buf_five_bytes, ptr->s.src, sizeof(ptr->s.src));
	buf_five_bytes[sizeof(ptr->s.src)] = '\0';
	return buf_five_bytes;
}

int
message_header_pid(const message_header_s* ptr)
{
	char buf[sizeof(ptr->s.pid)+1];
	memcpy(buf, ptr->s.pid, sizeof(ptr->s.pid));
	buf[sizeof(ptr->s.pid)] = '\0';
	return atoi(buf);
}

const ASC15_TIME*
message_header_txtime(const message_header_s* ptr)
{
	return &(ptr->s.txtime);
}

struct timeval*
message_header_txtime_tv(const message_header_s* ptr, struct timeval* dst)
{
	if (!(ASC15_TIME_to_t((void*) &(ptr->s.txtime), 0, 15, dst))) return NULL;
	return dst;
}

int
message_header_length(const message_header_s* ptr)
{
	char buf[sizeof(ptr->s.length)+1];
	memcpy(buf, ptr->s.length, sizeof(ptr->s.length));
	buf[sizeof(ptr->s.length)] = '\0';
	return atoi(buf);
}

char*
status_content_type_set(const char* type, char *buf_three_bytes)
{
	buf_three_bytes[0] = *type;
	if (buf_three_bytes[0] == 'L')
		{
			buf_three_bytes[1] = *(type + 1);
			buf_three_bytes[2] = '\0';
		}
	else
		{
			buf_three_bytes[1] = '\0';
		}
	return buf_three_bytes;
}

char*
message_header_status_content_type(const message_header_s* ptr, char *buf_three_bytes)
{
	return status_content_type_set((char *) ptr + sizeof(message_header_s), buf_three_bytes);
}

int
message_header_valid(const message_header_s* ptr)
{
	struct timeval tv;

	/* size shuold be more than zero */
	if (message_header_length(ptr) < 0) return 0;

	/* type should be MO */
	if (strncmp("MO", ptr->s.type, sizeof(ptr->s.type))) return 0;

	/* time stamp should be valid */
	if (!message_header_txtime_tv(ptr, &tv)) return 0;
	
	return 1;
}

const char*
message_header_set_type(message_header_s* ptr, const char *type)
{
	unsigned int i;
	for(i = 0; i < sizeof(ptr->s.type) && type[i] != '\0'; i++)
		{
			ptr->s.type[i] = type[i];
		}
	for(; i < sizeof(ptr->s.type); i++)
		{
			ptr->s.type[i] = '%';
		}
	return type;
}

const char*
message_header_set_dst(message_header_s* ptr, const char *dst)
{
	unsigned int i;
	for(i = 0; i < sizeof(ptr->s.dst) && dst[i] != '\0'; i++)
		{
			ptr->s.dst[i] = dst[i];
		}
	for(; i < sizeof(ptr->s.dst); i++)
		{
			ptr->s.dst[i] = '%';
		}
	return dst;
}

const char*
message_header_set_src(message_header_s* ptr, const char *src)
{
	unsigned int i;
	for(i = 0; i < sizeof(ptr->s.src) && src[i] != '\0'; i++)
		{
			ptr->s.src[i] = src[i];
		}
	for(; i < sizeof(ptr->s.src); i++)
		{
			ptr->s.src[i] = '%';
		}
	return src;
}

int
message_header_set_pid(message_header_s* ptr, const int pid)
{
	char buf[sizeof(ptr->s.pid) + 1];
	snprintf(buf, sizeof(buf), "%04d", pid);
	memcpy(ptr->s.pid, buf, sizeof(ptr->s.pid));
	return pid;
}

#define _PART_TIME_STR_LEN (sizeof("YYYYMMDDhhmmss") - 1)
const struct timeval*
message_header_set_txtime(message_header_s* ptr, const struct timeval *txtime)
{
	struct tm tm;
	char buf[sizeof(ptr->s.txtime) + 1];

	memset(&tm, 0, sizeof(tm));
#ifdef HAVE_GMTIME_R
	gmtime_r(&(txtime->tv_sec), &tm);
#else
	{
		struct tm *tmptr;
		tmptr = gmtime(&(txtime->tv_sec));
		tm = *tmptr;
	}
#endif
	strftime(buf, sizeof(buf), "%Y%m%d%H%M%S", &tm);
	snprintf(buf + _PART_TIME_STR_LEN, sizeof(buf) - _PART_TIME_STR_LEN, "%.0f", (double)(txtime->tv_usec) * 1e-5);

	memcpy(&(ptr->s.txtime), buf, sizeof(ptr->s.txtime));
	return txtime;
}

int
message_header_set_length(message_header_s* ptr, const int length)
{
	char buf[sizeof(ptr->s.length) + 1];
	snprintf(buf, sizeof(buf), "%04d", length);
	memcpy(&(ptr->s.length), buf, sizeof(ptr->s.length));
	return length;
}

