/* memory-manage.c: memory management for tsc-message */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <melco/tsc-message.h>
#include <melco/tree-hash.h>
#include <melco/status-hash.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

MonitorDataMessage *
MonitorDataMessage_alloc(void)
{
	MonitorDataMessage *self;
	self = (MonitorDataMessage *) malloc(sizeof(MonitorDataMessage));
	if (!self) return NULL;
	MonitorDataMessage_init(self);
	self->self_free = free;
	return self;
}

void
MonitorDataMessage_init(MonitorDataMessage *self)
{
	self->d = NULL;
	self->d_free = NULL;
	self->length = 0;
	*(self->type) = '\0';
	self->refcount = 0;
	self->self_free = NULL;
}

MonitorDataMessage *
MonitorDataMessage_incref(MonitorDataMessage *self)
{
	if (self) self->refcount++;
	return self;
}

MonitorDataMessage *
MonitorDataMessage_decref(MonitorDataMessage *self)
{
	if (self) self->refcount--;
	return MonitorDataMessage_free_if_noref(self);
}

MonitorDataMessage *
MonitorDataMessage_free_if_noref(MonitorDataMessage *self)
{
	if (self && self->refcount < 1)
		return MonitorDataMessage_free(self);
	else
		return self;
}

MonitorDataMessage *
MonitorDataMessage_free(MonitorDataMessage *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					MonitorDataMessage_init(self);
				}
			return NULL;
		}
	return self;
}

InstrumentStatus *
InstrumentStatus_alloc(MonitorDataMessage *parent)
{
	InstrumentStatus *self;
	self = (InstrumentStatus *) malloc(sizeof(InstrumentStatus));
	if (!self) return NULL;
	InstrumentStatus_init(self, parent);
	self->self_free = free;
	return self;
}

void
InstrumentStatus_init(InstrumentStatus *self, MonitorDataMessage *parent)
{
	self->d = NULL;
	self->d_free = NULL;
	self->shortid = 0;
	*(self->type) = '\0';
	self->refcount = 0;
	self->self_free = NULL;
	self->parent = parent;
	self->length = 0;
	if (parent) MonitorDataMessage_incref(parent);
}

InstrumentStatus *
InstrumentStatus_incref(InstrumentStatus *self)
{
	if (self) self->refcount++;
	return self;
}

InstrumentStatus *
InstrumentStatus_decref(InstrumentStatus *self)
{
	if (self)
		{
			self->refcount--;
			if (self->refcount < 1)
				{
					InstrumentStatus_free(self);
					return NULL;
				}
		}
	return self;
}

InstrumentStatus *
InstrumentStatus_free(InstrumentStatus *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->parent) self->parent = MonitorDataMessage_decref(self->parent);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					InstrumentStatus_init(self, NULL);
				}
			return NULL;
		}
	return self;
}

StatusEntry *
StatusEntry_alloc(InstrumentStatus *parent)
{
	StatusEntry *self;
	self = (StatusEntry *) malloc(sizeof(StatusEntry));
	if (!self) return NULL;
	StatusEntry_init(self, parent);
	self->self_free = free;
	return self;
}

void
StatusEntry_init(StatusEntry *self, InstrumentStatus *parent)
{
	self->d = NULL;
	self->d_free = NULL;
	self->info = NULL;
	self->refcount = 0;
	self->self_free = NULL;
	self->parent = parent;
	if (parent) InstrumentStatus_incref(parent);
}

StatusEntry *
StatusEntry_incref(StatusEntry *self)
{
	if (self) self->refcount++;
	return self;
}

StatusEntry *
StatusEntry_decref(StatusEntry *self)
{
	if (self)
		{
			self->refcount--;
			if (self->refcount < 1)
				{
					StatusEntry_free(self);
					return NULL;
				}
		}
	return self;
}

StatusEntry *
StatusEntry_free(StatusEntry *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->parent) self->parent = InstrumentStatus_decref(self->parent);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					StatusEntry_init(self, NULL);
				}
			return NULL;
		}
	return self;
}

CommandMessage *
CommandMessage_alloc(void)
{
	CommandMessage *self;
	self = (CommandMessage *) malloc(sizeof(CommandMessage));
	if (!self) return NULL;
	CommandMessage_init(self);
	self->self_free = free;
	return self;
}

void
CommandMessage_init(CommandMessage *self)
{
	self->d = NULL;
	self->d_free = NULL;
	self->seqnum = 0;
	self->id = 0;
	self->parameters = NULL;
	self->par_free = NULL;
	self->parlen = 0;
	self->refcount = 0;
	self->self_free = NULL;
}

CommandMessage *
CommandMessage_incref(CommandMessage *self)
{
	if (self) self->refcount++;
	return self;
}

CommandMessage *
CommandMessage_decref(CommandMessage *self)
{
	if (self)
		{
			self->refcount--;
			if (self->refcount < 1)
				{
					CommandMessage_free(self);
					return NULL;
				}
		}
	return self;
}

CommandMessage *
CommandMessage_free(CommandMessage *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->par_free)
				{
					(self->par_free)(self->d);
					self->par_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					CommandMessage_init(self);
				}
			return NULL;
		}
	return self;
}

ReceptionResponseMessage *
ReceptionResponseMessage_alloc(void)
{
	ReceptionResponseMessage *self;
	self = (ReceptionResponseMessage *) malloc(sizeof(ReceptionResponseMessage));
	if (!self) return NULL;
	ReceptionResponseMessage_init(self);
	self->self_free = free;
	return self;
}

void
ReceptionResponseMessage_init(ReceptionResponseMessage *self)
{
	self->d = NULL;
	self->d_free = NULL;
	self->seqnum = 0;
	self->id = 0;
	self->rxnum = 0;
	self->result = response_nil;
	self->endtime = 0;
	self->refcount = 0;
	self->self_free = NULL;
}

ReceptionResponseMessage *
ReceptionResponseMessage_incref(ReceptionResponseMessage *self)
{
	if (self) self->refcount++;
	return self;
}

ReceptionResponseMessage *
ReceptionResponseMessage_decref(ReceptionResponseMessage *self)
{
	if (self)
		{
			self->refcount--;
			if (self->refcount < 1)
				{
					ReceptionResponseMessage_free(self);
					return NULL;
				}
		}
	return self;
}

ReceptionResponseMessage *
ReceptionResponseMessage_free(ReceptionResponseMessage *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					ReceptionResponseMessage_init(self);
				}
			return NULL;
		}
	return self;
}

CompletionResponseMessage *
CompletionResponseMessage_alloc(void)
{
	CompletionResponseMessage *self;
	self = (CompletionResponseMessage *) malloc(sizeof(CompletionResponseMessage));
	if (!self) return NULL;
	CompletionResponseMessage_init(self);
	self->self_free = free;
	return self;
}

void
CompletionResponseMessage_init(CompletionResponseMessage *self)
{
	self->d = NULL;
	self->d_free = NULL;
	self->seqnum = 0;
	self->id = 0;
	self->rxnum = 0;
	self->result = response_nil;
	self->errorcode = 0;
	self->data = NULL;
	self->datalen = 0;
	self->data_free = NULL;
	self->refcount = 0;
	self->self_free = NULL;
}

CompletionResponseMessage *
CompletionResponseMessage_incref(CompletionResponseMessage *self)
{
	if (self) self->refcount++;
	return self;
}

CompletionResponseMessage *
CompletionResponseMessage_decref(CompletionResponseMessage *self)
{
	if (self)
		{
			self->refcount--;
			if (self->refcount < 1)
				{
					CompletionResponseMessage_free(self);
					return NULL;
				}
		}
	return self;
}

CompletionResponseMessage *
CompletionResponseMessage_free(CompletionResponseMessage *self)
{
	if (self)
		{
			assert(self->refcount < 1);
			if (self->d_free)
				{
					(self->d_free)(self->d);
					self->d_free = NULL;
				}
			if (self->data_free)
				{
					(self->data_free)(self->data);
					self->data_free = NULL;
				}
			if (self->self_free)
				{
					(self->self_free)(self);
				}
			else
				{
					CompletionResponseMessage_init(self);
				}
			return NULL;
		}
	return self;
}
