#!/usr/bin/env python
# vim: set ts=2 sw=2 ai noet:
#
# usage: makekeywords.py status-hash-file csvfile(s)
#
# Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>
#
# Permission is granted for use, copying, modification, distribution,
# and distribution of modified versions of this work under the terms of
# GPL version 2 or later.
#

import sys
import string
import csv
import re
import os
import os.path
import struct

import mitsubishi
import statusentry

# status tree
tree_struct = """{
	/* name/keyword of entry: e.g. 0006E or 00B2L2 */
	char *name;
	/* length in bytes without the 19-byte header */
	int bytes;
	/* data pointer array index */
	int index;
}"""
tree_struct_oneline = tree_struct[:]
tree_struct_oneline = re.sub('\/\*.*\*/', '', tree_struct_oneline)
tree_struct_oneline = re.sub('\n', '', tree_struct_oneline)
tree_struct_oneline = re.sub('\{\s+', '{', tree_struct_oneline)
tree_struct_oneline = re.sub('\s+', ' ', tree_struct_oneline)
tree_struct_init = ',0,0'

# entry info
# common arguments for reader functions
reader_args = 'const void *addr, const unsigned char mask, const size_t length, void *dst';

# status entry hash information
def entry_hash_struct(info_array_struct):
	return '{char *name; struct %s* addr;}' % info_array_struct

entry_array_struct = """{
	/* name/keyword of entry */
	char *name;
	/* E, S, L, L2, or L3 */
	char *type;
	/* instrument ID */
	unsigned int shortid;
	/* status ID */
	unsigned int id;
	/* offset in bytes from the instrument data head */
	int offset;
	/* length of data (1 for a bitfield) */
	int bytes;
	/* mask to extract bitfield information (0 otherwise) */
	unsigned char mask;
	/* function to extract the double value */
	void* (*to_d)(%s);
	/* function to extract the int value */
	void* (*to_i)(%s);
	/* function to extract the char* string, and its length (add 1 for buffer) */
	void* (*to_s)(%s);
	int (*to_s_length)(%s);
	/* function to extract the struct timeval value */
	void* (*to_t)(%s);
}""" % (reader_args, reader_args, reader_args, reader_args, reader_args)

# main
if __name__ == "__main__":

	try:
		status_src = sys.argv[1]
		status_hdr = sys.argv[2]
		tree_src = sys.argv[3]
		tree_hdr = sys.argv[4]
		predefined_solver = sys.argv[5]
		csvfiles = sys.argv[6:]
		if len(csvfiles) < 1:
			raise IndexError
	except IndexError:
		sys.stderr.write('usage: %s dst-status-src dst-status-header dst-tree-src dst-tree-header predefined-solver csvfiles\n\tmakes keyword table for gperf.\n' % sys.argv[0])
		sys.exit(1)

	# read
	sys.stderr.write('reading CSV files...\n')
	entries = statusentry.TelStatusEntries()
	for csvfile in csvfiles:
		try:
			for e in statusentry.TelStatusCsv(csvfile).parse():
				entries.append(e)
		except KeyError, err:
			sys.stderr.write("%s:%s\n" % (csvfile, err))
			continue
		except ValueError, err:
			sys.stderr.write("%s:%s\n" % (csvfile, err))
			continue

	# add a dummy entry for L-0051 (OBE) for which we do not have a CSV file
	entries.append_dummy('L', '0051')

	# check namespace collisions
	sys.stderr.write('checking name space collisions...\n')
	entries.namespace.read_predefined(predefined_solver)
	entries.namespace.solve_collisions()
	es = []
	c = entries.namespace.collisions()
	c.sort()
	for n in c:
		for e in entries.namespace.entries[n]:
			es.append(e)
	if len(es) > 0:
		sys.stderr.write('could not solve name collisions:\n')
		es.sort(lambda x,y: cmp(x.id,y.id))
		for e in es:
			sys.stderr.write('	%s - %s\n' % (e.id,e.name))
	if entries.namespace.newdefined:
		sys.stderr.write('names are newly found: ')
		entries.namespace.save_predefined(predefined_solver)
		sys.stderr.write('recorded to %s\n' % predefined_solver)

	# global names in C
	tree_struct_name = 'melco_tree_s'
	tree_lookup_func = 'melco_tree_lookup'
	tree_ids = 'melco_tree_shortid'
	tree_types = 'melco_tree_type'
	entry_info_array_name = '_status_entry_array'
	info_array_struct = 'melco_status_s'
	info_struct_name = '_status_index_s'
	info_struct_init = ',0'
	info_lookup_func = 'melco_status_lookup'
	info_ith_func = 'melco_status_ith'
	info_wordlen = 'melco_status_word_len_max'
	info_total_func = 'melco_status_total'

	walker = entries.to_a()

	# write keyword table for status tree
	sys.stderr.write('creating status tree source file: %s...\n' % tree_src)
	tree_length = 0
	tree_max_word = 0
	types = entries.types()
	types.sort()
	tree_array = []
	id_array = []
	type_array = []
	for type in types:
		ss = entries.shortids(type)
		ss.sort()
		for shortid in ss:
			name = '%s%s' % (statusentry.realshortid(type, shortid), statusentry.extype(type, shortid))
			bytes = entries.length(type, shortid)
			tree_array.append('"%s", %d, %d\n' % (name, bytes, tree_length))
			id_array.append('\t0x%s,\n' % statusentry.realshortid(type, shortid))
			type_array.append('\t"%s",\n' % statusentry.extype(type, shortid))
			namelen = len(name)
			if namelen > tree_max_word:
				tree_max_word = namelen
			tree_length += 1

	dst = open(tree_src, 'w')
	dst.write('%%{\n#include <%s>\n' % tree_hdr)
	dst.write('unsigned int %s[] = {\n' % tree_ids)
	for l in id_array:
		dst.write(l)
	dst.write('};\n')
	dst.write('char *%s[] = {\n' % tree_types)
	for l in type_array:
		dst.write(l)
	dst.write('};\n')
	dst.write('%}\n')
	dst.write('struct %s %s;\n' % (tree_struct_name, tree_struct_oneline))
	dst.write('%%define initializer-suffix %s\n' % (tree_struct_init))
	dst.write('%%\n')
	for l in tree_array:
		dst.write(l)
	dst.close()

	# write header file for status tree
	macro = os.path.basename(tree_hdr)
	macro = re.sub('\.', '_', macro)
	macro = re.sub('-', '_', macro)
	macro = macro.upper()
	dst = open(tree_hdr, 'w')
	sys.stderr.write('creating status tree header file: %s...\n' % tree_hdr)
	dst.write('/* %s: header file for status tree generated by %s */\n\n' % (tree_hdr, sys.argv[0]))
	dst.write('#ifndef %s\n#define %s\n\n' % (macro, macro))
	dst.write('/* telescope status index */\n')
	dst.write('struct %s %s;\n' % (tree_struct_name, tree_struct))
	dst.write('struct %s *%s(const char *str, unsigned int len);\n\n' % (tree_struct_name, tree_lookup_func))
	dst.write('#define MELCO_TREE_LENGTH %d\n' % tree_length)
	dst.write('#define MELCO_TREE_MAX_WORD_LENGTH %d\n' % tree_max_word)
	dst.write('\n/* short IDs and types of MELCO_TREE_LENGTH elements */\n')
	dst.write('extern unsigned int %s[];\n' % tree_ids)
	dst.write('extern char *%s[];\n' % tree_types)
	dst.write('\n#endif\t/* %s */\n\n' % macro)
	
	# write header file for status entry info
	status_hash_length = 0
	status_hash_word_length = 0
	macro = os.path.basename(status_hdr)
	macro = re.sub('\.', '_', macro)
	macro = re.sub('-', '_', macro)
	macro = macro.upper()
	dst = open(status_hdr, 'w')
	sys.stderr.write('creating status entry info header file: %s...\n' % status_hdr)
	dst.write('/* %s: header file for status hash generated by %s */\n\n' % (status_hdr, sys.argv[0]))
	dst.write('#ifndef %s\n#define %s\n\n' % (macro, macro))
	dst.write('/* telescope status entry */\n')
	dst.write('struct %s %s;\n' % (info_array_struct, entry_array_struct))
	dst.write('struct %s* %s(const char *str, unsigned int len);\n' % (info_array_struct, info_lookup_func))
	dst.write('struct %s* %s(const int i);\n' % (info_array_struct, info_ith_func))
	dst.write('int %s(void);\n' % info_wordlen)
	dst.write('int %s(void);\n' % info_total_func)
	dst.write('\n')
	dst.write('#endif\t/* %s */\n\n' % macro)
	dst.write('/* available keywords: keyword\t[type-shortid byte:bit-offset id (format)] */\n/*\n')
	names = walker[:]
	names.sort(lambda x,y:cmp(x.sort_string(),y.sort_string()))
	for e in names:
		t = e.type
		s = e.shortid
		if e.is_bitfield():
			loc = "%d:%d" % (e.byteoffset, e.bitoffset)
			format = ""
		else:
			loc = "%d" % e.byteoffset
			format = " " + e.data_format()
		if len(e.keyword()) > status_hash_word_length:
			status_hash_word_length = len(e.keyword())
		dst.write('  %s\t[%s-%s %s %s%s]\n' % (e.keyword(), statusentry.extype(t, s), statusentry.realshortid(t, s), loc, e.id, format))
		status_hash_length += 1
	dst.write('*/\n')
	dst.close()

	# write keyword table for status entry info
	sys.stderr.write('creating status entry info source file: %s...\n' % status_src)
	dst = open(status_src, 'w')
	dst.write('%{\n')
	dst.write('#include <melco/status-types.h>\n')
	dst.write('#include <%s>\n\n' % status_hdr)

	# interface function
	dst.write('struct %s %s;\n' % (info_struct_name, entry_hash_struct(info_array_struct)))
	dst.write("""#ifdef __GNUC__
__inline
#endif
struct %s * in_word_set (register const char *str, register unsigned int len);
""" % info_struct_name)
	dst.write("""struct %s*
%s(const char *str, unsigned int len)
{
	struct %s *t;
	t = in_word_set(str, len);
	if (t)
		return t->addr;
	else
		return NULL;
}
""" % (info_array_struct, info_lookup_func, info_struct_name))

	# reader functions
	sys.stderr.write('  writing reader functions...\n')
	fs = entries.reader_function_names()
	fn = fs.keys()
	fn.sort()
	dst.write('/* status entry reader functions */\n')
	for f in fn:
		dst.write("/* %s;" % f)
		for e in fs[f][0:5]:
			dst.write(' ' + e.id)
		if len(fs[f]) > 5:
			dst.write(' ...')
		dst.write(' */\n')
	dst.write('\n')

	# array entries
	sys.stderr.write('  writing the array...\n')
	dst.write('/* status entry array in order of status ID */\n')
	dst.write('struct %s %s[] = {\n' % (info_array_struct, entry_info_array_name))
	c = 0;
	for e in walker:
		dst.write('\t' + e.array_entry(c))
		c += 1
	dst.write('};\n')

	# more interface functions
	dst.write("""struct %s*
%s(const int i)
{
	if (i < 0) return NULL;
	if (i >= %d) return NULL;
	return &%s[i];
}
""" % (info_array_struct, info_ith_func, status_hash_length, entry_info_array_name))

	dst.write("""int
%s(void)
{
	return %d;
}
""" % (info_wordlen, status_hash_word_length))

	dst.write("""int
%s(void)
{
	return %d;
}
""" % (info_total_func, status_hash_length))

	dst.write('\n%}\n')

	# hash entries
	sys.stderr.write('  writing the keyword table...\n')
	c = 0;
	dst.write('struct %s %s;\n' % (info_struct_name, entry_hash_struct(info_array_struct)))
	dst.write('%%define initializer-suffix %s\n' % (info_struct_init))
	dst.write('%%\n')
	for e in walker:
		dst.write(e.hash_entry(c, entry_info_array_name))
		c += 1
	dst.close()

	sys.stderr.write('done.\n')
