/* dump-tsc-log.c: dump status data from log files from SOSS */
/*

Copyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>

Permission is granted for use, copying, modification, distribution,
and distribution of modified versions of this work under the terms of
GPL version 2 or later.

*/

/* If we're not using GNU C, elide __attribute__ */
#ifndef __GNUC__
#define  __attribute__(x)  /*NOTHING*/
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef PACKAGE_STRING
#define PACKAGE_STRING "unknown package"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <libgen.h>
#include <time.h>
#include <ctype.h>

#include <melco/status-hash.h>
#include <melco/status-ids.h>
#include <melco/snprintf-compat.h>
#include <melco/tsc-message.h>

#ifdef COUNT_MALLOC
#include <count-malloc.h>
#endif

#define SIGN(x) (((x) < 0) ? -1 : (((x) > 0) ? 1 : 0))

#define DEF_DSTDIR "."
#define TZBUFLEN 10	/* is this enough? */
#define NTIMESPAN 6	/* start-end, start-end, original start, end */
#define DAYSEC 24L*60*60	/* seconds in a day (ignoring leap seconds) */
#define HMSBUFLEN 10	/* HH:MM:SS\0 */

/* flags whether to use different status logs */
struct usage_flags {
	int lstat;
	int sstat;
	int vstat;
};

/* list of names */
struct name_list_s {
	char *name;	/* will not be free()ed */
	struct name_list_s *next;
};

/* list of names for a short ID */
struct shortid_list_s {
	char type[3];	/* E, S, L, L2, or L3 */
	unsigned int shortid;	/* instrument ID */
	FILE *stream;	/* output file stream */
	struct timeval prevtime;	/* rx time of previous output */
	time_t offset;
	struct name_list_s names;
	struct shortid_list_s *next;	/* will be free()ed */
	int show_hex;
	struct timeval txtime;	/* tx time of the packet */
	int show_tsc_time;
};

/* global */
static char *tzbuffer = NULL;

/* prototypes */
void usage(FILE *stream);
void name_list(FILE *stream);
void show_version(FILE *stream, const int show_long);
void add_entry(struct shortid_list_s *tree, struct usage_flags *flag, const char *entry_name, const int vf);
int has_entry(const struct shortid_list_s *tree);
long hms_to_sec(const char *hms);
void sec_to_hms(const long sec, char *ten_byte, char *sep);
void set_time_zone(const char *tz, const int vf);
void free_time_zone(void);
void set_time_span(long timespan[], const char *timerange, int vf);
int in_time_span(const long timespan[], time_t t);

/* name list */
struct name_list_s* append_name(struct name_list_s *root, char *name);
void free_names(struct name_list_s *root);

/* shortid-type list */
struct shortid_list_s* append_shortid(struct shortid_list_s *self, const struct melco_status_s* info);	/* append or find the shortid entry and return it */
void free_shortids(struct shortid_list_s* root);

struct shortid_list_s* append_name_to_shortid(struct shortid_list_s *root, const struct melco_status_s *info);

char* dstfilename(struct shortid_list_s *names, const char *srcfilename, const char* dstdir, const long timespan[]); /* has to be free()ed afterwards */
int shortid_is_open(const struct shortid_list_s *self);
FILE* shortid_fopen(struct shortid_list_s *self, const char *filename);
void shortid_fclose(struct shortid_list_s *self);
void shortid_print_header(struct shortid_list_s *names, InstrumentStatus *inst);
int shortid_print_value(struct shortid_list_s *names, InstrumentStatus* inst);

/* usage and version */
void
usage(FILE *stream)
{
	fputs("usage: dump-tsc-log [options] status-log-files\n", stream);
	fputs("\treads status log files and dumps requested data into files.\n\n", stream);
	fputs("Output are created as [input name]-[status type]-[instrument id].log\n", stream);
	fputs("with reception date and time from LCUs on MLPx (UTC)\n", stream);
	fputs("in the first coulmns and values in the following columns.\n\n", stream);
	fputs("OPTIONS\n", stream);
	fputs("-e entry-name  : retrieves status with ENTRY-NAME.\n", stream);
	fputs("                 Supply -l option for available names.\n", stream);
	fputs("                 A status ID in hex can also be supplied.\n", stream);
	fputs("\n", stream);
	fputs("-E             : additional entries to be read from standard input.\n", stream);
	fputs("                 lines beginning with `#' and strings blank + `[' are ignored.\n", stream);
	fputs("\n", stream);
	fputs("-d dstdir      : create log files in DSTDIR (default: .)\n", stream);
	fputs("\n", stream);
	fputs("-a             : scan all the bytes in the status data\n", stream);
	fputs("                 (default: believe predefined status data lenghts)\n", stream);
	fputs("\n", stream);
	fputs("-n             : NOT assume Fujitsu's file name convention: filenames begin\n", stream);
	fputs("                 with TSCL, TSCS, TSCV for long, short, variable period status.\n", stream);
	fputs("\n", stream);
	fputs("-l             : list all the available status names and exit.\n", stream);
	fputs("                 You may want to pipe the output to a pager.\n", stream);
	fputs("\n", stream);
	fputs("-z             : use local timezone insted of UTC.\n", stream);
	fputs("                 Local time can be specified using TZ environment variable.\n", stream);
	fputs("\n", stream);
	fputs("-r hh:mm-hh:mm : limit time range of output in UT or in localtime when\n", stream);
	fputs("                 -z option is supplied.\n", stream);
	fputs("                 Note that the time is limited referring Tx time of\n", stream);
	fputs("                 status package while Rx time of status is recorded in\n", stream);
	fputs("                 output.\n", stream);
	fputs("\n", stream);
	fputs("-b             : show raw data as hexadecimal numbers.\n", stream);
	fputs("-t             : add timestamp by TSC (Tx time) at time of reception from MLP.\n", stream);
	fputs("-v             : increase verbosity.\n", stream);
	fputs("-q             : decrease verbosity.\n", stream);
}

void
show_version(FILE *stream, const int show_long)
{
	fputs("dump-tsc-long in ", stream);
	fputs(PACKAGE_STRING, stream);
	fputs("\n", stream);
	if (show_long)
		{
			fputs("\nCopyright (C) 2005 by Daigo Tomono <tomono at subaru.naoj.org>\n\n", stream);

			fputs("Permission is granted for use, copying, modification, distribution,\n", stream);
			fputs("and distribution of modified versions of this work under the terms of\n", stream);
			fputs("GPL version 2 or later.\n", stream);
		}
}

/* show list of available names */
void
name_list(FILE *stream)
{
	struct melco_status_s *info;
	int i, n;
	n = melco_status_total();
	fprintf(stream, "#name\t[type-shortid location id]\n");
	for(i = 0; i < n; i++)
		{
			info = melco_status_ith(i);
			if (info->mask)
				{
					int b, m;
					m = info->mask;
					for(b = 0; (m & 0x01) == 0; b++) m >>= 1;
					fprintf(stream, "%s\t[%s-%04X %d:%d %07X]\n", info->name, info->type, info->shortid, info->offset, b, info->id);
				}
			else
				{
					fprintf(stream, "%s\t[%s-%04X %d %07X]\n", info->name, info->type, info->shortid, info->offset, info->id);
				}
		}
}

/* list of names - allow identical names to be registered twice */
struct name_list_s*
append_name(struct name_list_s *root, char *name)
{
	struct name_list_s *cur, *new;
	cur = root->next;
	if (cur)
		while(cur->next) cur = cur->next;
	else
		cur = root;
	new = (struct name_list_s*) malloc(sizeof(struct name_list_s));
	cur->next = new;
	new->name = name;
	new->next = NULL;
	return new;
}

void
free_names(struct name_list_s *root)
{
	struct name_list_s *cur, *next;
	cur = root->next;
	while(cur)
		{
			next = cur->next;
			free(cur);
			cur = next;
		}
}

/* list of short-ids - no duplicate entries */
struct shortid_list_s*
append_shortid(struct shortid_list_s *root, const struct melco_status_s* info)
{
	struct shortid_list_s *cur, *new;
	if (!root) return NULL;
	cur = root;
	while(cur->next)	{
		if (!strcmp(cur->next->type, info->type) && (cur->next->shortid == info->shortid))
			return cur->next;
		cur = cur->next;
	}
	new = (struct shortid_list_s *) malloc(sizeof(struct shortid_list_s));
	cur->next = new;

	strncpy(new->type, info->type, sizeof(new->type));
	new->type[sizeof(new->type)-1] = '\0';
	new->shortid = info->shortid;
	new->stream = NULL;
	new->prevtime.tv_sec = 0;
	new->prevtime.tv_usec = 0;
	new->offset = 0;
	new->names.name = NULL;
	new->names.next = NULL;
	new->next = NULL;
	return new;
}

void
free_shortids(struct shortid_list_s* root)
{
	struct shortid_list_s *cur, *next;
	cur = root->next;
	while(cur)
		{
			next = cur->next;
			shortid_fclose(cur);
			free_names(&(cur->names));
			free(cur);
			cur = next;
		}
}

struct shortid_list_s*
append_name_to_shortid(struct shortid_list_s *root, const struct melco_status_s *info)
{
	struct shortid_list_s *cur;

	cur = append_shortid(root, info);
	append_name(&(cur->names), info->name);
	return cur;
}

char* dstfilename(struct shortid_list_s *self, const char *srcfilename, const char* dstdir, const long timespan[])
{
	int len;
	char *r;
	char *basec, *bname;
	char r1[HMSBUFLEN], r2[HMSBUFLEN];
	basec = strdup(srcfilename);
	bname = basename(basec);
	if (timespan)
		{
			sec_to_hms(timespan[4], r1, "");
			sec_to_hms(timespan[5], r2, "");
			len = snprintf_length("%s/%s-%s-%s-%s-%04X.log", dstdir, bname, r1, r2, self->type, self->shortid);
		}
	else
		len = snprintf_length("%s/%s-%s-%04X.log", dstdir, bname, self->type, self->shortid);
	r = (char*) malloc(len + 1);
	if (r)
		{
			if (timespan)
				snprintf(r, len+1, "%s/%s-%s-%s-%s-%04X.log", dstdir, bname, r1, r2, self->type, self->shortid);
			else
				snprintf(r, len+1, "%s/%s-%s-%04X.log", dstdir, bname, self->type, self->shortid);
		}
	free(basec);
	return r;
}

int
shortid_is_open(const struct shortid_list_s *self)
{
	return (self->stream != NULL);
}

FILE*
shortid_fopen(struct shortid_list_s *self, const char *filename)
{
	self->stream = fopen(filename, "w");
	return self->stream;
}

void
shortid_fclose(struct shortid_list_s *self)
{
	if (self->stream)
		{
			fclose(self->stream);
			self->stream = NULL;
		}
}

void
shortid_print_header(struct shortid_list_s *self, InstrumentStatus *inst)
{
	struct name_list_s *cur;
	struct timeval tv;
	struct tm tm;
	char tzbuf[TZBUFLEN];

	InstrumentStatus_rxtime_tv(inst, &tv);
	memset(&tm, 0, sizeof(tm));
#ifdef HAVE_LOCALTIME_R
	localtime_r(&(tv.tv_sec), &tm);
#else
	{
		struct tm *tmptr;
		tmptr = localtime(&(tv.tv_sec));
		tm = *tmptr;
	}
#endif
	strftime(tzbuf, sizeof(tzbuf), "%Z", &tm);
	tzbuf[sizeof(tzbuf)-1] = '\0';
	fprintf(self->stream, "#rxdate\trxtime(%s)\tsec-from-00:00:00UTC", tzbuf);
	if (self->show_tsc_time) fprintf(self->stream, "\ttxdate\ttxtime(%s)", tzbuf);
	cur = self->names.next;
	while(cur)
		{
			fputs("\t", self->stream);
			fputs((char*) cur->name, self->stream);
			cur = cur->next;
		}
	fputs("\n", self->stream);

	/* we want at least a line of data in the file */
	self->prevtime.tv_sec = 0;
	self->prevtime.tv_usec = 0;
	self->offset = 0;
}

int
shortid_print_value(struct shortid_list_s *self, InstrumentStatus* inst)
{
	int r = 0;
	struct timeval x, x_round;
	struct tm tm;
	char datetime[11];
	double sec;
	struct name_list_s *cur;

	/* date-time header */
	InstrumentStatus_rxtime_tv(inst, &x);
	if (x.tv_sec == self->prevtime.tv_sec && x.tv_usec == self->prevtime.tv_usec) return 0;	/* same time stamp */
	if (0 == self->prevtime.tv_sec) self->offset = (x.tv_sec / 86400) * 86400;
	self->prevtime = x;

	x_round = x;
	x_round.tv_usec = (((x.tv_usec / 50000) + 1) / 2)*100000;
	if(x_round.tv_usec >= 1000000)
		{
			x_round.tv_sec += x_round.tv_usec / 1000000;
			x_round.tv_usec = x_round.tv_usec % 1000000;
		}

	memset(&tm, 0, sizeof(tm));
#ifdef HAVE_LOCALTIME_R
	localtime_r(&(x_round.tv_sec), &tm);
#else
	{
		struct tm *tmptr;
		tmptr = localtime(&(x_round.tv_sec));
		tm = *tmptr;
	}
#endif

	strftime(datetime, sizeof(datetime), "%Y-%m-%d", &tm);
	fputs(datetime, self->stream);
	fputs("\t", self->stream);
	strftime(datetime, sizeof(datetime), "%H:%M:", &tm);
	fputs(datetime, self->stream);

	sec = (double) tm.tm_sec;
	sec += x_round.tv_usec * 1e-6;
	if (tm.tm_sec < 10)
		{
			fputs("0", self->stream);
		}
	fprintf(self->stream, "%.1f", sec);

	fprintf(self->stream, "\t%.1f", (double) (x_round.tv_sec - self->offset) + (double) x_round.tv_usec * 1e-6 );

	/* TSC time */
	if (self->show_tsc_time)
		{
			x_round = self->txtime;
			x_round.tv_usec = (((self->txtime.tv_usec / 50000) + 1) / 2)*100000;
			if(x_round.tv_usec >= 1000000)
				{
					x_round.tv_sec += x_round.tv_usec / 1000000;
					x_round.tv_usec = x_round.tv_usec % 1000000;
				}

			memset(&tm, 0, sizeof(tm));
#ifdef HAVE_LOCALTIME_R
			localtime_r(&(x_round.tv_sec), &tm);
#else
			{
				struct tm *tmptr;
				tmptr = localtime(&(x_round.tv_sec));
				tm = *tmptr;
			}
#endif

			fputc('\t', self->stream);
			strftime(datetime, sizeof(datetime), "%Y-%m-%d", &tm);
			fputs(datetime, self->stream);
			fputc('\t', self->stream);
			strftime(datetime, sizeof(datetime), "%H:%M:", &tm);
			fputs(datetime, self->stream);

			sec = (double) tm.tm_sec;
			sec += x_round.tv_usec * 1e-6;
			if (tm.tm_sec < 10)
				{
					fputs("0", self->stream);
				}
			fprintf(self->stream, "%.1f", sec);
		}

	/* values */
	cur = self->names.next;
	while(cur)
		{
			StatusEntry *entry;
			fputs("\t", self->stream);
			if ((entry = InstrumentStatus_get_status_entry(inst, cur->name)))
				{
					int len;
					char *buf;
					len = self->show_hex ? StatusEntry_to_hex_length(entry) : StatusEntry_to_s_length(entry);
					if (len > 0)
						buf = (char*) malloc(len + 1);
					else
						buf = NULL;
					if (buf)
						{
							if (self->show_hex ? StatusEntry_to_hex(entry, buf) : StatusEntry_to_s(entry, buf))
								r++;
							else
								fputs("*", self->stream);
							fputs(buf, self->stream);
							free(buf);
						}
					else
						{
							fputs("**", self->stream);
						}
					StatusEntry_free(entry);
				}
			else
				{
					fputs("*", self->stream);
				}
			cur = cur->next;
		}
	fputs("\n", self->stream);
	return r;
}

void
add_entry(struct shortid_list_s *tree, struct usage_flags *flag, const char *entry_name, const int vf)
{
	struct melco_status_s* melco_status;

	if (strlen(entry_name) < 1)
		{
			fputs("WARNING: empty entry name supplied\n", stderr);
			return;
		}

	/* try to treat input as a name */
	melco_status = melco_status_lookup(entry_name, strlen(entry_name));
	if (melco_status && vf > 0) fprintf(stderr, "requesting: %s\n", entry_name);

	/* try to treat input as a hexagonal status ID */
	if (!melco_status)
		{
			unsigned int id;
			char *endptr;

			id = (unsigned int) strtoul(entry_name, &endptr, 16);
			if (*endptr == '\0')
				{
					melco_status = melco_status_from_id(id);
					if (melco_status)
						{
							if (vf > 0) fprintf(stderr, "requesting: ID:%07X %s\n", id, melco_status->name);
						}
					else
						{
							fprintf(stderr, "WARNING: status entry with ID \"%s\" not found.\n", entry_name);
							return;
						}
				}
		}

	/* add the entry */
	if (melco_status)
		{
			if (!append_name_to_shortid(tree, melco_status))
				{
					fprintf(stderr, "WARNING: error in adding the entry named \"%s\".\n", entry_name);
					return;
				}
			switch(melco_status->type[0])
				{
					case 'L': flag->lstat = 1; break;
					case 'S': flag->sstat = 1; break;
					case 'E':	flag->vstat = 1; break;
					default:
						fprintf(stderr, "bug: unknown status type %s for %s\n", melco_status->type, entry_name);
						exit(EXIT_FAILURE);
				}
			return;
		}

	fprintf(stderr, "WARNING: status entry named or with ID \"%s\" not found.\n", entry_name);
}

int
has_entry(const struct shortid_list_s *tree)
{
	return !!(tree->next);
}

void
set_time_zone(const char *tz, const int vf)
{
	if (tz)
		{
			size_t l __attribute__ ((unused));
			free_time_zone();
#ifdef HAVE_SETENV
			setenv("TZ", tz, 1);
#else
			l = 3 + strlen(tz) + 1;	/* TZ=...\0 */
			tzbuffer = (char*) malloc(l);
			if (!tzbuffer)
				{
					perror("malloc()");
					exit(EXIT_FAILURE);
				}
			snprintf(tzbuffer, l, "TZ=%s", tz);
			if (putenv(tzbuffer))
				{
					perror("tzbuffer()");
					exit(EXIT_FAILURE);
				}
#endif
		}
	tzset();
	if (strlen(tzname[1]) < 1)
		{
			const char *msg;
			msg = tz;
			if (!msg) msg = getenv("TZ");
			if (!msg) msg = tzname[0];
			fprintf(stderr, "WARNING: DST zone name is empty. `%s' might not be recognized as you want.\n", msg);
		}
	else
		{
			if (vf > 0)
				fprintf(stderr, "setting time zone to %s.\n", tzname[0]);
		}
}

void
free_time_zone(void)
{
#ifdef HAVE_SETENV
	unsetenv("TZ");
#else
	putenv("TZ=UTC");
#endif
	if (tzbuffer)
		{
			free(tzbuffer);
			tzbuffer = NULL;
		}
}

long
hms_to_sec(const char *hms)
{
	long r, c;
	const char *cptr;
	char *eptr;
	int ncolon;

	ncolon = 0;
	cptr = hms;
	r = 0;
	while(1)
		{
			c = strtol(cptr, &eptr, 10);
			if (eptr == cptr) return -1;
			r *= 60;
			r += c;
			ncolon++;
			if (*eptr == '\0' || *eptr == '-') break;
			if (*eptr != ':') return -1;
			if (ncolon > 2) return -1;
			cptr = eptr + 1;
		}
	for(; ncolon < 3; ncolon++) r *= 60;
	while(r < 0) r += DAYSEC;
	r %= DAYSEC;
	return r;
}

void
sec_to_hms(const long sec, char *ten_byte, char *sep)
{
	long ss, h, m, s;
	ss = sec;
	while(ss < 0) ss += DAYSEC;
	ss %= DAYSEC;
	h = ss/3600L;
	m = (ss/60L) % 60L;
	s = ss % 60L;
	snprintf(ten_byte, HMSBUFLEN, "%02ld%s%02ld%s%02ld", h, sep, m, sep, s);
}

void
set_time_span(long timespan[], const char *timerange, int vf)
{
	long t[2];
	char *minus;

	minus = strchr(timerange, '-');
	if (!minus)
		{
			fprintf(stderr, "time range `%s' does not contain `-'\n", timerange);
			exit(EXIT_FAILURE);
		}

	/* t1 */
	if (timerange < minus)
		{
			t[0] = hms_to_sec(timerange);
			if (t[0] < 0)
				{
					fprintf(stderr, "malformed time specified in `%s'\n", timerange);
					exit(EXIT_FAILURE);
				}
		}
	else
		t[0] = 0;

	/* t2 */
	if (*(minus + 1))
		{
			t[1] = hms_to_sec(minus + 1);
			if (t[1] < 0)
				{
					fprintf(stderr, "malformed time in  `%s'\n", timerange);
					exit(EXIT_FAILURE);
				}
		}
	else
		t[1] = DAYSEC;

	timespan[4] = t[0];
	timespan[5] = t[1];
	if (t[0] < t[1])
		{
			timespan[0] = t[0];
			timespan[1] = t[1];
			timespan[2] = -1;
			timespan[3] = -1;
		}
	else
		{
			timespan[0] = 0;
			timespan[1] = t[1];
			timespan[2] = t[0];
			timespan[3] = DAYSEC;
		}

	if (vf > 0)
		{
			char r1[HMSBUFLEN], r2[HMSBUFLEN];
			sec_to_hms(timespan[4], r1, ":");
			sec_to_hms(timespan[5], r2, ":");
			fprintf(stderr, "limiting time range to %s-%s\n", r1, r2);
		}
}

int
in_time_span(const long timespan[], time_t t)
{
	int r;
	long tt;
	struct tm tm;

	memset(&tm, 0, sizeof(tm));
#ifdef HAVE_LOCALTIME_R
	localtime_r(&t, &tm);
#else
	{
		struct tm *tmptr;
		tmptr = localtime(&t);
		tm = *tmptr;
	}
#endif

	tt = tm.tm_sec + tm.tm_min*60 + tm.tm_hour*60*60;
	r = 0;
	if (SIGN(timespan[0] - tt)*SIGN(timespan[1] - tt) <= 0) r = 1;
	if (timespan[2] >= 0 && SIGN(timespan[2] - tt)*SIGN(timespan[3] - tt) <= 0) r = 1;

	return r;
}

int
main(int argc, char *argv[])
{
	struct shortid_list_s tree;
	const char* dstdir;
	char *timerange;
	long timespan[NTIMESPAN];
	int vf;
	int tz_specified;
	int skip_with_filename;
	int show_hex;
	int show_tsc_time;
	struct usage_flags uf;
	scan_option_t scan_opt;

	tree.next = NULL;
	vf = 1;
	scan_opt = scan_strict;
	uf.lstat = 0;
	uf.sstat = 0;
	uf.vstat = 0;
	skip_with_filename = 1;
	tz_specified = 0;
	timerange = NULL;
	show_hex = 0;
	show_tsc_time = 0;

	/* command line options */
	{
		int r;
		char *entrybuf, *end_of_entry;
		int entrybuflen, inputlen, more_in_line;
		entrybuflen = melco_status_word_len_max() + 2;	/* keyword + \n + \0 */
		if ((entrybuf = (char *) malloc(entrybuflen)) == NULL)
			{
				perror(argv[0]);
				exit(EXIT_FAILURE);
			}
		dstdir = DEF_DSTDIR;
		while((r = getopt(argc, argv, "Vhe:vqd:anlEzr:bt")) != -1)
			{
				switch(r)
					{
					case 'V':	/* version */
						show_version(stdout, 1);
						free(entrybuf);
						return EXIT_SUCCESS;
					case 'h':	/* help */
						usage(stdout);
						free(entrybuf);
						return EXIT_SUCCESS;
					case 'l':	/* list */
						name_list(stdout);
						free(entrybuf);
						return EXIT_SUCCESS;
					case 'E':	/* add entries from stdin */
						if (vf > 1) fprintf(stderr, "reading request for entries from stdin.\n");
						while(1)
							{
								fgets(entrybuf, entrybuflen, stdin);
								if (feof(stdin)) break;
								inputlen = strlen(entrybuf);
								more_in_line = (entrybuf[inputlen - 1] != '\n');
								if (entrybuf[0] != '#')	/* avoid comment lines */
									{
										/* omit `[...' */
										end_of_entry = strchr(entrybuf, '[');
										if (end_of_entry)
											*end_of_entry = '\0';
										else
											end_of_entry = entrybuf + inputlen;
										/* omit trailing spaces */
										while(entrybuf < end_of_entry && isspace((int) *(end_of_entry - 1)))
{
											end_of_entry--;
}
										*end_of_entry = '\0';
										/* add the entry to our list */
										add_entry(&tree, &uf, entrybuf, vf);
									}
								while(more_in_line && !feof(stdin))	/* read chars left */
									{
										fgets(entrybuf, entrybuflen, stdin);
										more_in_line = (entrybuf[strlen(entrybuf) - 1] != '\n');
									}
							}
						break;
					case 'e':	/* add an entry */
						add_entry(&tree, &uf, optarg, vf);
						break;
					case 'd':	/* dstdir */
						dstdir = optarg;
						break;
					case 'a':	/* scan all */
						scan_opt = scan_all;
						break;
					case 'n':	/* not assume filename convention */
						skip_with_filename = 0;
						break;
					case 'z':	/* time zone */
						set_time_zone(NULL, vf);
						tz_specified = 1;
						break;
					case 'r':	/* time range */
						timerange = optarg;
						set_time_span(timespan, timerange, vf);
						break;
					case 'v':	/* louder */
						vf += 1;
						break;
					case 'q':	/* more quiet */
						vf -= 1;
						break;
					case 'b':	/* hex presentation */
						if (vf > 0) fprintf(stderr, "recording raw data in hexadecimal expression\n");
						show_hex = 1;
						break;
					case 't':	/* TSC time */
						if (vf > 0) fprintf(stderr, "recording time stamp added by TSC\n");
						show_tsc_time = 1;
						break;
					default:
						usage(stderr);
						free(entrybuf);
						return EXIT_FAILURE;
					}
			}
		if (!tz_specified) set_time_zone("UTC", vf - 1);
		free(entrybuf);
		if (!has_entry(&tree))
			{
				fputs("WARNING: no status entry specified. Just scanning through input files.\n", stderr);
			}
	}

	/* scan through input files */
	if (optind >= argc)
		{
			fputs("ERROR: no input file specified.\n", stderr);
			exit(EXIT_FAILURE);
		}
	for(; optind < argc; optind++)
		{
			char *srcfile;
			FILE *src;
			int srcused, canskip;

			srcfile = argv[optind];
			if (skip_with_filename)
				{
					int use_this;
					int lsv_pos;
					char *basec, *bname;
					basec = strdup(srcfile);
					bname = basename(basec);
					use_this = 0;
					lsv_pos = -1;
					if (strlen(bname) >= 4 && !strncmp("TSC", bname, 3))
						{
							lsv_pos = 3;
						}
					else
						{
							if (strlen(bname) >= 1) lsv_pos = 0;
						}
					if (lsv_pos >= 0)
						{
							switch (bname[lsv_pos])
								{
									case 'L':	if (uf.lstat) use_this = 1; break;
									case 'S':	if (uf.sstat) use_this = 1; break;
									case 'V':	if (uf.vstat) use_this = 1; break;
								}
						}
					free(basec);
					if (has_entry(&tree) && !use_this)
						{
							if (vf > 1) fprintf(stderr, "skipping %s which is not used.\n", srcfile);
							continue;
						}
				}

			src = fopen(srcfile, "r");
			if (!src)
				{
					perror(srcfile);
					continue;
				}
			if (vf > 0) fprintf(stderr, "reading %s\n", srcfile);

			/* read through the file */
			srcused = 0;
			while(!feof(src))
				{
					MonitorDataMessage *data;
					struct timeval txtime;
					StatusBank db;
					tsc_error_t e;
					int ninst;
					struct shortid_list_s* cur_inst;

					/* new block */
					data = MonitorDataMessage_alloc();
					if ((e = fread_MonitorDataMessage(data, src, block_none_or_fujitsu, 0)) != tsc_success)
						{
							switch(e)
								{
									case tsc_error_stream:
										if (!feof(src)) perror(srcfile);
										break;
									case tsc_error_malloc:
										fprintf(stderr, "%s: out of memory.\n", srcfile);
										break;
									case tsc_error_malformed:
										fprintf(stderr, "%s: malformed header.\n", srcfile);
										break;
									case tsc_success:
									default:
										break;
										/* never reach here */
								}
							continue;
						}
					if (vf > 1) fprintf(stderr, "monitor data message type: %s\n", data->type);

					/* check time */
					if (timerange || show_tsc_time)
						message_header_txtime_tv(data->d, &txtime);
					if (timerange)
						{
							if (!in_time_span(timespan, txtime.tv_sec))
								{
									MonitorDataMessage_decref(data);
									if (vf > 1)
										{
											char hms[HMSBUFLEN];
											sec_to_hms(txtime.tv_sec, hms, ":");
											fprintf(stderr, "skipping monitor data message transmitted at %s.\n", hms);
										}
									goto next_block;
								}
						}

					/* check types */
					{
						switch(data->type[0])
							{
								case 'L': canskip = 0; break;
								case 'S':
									/* only one content in all monitor data message packets */
									if (has_entry(&tree))
										{
											canskip = 1;
										}
									else
										{
											canskip = 0;	/* scan mode */
										}
									break;
								case 'E': canskip = 0; break;
								default:
									fprintf(stderr, "%s: unknown monitor data message type: %c\n", srcfile, data->type[0]);
									goto next_file;
							}
					}
					if (vf > 2) fprintf(stderr, "  %33.33s - ", data->d->buf);

					StatusBank_init(&db);
					ninst = MonitorDataMessage_scan(data, &db, scan_opt);
					if (vf > 2) 
						{
							fprintf(stderr, "%d instruments\n", ninst);
						}
					if (ninst < 0)
						{
							if (0 < vf && vf <= 2)
								{
										fprintf(stderr, "%33.33s:\n  might missed some status. Add -a option for a complete scan\n", data->d->buf);
								}
							if (2 < vf)
								{
										fputs(" might missed some status. Add -a option for a complete scan\n", stderr);
								}
						}

					/* look into each instrument requested */
					cur_inst = tree.next;
					while(cur_inst)
						{
							InstrumentStatus *inst;

							cur_inst->show_tsc_time = show_tsc_time;
							cur_inst->txtime = txtime;
							cur_inst->show_hex = show_hex;

							/* check if we have a status requested */
							inst = StatusBank_get_inst(&db, cur_inst->type, cur_inst->shortid);
							if (inst)
								{
									int nentry;

									if (!shortid_is_open(cur_inst))
										{
											char *dst;
											dst = dstfilename(cur_inst, srcfile, dstdir, timerange ? timespan : NULL);
											if (!shortid_fopen(cur_inst, dst))
												{
													perror(dst);
													exit(EXIT_FAILURE);
												}
											if (vf > 0 && dst) fprintf(stderr, "  creating %s\n", dst);
											srcused = 1;
											shortid_print_header(cur_inst, inst);
											free(dst);
										}

									nentry = shortid_print_value(cur_inst, inst);
									if (vf > 2) fprintf(stderr, "  %d entr%s\n", nentry, nentry != 1 ? "ies" : "y");
									if (ferror(cur_inst->stream))
										{
											char *dst;
											dst = dstfilename(cur_inst, srcfile, dstdir, timerange ? timespan : NULL);
											perror(dst);
											free(dst);
											exit(EXIT_FAILURE);
										}
								}
							cur_inst = cur_inst->next;
						}
					StatusBank_discard_all_inst(&db);
					if (canskip && !srcused)
						{
							if (vf > 1) fputs("  skipping because this file is not used.\n", stderr);
							break;
						}
					next_block:
						{
						}
				}

			/* close the input and output */
			next_file:
			{
				struct shortid_list_s* cur_inst;
				cur_inst = tree.next;
				while(cur_inst)
					{
						shortid_fclose(cur_inst);
						cur_inst = cur_inst->next;
					}
			}
			fclose(src);
		}

	free_shortids(&tree);
	free_time_zone();
	return EXIT_SUCCESS;
}
