#!/usr/bin/env ruby
#
# usage: ruby combine-tsc-log.rb [options] LOGFILE1 LOGFILE2 ...
#
# extract status from LOGFILE2, ... for times in each line in LOGFILE1,
# combine them together and display it into the stdout.
#
# format of logfiles should be: date\ttime\tdata....
#
# options: -w SECONDS: fetch data within target-SECONDS and target+SECONDS
#          -u        : use with -w: only uses each line once
#


class Line
	attr_reader :line, :ncols, :datastr, :type, :time
	NtimeCols = 2
	def initialize( line, tzsrc = 'UTC' )
		@line = line.chomp
		@cols = @line.split( /\t/ )
		@ncols = @cols.size - NtimeCols	# number of data columns not including time column
		if @ncols >= 0 then
			@type = /^#/ =~ @line ? :comment : :data
			@datastr = [ @cols[1], @cols[3..-1] ].join( "\t" )
		else
			@type = :other
			@datastr = ''
		end
		self.tz = tzsrc
	end

	def tz=(newtz)
		if :data == @type then
			t = @cols[0..1].join( " " ).split( /[ \-:]/ )
			fsec = t[5] ? Float( t[5] ) : 0
			t[5] = fsec.floor
			t[6] = t[5] ? (fsec*1e6 - t[5]*1e6) : 0
			begin
				if newtz && 'UTC' != newtz then
					tzorig = ENV['TZ']
				ENV['TZ'] = newtz
					@time = Time.local( *t )
					ENV['TZ'] = tzorig
				else
					@time = Time.gm( *t )
				end
			rescue ArgumentError
				$stderr.puts "Error in time: #{@cols[0..1].join( " " ).inspect}"
				raise
			end
		else
			@time = nil
		end
		return newtz
	end
end

class DummyLine < Line
	def initialize( ncols, dummydata = '*' )
		@cols = [ dummydata ] * ( ncols + Line::NtimeCols )
		@line = @cols.join( "\t" )
		@datastr = [ @cols[1], @cols[3..-1] ].join( "\t" )
		@type = :data
		@time = nil
	end

	def tz=(newtz)
		return newtz
	end
end

class ArrayWithDefault < Array
	def default=(x)
		@default = x
	end

	def at_or_default(index)
		if 0 <= index and index < self.size then
			self[index]
		else
			@default
		end
	end
end

class Log
	attr_reader :header

	def initialize( path, isprimary = false )
		@path = path
		@file = File.open( path )
		@ncols = nil	# number of data columns not including time column
		@dummy = nil	# line without a data
		@tz = nil
		@linebuf = Array.new
		@primary = isprimary
		@prevline = nil
		headerstr = ''
		otherstr = ''
		while( not @ncols and not @file.eof? )
			l = Line.new( @file.gets )
			case l.type
			when :comment
				headerstr += ( @primary ? l.line : l.datastr )
				@ncols = l.ncols
				@dummy = DummyLine.new( @ncols )
				if /time\(([^)]+)\)/i =~ l.line then
					@tz = $1
				end
			when :other
				otherstr += l.line + "\n" if @primary
			when :data
				@linebuf.push( l )
			end
		end
		@header = otherstr + headerstr
		@linebuf.each{ |l| l.tz = @tz }
	end

	def fetch
		r = nil
		until @file.eof?
			r = @linebuf.empty? ? Line.new( @file.gets, @tz ) : @linebuf.shift
			break if @primary or :data == r.type
		end
		@prevline = r
		return r
	end

	def find( time )	# find latest line assuming time-sorted lines

		# still in the past
		if @prevline and time <= @prevline.time then
			return @prevline
		end

		# peek into the future
		until @file.eof? and @linebuf.empty?
			# find the next data
			begin
				newline = @linebuf.empty? ? Line.new( @file.gets, @tz ) : @linebuf.shift
			end while not @file.eof? and not newline.time

			# no more data
			if not newline.time
				return @prevline || @dummy
			end
			
			# bingo
			if time < newline.time then
				@linebuf.push( newline )
				return @prevline || @dummy
			end

			# try next
			@prevline = newline
		end

		return @prevline
	end

	def within( time_range, uniq = false )	# return all lines within the time range
	# cannot be used in combination with fetch or find
		if not @lines_within or uniq
			@lines_within = ArrayWithDefault.new
			@lines_within.default = @dummy
		end

		# throw away lines no longer needed
		while @lines_within[0] and @lines_within[0].time <= time_range.first
			@lines_within.shift
		end

		# fetch lines
		until @file.eof?
			# find the next data
			begin
				newline = @linebuf.empty? ? Line.new( @file.gets, @tz ) : @linebuf.shift
			end while not @file.eof? and not newline.time
			next if newline.time < time_range.first

			# data within the range
			if time_range.include?( newline.time )
				@lines_within.push( newline )
				next
			end

			# already out of the range
			@linebuf.push( newline )
			break
		end
		return @lines_within
	end

	def close
		@file.close
	end

	def eof?
		@file.eof?
	end
end

require 'optparse'
parser = OptionParser.new
opts = Hash.new
opts[:w] = nil
opts[:u] = false
parser.banner = "#{File.basename( $0 )}: combines log files.\nusage: #{File.basename( $0 )} [optins] files..."
parser.on( '-h', '--help', 'prints help' ) {
	puts parser.help
	exit 0
}
parser.on( '-w', '--time-width SECONDS', 'fetch logs within SECONDS from target' ) { |v|
	opts[:w] = Float( v )
}
parser.on( '-u', '--uniq', 'only uses each line once (use with -w)' ) {
	opts[:u] = true
}
begin
	parser.parse!
rescue OptionParser::ParseError => err
	$stderr.puts err.message
	$stderr.puts parser.help
	exit 1
end

p = true	# first file will be primary
logs = ARGV.map{ |path| l = Log.new( path, p ); p = false; l }

# header
puts logs.map{ |l| l.header }.join( "\t" )

# data
until logs[0].eof?
	p = logs[0].fetch
	case p.type
	when :data
		if not p.time
			puts p.line
		elsif not opts[:w]
			s = logs[1..-1].map{ |l| l.find( p.time ) }
			puts [ p.line, s.map{ |l| l.datastr } ].join( "\t" )
		else
			range = ((p.time-opts[:w])..(p.time+opts[:w]))
			s = logs[1..-1].map{ |l| l.within( range, opts[:u] ) }
			([1] + s.map{ |a| a.size }).max.times do |i|
				puts [ p.line, s.map{ |l| l.at_or_default(i).datastr } ].join( "\t" )
			end
		end
	when :other
		puts p.line
	end
end

# clean up
logs.each do |l|
	l.close
end

