/* optimize_test.c : test suite for optimize.h and optimize.c */

static char *rcsid __attribute__((unused)) =
	"$Id: optimize_test.c,v 1.2 2004/10/27 20:50:43 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <float.h>
#include <getopt.h>
#include <signal.h>
#include <assert.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_multimin.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include "lens.h"
#include "version.h"
#include "optimize.h"

#define NBASELENS 2

int main( void );
void set_baseoptics( optics_minimizer_s *x );
void free_baseoptics( optics_minimizer_s *x );

/* sets *optics, star_obj, pupil_obj */
void set_baseoptics( optics_minimizer_s *x )
{
	int i;
	static lens_s lenses[NBASELENS], *l_ptr[NBASELENS];
	static lenses_s optics;
	inf_double_s star, pupil;
	inf_double_s img, mag;
	char *s;

	for( i = 0; i < NBASELENS; i++)
		{
			l_ptr[i] = &lenses[i];
		}

	to_lens( &lenses[0], 100, 300 );
	to_lens( &lenses[1], 200, 200 );

	lenses[0].feff_is_free = 0;
	lenses[0].thick_is_free = 0;
	lenses[1].feff_is_free = 1;
	lenses[1].thick_is_free = 1;

	optics.n = NBASELENS;
	optics.lenses = l_ptr;

	star = to_inf_double( 100 );
	pupil = to_inf_inf( 1 );

	x->optics = &optics;
	x->star_obj = star;
	x->pupil_obj = pupil;

	fputs( "base lens design set:\n", stderr );
	fprintf( stderr, " %s\n", ( s = lenses_inspect( &optics ) ) );
	if( s ) free( s );
	fprintf( stderr, "star : %s\n", ( s = lenses_params( &optics, star ) ) );
	if( s ) free( s );
	fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &optics, pupil ) ) );
	if( s ) free( s );

	img = lenses_img( &optics, star, -1 );
	mag = lenses_mag( &optics, star, -1 );
	assert( id_is_equal_to( img, to_inf_double( 0 ), 1e-5 ) );
	assert( id_is_equal_to( mag, to_inf_double( -2 ), 1e-5 ) );

	img = lenses_img( &optics, pupil, -1 );
	mag = lenses_mag( &optics, pupil, -1 );
	assert( id_is_equal_to( img, to_inf_inf( 1 ), 1e-5 ) );
	assert( id_is_equal_to( mag, to_inf_double( 0.5 ), 1e-5 ) );
}

void free_baseoptics( optics_minimizer_s *x )
{
	/* currently there is nothing to do */
}

int
main( void )
{
	display_version( "optimize_test", stderr );

	/* signal handler */
	{
		struct timespec req, rem;
		req.tv_sec = 1;
		req.tv_nsec = 500000000;
		is_alarm = 0;
		signal( SIGALRM, &sig_handle );
		alarm( 1 );
		assert( is_alarm == 0 );
		fputs( "testing SIGALRM. Wait for at most 1.5 seconds.\nSignal should be caught after 1 sec:\n", stderr );
		assert( nanosleep( &req, &rem ) == -1 );
		fprintf( stderr, "%.2g sec remaining\n", rem.tv_sec + rem.tv_nsec * 1e-9 );
		alarm( 0 );
		signal( SIGALRM, SIG_DFL );
		assert( is_alarm == 1 );
	}

	/* error strings */
	{
		char *msg;
		msg = optimization_status_to_s( lens_opt_success );
		assert( strcmp( msg, "no error" ) == 0 );
		msg = optimization_status_to_s( lens_opt_toomany );
		assert( strcmp( msg, "too many iterations" ) == 0 );
	}

	/* helper functions (except for signal handler) */
	{
		optics_minimizer_s x;
		gsl_vector *vect;
		int i;
		char *s;
		double badmf, goodmf;

		set_baseoptics( &x );
		fprintf( stderr, "free parameters: %d\n", n_free_pars( &x ) );
		assert( n_free_pars( &x ) == NBASELENS );
		assert( vect = gsl_vector_alloc( n_free_pars( &x ) ) );

		x.mf = NULL;
		x.mf = mf_operand_add( x.mf, "star_img", x.star_obj, -1, mf_img, to_inf_double( 0 ), -1, 1, NULL );
		x.mf = mf_operand_add( x.mf, "pupil_img", x.pupil_obj, -1, mf_img, to_inf_inf( 1 ), -1, 1, NULL );
		x.mf = mf_operand_add( x.mf, "star_mag", x.star_obj, -1, mf_mag, to_inf_double( -2 ), -1, 1, NULL );
		x.mf = mf_operand_add( x.mf, "pupil_mag", x.pupil_obj, -1, mf_mag, to_inf_inf( 0.5 ), -1, 1, NULL );

		goodmf = merit_function( &x );
		fprintf( stderr, "merit:%g\n", goodmf );

		fputs( "optics_to_vector....\n", stderr );
		optics_to_vector( &x, vect );
		for( i = 0; i < NBASELENS; i++ )
			{
				fprintf( stderr, "freepar%d:%g, ", i, gsl_vector_get( vect, i ) );
				switch( i )
					{
						case 0: assert( gsl_vector_get( vect, i ) == 200 ); break;
						case 1: assert( gsl_vector_get( vect, i ) == 200 ); break;
					}
			}
		fputc( '\n', stderr );

		fputs( "changing the optics slightly....\n", stderr );
		gsl_vector_set( vect, 0, 205 );
		vector_to_optics( &x, vect );

		fprintf( stderr, "star : %s\n", ( s = lenses_params( x.optics, x.star_obj ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( x.optics, x.pupil_obj ) ) );
		if( s ) free( s );
		optics_to_vector( &x, vect );
		for( i = 0; i < NBASELENS; i++ )
			{
				fprintf( stderr, "freepar%d:%g, ", i, gsl_vector_get( vect, i ) );
				switch( i )
					{
						case 0: assert( gsl_vector_get( vect, i ) == 205 ); break;
						case 1: assert( gsl_vector_get( vect, i ) == 200 ); break;
					}
			}
		fputc( '\n', stderr );
		badmf = merit_function( &x );
		fprintf( stderr, "merit:%g\n", badmf );
		assert( badmf > goodmf );

		fputs( "optimization ...\n", stderr );
		goodmf = minimize( &x, 10, 100, 10, 1e-5 );
		fprintf( stderr, "resulting merit function:%g\n", goodmf );
		fputs( "resulting optics:\n", stderr );
		fprintf( stderr, "   star : %s\n", ( s = lenses_params( x.optics, x.star_obj ) ) );
		if( s ) free( s );
		fprintf( stderr, "   pupil: %s\n", ( s = lenses_params( x.optics, x.pupil_obj ) ) );
		if( s ) free( s );

		fputs( "  ", stderr );
		for( i = 0; i < x.optics->n; i++)
			{
				fprintf( stderr, "[L%d]%s ", i, ( s = lens_inspect( x.optics->lenses[i] ) ) );
				if( s ) free( s );
			}
		fputc( '\n', stderr );

		assert( badmf > goodmf );
		assert( x.final_status == lens_opt_success );
		assert( id_is_equal_to( x.optics->lenses[0]->feff, to_inf_double( 100 ), 1e-5 ) );
		assert( id_is_equal_to( x.optics->lenses[0]->thick, to_inf_double( 300 ), 1e-5 ) );
		assert( id_is_equal_to( x.optics->lenses[1]->feff, to_inf_double( 200 ), 1e-5 ) );
		assert( id_is_equal_to( x.optics->lenses[1]->thick, to_inf_double( 200 ), 2e-5 ) );

		gsl_vector_free( vect );
		free_baseoptics( &x );
	}

	/* position_from surfaces */
	fputs( "--- testing position_from surfaces ---\n", stderr );
	{
		optics_minimizer_s x;
		int i;
		char *s, *t;

		static lens_s lenses[3], *l_ptr[3];
		static lenses_s optics;
		static inf_double_s l1_thick_pars[2], l2_thick_pars[2];

		/* lenses */
		x.optics = &optics;
		optics.n = 3;
		optics.lenses = l_ptr;
		for( i = 0; i < optics.n; i++ )	l_ptr[i] = &lenses[i];
		/* L0 */
		to_lens( &lenses[0], 100, 100 );
		lenses[0].feff_is_free = 1;
		lenses[0].thick_is_free = 1;
		/* L1 */
		to_lens( &lenses[1], 0, 0 );
		lenses[1].feff = to_inf_inf( 1 );
		lenses[1].thick_type = position_from;
		lenses[1].thick_pars = l1_thick_pars;
		l1_thick_pars[0] = to_inf_double( 0 );
		l1_thick_pars[1] = to_inf_double( 200 );
		/* L2 */
		to_lens( &lenses[2], 0, 0 );
		lenses[2].feff = to_inf_double( 100 );
		lenses[2].thick_type = position_from;
		lenses[2].thick_pars = l2_thick_pars;
		l2_thick_pars[0] = to_inf_double( 0 );
		l2_thick_pars[1] = to_inf_double( 300 );

		/* mf */
		x.final_status = lens_opt_notdone;
		x.final_merit_function = 0;
		x.star_obj = to_inf_double( 50 );
		x.pupil_obj = to_inf_inf( 1 );
		x.mf = NULL;
		x.mf = mf_operand_add( x.mf, "star_img", to_inf_double( 50 ), 0, mf_img, to_inf_double( 0 ), -1, 1, NULL );
		x.mf = mf_operand_add( x.mf, "pupil_img", to_inf_inf( 1 ), 1, mf_img, to_inf_inf( 1 ), -1, 1, NULL );
		x.mf = mf_operand_add( x.mf, "pupil_on_stop_img", to_inf_inf( 1 ), 1, mf_img, to_inf_double( 0 ), 0, 1, NULL );

		/* optimize */
		assert( lenses_reconfigure( x.optics ) == -1 );
		fprintf( stderr, "before minim. :%s\n", s = lenses_inspect( x.optics ) );
		if( s ) free( s );
		fprintf( stderr, "               L1 pos:%s L2 pos:%s\n",
			s = id_to_s( l1_thick_pars[1], "%g" ),
			t = id_to_s( l2_thick_pars[1], "%g" ) );
		if( s ) free( s );
		if( t ) free( t );

		minimize( &x, 0, 1000, 10, 1e-5 );

		fprintf( stderr, "after  minim. :%s\n", s = lenses_inspect( x.optics ) );
		if( s ) free( s );
		fprintf( stderr, "               L1 pos:%s L2 pos:%s\n",
			s = id_to_s( l1_thick_pars[1], "%g" ),
			t = id_to_s( l2_thick_pars[1], "%g" ) );
		if( s ) free( s );
		if( t ) free( t );
		assert( id_is_equal_to( l1_thick_pars[1], to_inf_double( 200 ), 1e-5 ) );
		assert( id_is_equal_to( l2_thick_pars[1], to_inf_double( 300 ), 1e-5 ) );

		assert( lenses_reconfigure( x.optics ) == -1 );
		fprintf( stderr, "after  reconf.:%s\n", s = lenses_inspect( x.optics ) );
		if( s ) free( s );
		fprintf( stderr, "               L1 pos:%s L2 pos:%s\n",
			s = id_to_s( l1_thick_pars[1], "%g" ),
			t = id_to_s( l2_thick_pars[1], "%g" ) );
		if( t ) free( t );
		if( s ) free( s );
		assert( id_is_equal_to( l1_thick_pars[1], to_inf_double( 200 ), 1e-5 ) );
		assert( id_is_equal_to( l2_thick_pars[1], to_inf_double( 300 ), 1e-5 ) );

		mf_operand_free( x.mf );
	}

	fputs( "\ntests successfully finished.\n", stderr );
	return EXIT_SUCCESS;
}
