/* steering_mirror.c : designs a steering mirror optics */

static char *rcsid __attribute__((unused)) =
	"$Id: optimize.c,v 1.1.1.1 2004/10/27 20:14:07 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <signal.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_multimin.h>
#include "lens.h"
#include "optimize.h"

/* global variables ... */
int is_alarm;	// stops the optimization routine if this is 1

/* helpers */
void
sig_handle( int signum )
{
	signal (SIGALRM, SIG_IGN);

	if( !is_alarm )
		{
			is_alarm = 1;
			switch( signum )
				{
					case SIGALRM:
						fputs ("... Spending too much time in a step, ", stderr);
						break;
					default:
						fprintf (stderr, "... Unknown signal caught: %d, ", signum);
				}
			fputs ("ending the loop after this step.\n", stderr);
		}
}

/* counts number of free pars */
int
n_free_pars( const optics_minimizer_s *params )
{
	int i, n;
	n = 0;
	for( i = 0; i < params->optics->n; i++ )
		{
			if( params->optics->lenses[i]->feff_is_free ) n++;
			if( params->optics->lenses[i]->thick_is_free ) n++;
		}
	return n;
}

/* updates the vector */
void
optics_to_vector( optics_minimizer_s *params, gsl_vector *v )
{
	int i, p;
	p = 0;
	lenses_reconfigure( params->optics );
	for( i = 0; i < params->optics->n; i++ )
		{
			if( params->optics->lenses[i]->feff_is_free )
				{
					switch( params->optics->lenses[i]->feff_type )
						{
							case normal_feff:
								gsl_vector_set( v, p, id_to_double( params->optics->lenses[i]->feff ) );
								break;
							default: assert( 0 );
						}
					p++;
				}
			if( params->optics->lenses[i]->thick_is_free )
				{
					switch( params->optics->lenses[i]->thick_type )
						{
							case normal_thick:
								gsl_vector_set( v, p, id_to_double( params->optics->lenses[i]->thick ) );
								break;
							case position_from:
								gsl_vector_set( v, p, id_to_double( params->optics->lenses[i]->thick_pars[1] ) );
								break;
							default: assert( 0 );
						}
					p++;
				}
		}
} 

/* updates the optics */
void
vector_to_optics( optics_minimizer_s *params, const gsl_vector *v )
{
	int i, p;
	p = 0;
	for( i = 0; i < params->optics->n; i++ )
		{
			if( params->optics->lenses[i]->feff_is_free )
				{
					switch( params->optics->lenses[i]->feff_type )
						{
							case normal_feff:
								params->optics->lenses[i]->feff = to_inf_double( gsl_vector_get( v, p ) );
								break;
							default: assert( 0 );
						}
					p++;
				}
			if( params->optics->lenses[i]->thick_is_free )
				{
					switch( params->optics->lenses[i]->thick_type )
						{
							case normal_thick:
								params->optics->lenses[i]->thick = to_inf_double( gsl_vector_get( v, p ) );
								break;
							case position_from:
								params->optics->lenses[i]->thick_pars[1] = to_inf_double( gsl_vector_get( v, p ) );
								break;
							default: assert( 0 );
						}
					p++;
				}
		}
}

/* returns a string showing the status */
char *
optimization_status_to_s( optimization_status_t status )
{
	char *r;
	switch( status )
		{
			case lens_opt_success: r = "no error"; break;
			case lens_opt_timeout: r = "spent too match time in a step"; break;
			case lens_opt_nomemory: r = "memory allocation error"; break;
			case lens_opt_unexpected: r = "unexpected error in GSL"; break;
			case lens_opt_toomany: r = "too many iterations"; break;
			case lens_opt_notdone: r = "optimization not done"; break;
			default: r = "unknown error"; assert( 0 ); break;
		}
	return r;
}


/* measure how we are near to the goal */
double
merit_function( const optics_minimizer_s *params )
{
	double r;
	merit_function_operands_s *cur;

	r = 0;
	cur = params->mf;
	while( cur )
		{
			r += mf_operand_delta( cur, params->optics );
			cur = cur->next;
		}

	return r;
}

/* weight sum of merit function */
double
total_weight( const optics_minimizer_s *params )
{
	double r;
	merit_function_operands_s *cur;

	r = 0;
	cur = params->mf;
	while( cur )
		{
			r += fabs( cur->weight );
			cur = cur->next;
		}

	return r;
}

/* GSL interface for the merit_function() */
double
gsl_merit_function( const gsl_vector *x, void *params )
{
	vector_to_optics( (optics_minimizer_s *) params, x );
	lenses_reconfigure( ((optics_minimizer_s *) params)->optics );
	return merit_function( (optics_minimizer_s *) params );
}

/* minimizes the merit function */
double
minimize( optics_minimizer_s *params, const int vf, const size_t maxiter, const unsigned int maxsteptime, const double epsabs )
{
	double r;	// return value

	size_t iter;	// number of iterations
	size_t nfree;	// number of free parameters
	int status;	// GSL minimization status
	int rval;	// GSL iteration status
	double ssval;	// GSL minimizer size
	const gsl_multimin_fminimizer_type *T = gsl_multimin_fminimizer_nmsimplex;
		// minimization algorithm to be used (no other choics for fminimzer)
	gsl_multimin_fminimizer *s = NULL;	// minimizer instance
	gsl_vector *x = NULL;	// free parameters
	gsl_vector *ss = NULL;	// vertex size vector
	gsl_multimin_function func;	// function to be minimized

	char *t;

	/* output initialization */
	r = DBL_MAX;
	params->final_status = lens_opt_success;
	params->final_merit_function = merit_function( params );

	/* memory allocation and vector initialization for GSL */
	nfree = n_free_pars( params );
	if( !( ss = gsl_vector_alloc( nfree ) ) )
		{
			if( vf > 0 )
				{
					fprintf( stderr, "%s:%d failed to allocate space for ss for %d free parameters.\n", __FILE__, __LINE__, nfree );
				}
			params->final_status = lens_opt_nomemory;
			return r;
		}
	gsl_vector_set_all( ss, 1.0 );
	if( !( x = gsl_vector_alloc( nfree ) ) )
		{
			gsl_vector_free( ss );
			if( vf > 0 )
				{
					fprintf( stderr, "%s:%d failed to allocate space for x for %d free parameters.\n", __FILE__, __LINE__, nfree );
				}
			params->final_status = lens_opt_nomemory;
			return r;
		}
	optics_to_vector( params, x );

	/* function settings */
	func.f = &gsl_merit_function;
	func.n = nfree;
	func.params = (void *) params;

	/* minimizer allocations */
	s = gsl_multimin_fminimizer_alloc( T, nfree );
	gsl_multimin_fminimizer_set( s, &func, x, ss );

	/* show initial status */
	if( vf > 0 ) fprintf( stderr, "Ini: MF:%-8.2e\n", merit_function( params ) );
	if( vf > 2 )
		{
			fputs( t = lenses_inspect( params->optics ), stderr );
			fputs( "\n", stderr );
			if( t ) free( t );
		}
	if( vf > 1 )
		{
			fprintf( stderr, "   star : %s\n", ( t = lenses_params( params->optics, params->star_obj ) ) );
			if( t ) free( t );
			fprintf( stderr, "   pupil: %s\n", ( t = lenses_params( params->optics, params->pupil_obj ) ) );
			if( t ) free( t );
		}

	/* iteration */
	iter = 0;
	rval = GSL_CONTINUE;
	status = GSL_SUCCESS;
	while( rval == GSL_CONTINUE )
		{
			/* iterate */
			iter++;
			if( iter > maxiter ) break;
			is_alarm = 0;
			alarm( maxsteptime );
			status = gsl_multimin_fminimizer_iterate( s );
			alarm( 0 );
			if( status ) break;

			/* check results */
			rval = gsl_multimin_test_size( gsl_multimin_fminimizer_size( s ), epsabs );
			ssval = gsl_multimin_fminimizer_size( s );
			if( vf > 0 )
				{
					fprintf( stderr, "%-3d: MF:%-8.2e ss:%-8.2e\n", iter, s->fval, ssval );
				}
			if( vf > 2 )
				{
					fputs( t = lenses_inspect( params->optics ), stderr );
					fputs( "\n", stderr );
					if( t ) free( t );
				}
			if( vf > 1 )
				{
					fprintf( stderr, "   star : %s\n", ( t = lenses_params( params->optics, params->star_obj ) ) );
					if( t ) free( t );
					fprintf( stderr, "   pupil: %s\n", ( t = lenses_params( params->optics, params->pupil_obj ) ) );
					if( t ) free( t );
				}
			if( is_alarm ) break;
		}
	vector_to_optics( params, s->x );
	params->final_merit_function = merit_function( params );
	r = params->final_merit_function;

	/* check iteration results */
	if( iter > maxiter )
		{
			params->final_status = lens_opt_toomany;
			if( vf > 0 ) fputs( "too many iterations.\n", stderr );
		}
	else
		{
			if( status )
				{
					params->final_status = lens_opt_unexpected;
					if( vf > 0 ) fputs( "unexpected error in GSL.\n", stderr );
				}
			else
				{
					if( is_alarm )
						{
							params->final_status = lens_opt_timeout;
							if( vf > 0 ) fputs( "spent too match time in a step.\n", stderr );
						}
					else
						{
							if( rval == GSL_SUCCESS )
								{
									params->final_status = lens_opt_success;
									if( vf > 1 ) fputs( "iteration succesfully converged.\n", stderr );
								}
							else
								{
									fprintf( stderr, "%s:%d unexpected situation!\n", __FILE__, __LINE__ );
									exit( EXIT_FAILURE );
								}
						}
				}
		}

	/* free the memories */
	if( ss ) gsl_vector_free( ss );
	if( x ) gsl_vector_free( x );

	return r;
}

/* adds an operand. orig can be NULL */
merit_function_operands_s *
mf_operand_add( merit_function_operands_s *orig, const char *name, inf_double_s object, int object_number, mf_operand_t type, inf_double_s goal, int surface, double weight, char *comment )
{
	merit_function_operands_s *r, *new;

	/* memory allocation */
	if( orig )
		{
			r = orig;
			new = r;
			while( new->next ) new = new->next;
			new->next = (merit_function_operands_s *) malloc( sizeof( merit_function_operands_s ) );
			new = new->next;
		}
	else
		{
			r = (merit_function_operands_s *) malloc( sizeof( merit_function_operands_s ) );
			new = r;
		}
	if( !new ) return NULL;

	/* initiazliation */
	new->name = strdup( name );
	new->object = object;
	new->object_number = object_number;
	new->type = type;
	new->goal = goal;
	new->surface = surface;
	new->weight = weight;
	new->comment = comment;
	new->next = NULL;

	return r;
}

/* merit function for the current operand */
inf_double_s
mf_operand_value( const merit_function_operands_s *cur, const lenses_s *optics )
{
	switch( cur->type )
		{
			case mf_img:
				return lenses_img( optics, cur->object, cur->surface );
			case mf_mag:
				return lenses_mag( optics, cur->object, cur->surface );
			case mf_amag:
				return id_abs( lenses_mag( optics, cur->object, cur->surface ) );
			case mf_none:
				return to_inf_double( 0 );
			default:
				assert( 0 );
		}
	return to_inf_double( 0 );
}

double
mf_operand_delta( const merit_function_operands_s *cur, const lenses_s *optics )
{
	double d;
	inf_double_s current;

	if( cur->weight <= 0 || cur->type == mf_none ) return 0;

	current = mf_operand_value( cur, optics );
	if( id_is_finite( cur->goal ) )
		{
			d = id_to_double( current ) - id_to_double( cur->goal );
		}
	else
		{
			d = id_to_double( id_reciprocal( current ) ) - id_to_double( id_reciprocal( cur->goal ) );
		}
	return d * d * cur->weight;
}

/* frees all the operands in the list */
void
mf_operand_free( merit_function_operands_s *start )
{
	merit_function_operands_s *cur, *next;
	cur = start;
	while( cur )
		{
			next = cur->next;
			if( cur->comment) free( cur->comment );
			free( cur );
			cur = next;
		}
}
