/* lenscalc.c : calculates and optimizes a paraxial optics */

static char *rcsid __attribute__((unused)) =
	"$Id: lenscalc.c,v 1.2 2004/10/27 20:50:43 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <libgen.h>
#include <string.h>
#include "lens.h"
#include "optimize.h"
#include "lens.h"
#include "conffile.h"
#include "version.h"

#define DEF_FILENAME "-"
#define DEF_MAX_ITER 1000
#define DEF_MAX_TIME 10
#define DEF_ABS_EPS 1e-10
#define DEF_VF 1

/* command line options */
#define OPTIONS "o:i:t:e:nvqhHV"
#define OPTIONS_LONG {\
	{"output", 1, NULL, 'o'},\
	{"max-iteration", 1, NULL, 'i'},\
	{"max-step-time", 1, NULL, 't'},\
	{"absolute-epsilon", 1, NULL, 'e'},\
	{"no-optimization", 0, NULL, 'n'},\
	{"verbose", 0, NULL, 'v'},\
	{"quiet", 0, NULL, 'q'},\
	{"help", 0, NULL, 'h'},\
	{"version", 0, NULL, 'V'},\
	{NULL, 0, NULL, 0}\
}

typedef struct
{
	char *infilename;
	char *outfilename;
	size_t maxiter;
	unsigned int maxsteptime;
	double epsabs;
	int vf;
	int optimize;
} lenscalc_config_s;

int main( int argc, char *argv[] );
int lenscalc_get_options( int argc, char *argv[], lenscalc_config_s *config );
void usage_long( char *argv[], FILE *stream );
void usage_short( char *argv[], FILE *stream );
void usage_and_options( char *argv[], FILE *stream, int isdesc );
void version( char *argv[], FILE *stream );

void
version( char *argv[], FILE *stream )
{
	char *cmdname, *path;
	path = strdup( argv[0] );
	cmdname = basename( path );
	if( !cmdname )
		{
			perror( "basename" );
			exit( EXIT_FAILURE );
		}

	display_version( cmdname, stream );
	free( path );
}

void
usage_long( char *argv[], FILE *stream )
{
	usage_and_options( argv, stream, 1 );
	fputs( "\n", stream );
	lenses_file_usage( stream );
}

void
usage_short( char *argv[], FILE *stream )
{
	usage_and_options( argv, stream, 0 );
}

void
usage_and_options( char *argv[], FILE *stream, int isdesc )
{
	char *cmdname, *path;
	path = strdup( argv[0] );
	cmdname = basename( path );
	if( !cmdname )
		{
			perror( "basename" );
			exit( EXIT_FAILURE );
		}

	/* description of the program */
	fprintf( stream, "usage: %s [options] input_lens_file\n\n", cmdname );

	if( isdesc ) fputs("\
This program calculates a paraxial optical system as specified in the\n\
lens file with a certain number of lenses. Namely, it measures image\n\
distance of the star and the pupil from the last surface, and spatial or\n\
angular magnification for the star and the pupil.\n\n", stream );

	fprintf( stream, "\
options [defaults]:\n\
	-o, --output=output_lens_file : output file [stdout]\n\
	-i, --max-iteration=N : limits number of iterations [%d]\n\
	-t, --max-step-time=S : limits duration of an iteration step [%d]\n\
	-e, --absolute-epsilon=x : absolute epsilon [%g]\n\
	-n, --no-optimization : disables optimization\n\
\n\
	-v, --verbose : increases verbosity\n\
	-q, --quiet : decreases verbosity\n\
	-h : shows short description\n\
	-H, --help : shows long description\n\
	-V, --version : displays the version\n\
", DEF_MAX_ITER, DEF_MAX_TIME, DEF_ABS_EPS );

	free( path );
}

int
lenscalc_get_options( int argc, char *argv[], lenscalc_config_s *config )
{
	int c, this_optind;
	static struct option long_options[] = OPTIONS_LONG;
	config->infilename = DEF_FILENAME;
	config->outfilename = DEF_FILENAME;
	config->maxiter = DEF_MAX_ITER;
	config->maxsteptime = DEF_MAX_TIME;
	config->epsabs = DEF_ABS_EPS;
	config->optimize = 1;
	config->vf = DEF_VF;

	while( 1 )
		{
			this_optind = optind ? optind : 1;
			c = getopt_long( argc, argv, OPTIONS, long_options, NULL );
			if( c == -1 )
				break;
			switch( c )
				{
					case 'o':
						config->outfilename = optarg;
						break;
					case 'i':
						config->maxiter = atol( optarg );
						break;
					case 't':
						config->maxsteptime = atoi( optarg );
						break;
					case 'e':
						config->epsabs = atof( optarg );
						break;
					case 'n':
						config->optimize = 0;
						break;
					case 'v':
						config->vf++;
						break;
					case 'q':
						config->vf--;
						break;
					case 'V':
						version( argv, stdout );
						exit( EXIT_SUCCESS );
					case 'h':
						usage_short( argv, stdout );
						exit( EXIT_SUCCESS );
					case 'H':
						usage_long( argv, stdout );
						exit( EXIT_SUCCESS );
					case '?':
						usage_short( argv, stderr );
						exit( EXIT_FAILURE );
					default:
						fprintf( stderr, "oops: missing implementation for option: %s\n", argv[this_optind] );
						exit( EXIT_FAILURE );
				}
		}

	return optind;
}

int
main( int argc, char *argv[] )
{
	int argind;
	lenscalc_config_s config;
	optics_minimizer_s *lenses;
	int surf;

	char *cmdname, *path;
	path = strdup( argv[0] );
	cmdname = basename( path );
	if( !cmdname )
		{
			perror( "basename" );
			exit( EXIT_FAILURE );
		}

	argind = lenscalc_get_options( argc, argv, &config );
	if( argind < argc )
		{
			config.infilename = argv[argind];
		}
	else
		{
			usage_short( argv, stderr );
			fputs( "\ninput_lens_file is missing.\n", stderr );
			exit( EXIT_FAILURE );
		}

	if( config.vf > 1 ) fprintf( stderr, "%s: input file:%s output file:%s\n", cmdname, config.infilename, config.outfilename );

	/* read the input */
	lenses = lenses_read( config.infilename );
	if( !lenses ) exit( EXIT_FAILURE );
	surf = lenses_reconfigure( lenses->optics );
	if( surf != -1 )
		{
			fprintf( stderr, "%s: error in the parameter of surface %d in %s.\n", cmdname, surf, config.infilename );
			exit( EXIT_FAILURE );
		}

	/* view the optics */
	if( config.vf > 0 || n_free_pars( lenses ) == 0 )
		{
			char *msg;
			msg = lenses_inspect( lenses->optics );
			if( !msg ) exit( EXIT_FAILURE );
			fprintf( stderr, "%s: %s\n", config.infilename, msg );
			free( msg );
			msg = lenses_params( lenses->optics, lenses->star_obj );
			if( !msg ) exit( EXIT_FAILURE );
			fprintf( stderr, "\t star: %s\n", msg );
			free( msg );
			msg = lenses_params( lenses->optics, lenses->pupil_obj );
			if( !msg ) exit( EXIT_FAILURE );
			fprintf( stderr, "\tpupil: %s\n", msg );
			free( msg );
		}

	if( config.optimize && n_free_pars( lenses ) > 0 && total_weight( lenses ) > 0 )
		{
			/* view the initial status */
			if( config.vf > 0 )
				{
					fprintf( stderr, "initial merit function: %g\n", merit_function( lenses ) );
				}

			/* optimize */
			if( config.vf > 0 ) fputs( "optimizing...\n", stderr );
			minimize( lenses, config.vf - 1, config.maxiter, config.maxsteptime, config.epsabs );
			if( lenses->final_status ) fprintf( stderr, "warning: %s\n", optimization_status_to_s( lenses->final_status ) );

			/* view the optics */
			if( config.vf > 0 )
				{
					char *msg;
					fprintf( stderr, "resulting merit function: %g\n", lenses->final_merit_function );
					msg = lenses_inspect( lenses->optics );
					if( !msg ) exit( EXIT_FAILURE );
					fprintf( stderr, "optimized: %s\n", msg );
					free( msg );
					msg = lenses_params( lenses->optics, lenses->star_obj );
					if( !msg ) exit( EXIT_FAILURE );
					fprintf( stderr, "\t star: %s\n", msg );
					free( msg );
					msg = lenses_params( lenses->optics, lenses->pupil_obj );
					if( !msg ) exit( EXIT_FAILURE );
					fprintf( stderr, "\tpupil: %s\n", msg );
					free( msg );
				}

			/* output */
			if( lenses_write( lenses, config.outfilename ) ) exit( EXIT_FAILURE );
		}
	else
		{
			if( config.vf > 0 && lenses->mf )
				{
					fputs( "\n", stdout );
					lenses_write( lenses, "-" );
				}
		}

	free( path );
	lenses_free( lenses );
	return EXIT_SUCCESS;
}
