/* lens_test.c : test suite for lens.h and lens.c */

static char *rcsid __attribute__((unused)) =
	"$Id: lens_test.c,v 1.1.1.1 2004/10/27 20:14:07 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <float.h>
#include "lens.h"
#include "version.h"
#include "mallocprintf.h"

int main( void );

int
main( void )
{

	display_version( "lens_test", stderr );
	fputs( "\ntesting...\n", stderr );

	/* inf_double_s */
	{
		inf_double_s a, b, c;
		char *s, *t, *u;	// for inspection

		/* to_inf_double */
		a = to_inf_double( 1 );
		fprintf( stderr, "1=%g, ", a.v );
		assert( a.v == 1 );
		assert( a.inf == 0 );
		a = to_inf_double( DBL_MAX );
		fprintf( stderr, "DBL_MAX => %s, ", ( s = id_inspect( a ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( a ) );
		a = to_inf_double( -DBL_MAX );
		fprintf( stderr, "-DBL_MAX => %s\n", ( s = id_inspect( a ) ) );
		if( s ) free( s );
		assert( id_is_m_inf( a ) );

		/* to_inf_inf, id_inspect, and id_to_s */
		a = to_inf_inf( -2 );
		fprintf( stderr, "-inf => %s,%s, ", ( s = id_inspect( a ) ), ( t = id_to_s( a, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		assert( a.v == 0 );
		assert( a.inf == -1 );
		a = to_inf_inf( 0 );
		fprintf( stderr, "0 => %s,%s, ", ( s = id_inspect( a ) ), ( t = id_to_s( a, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		assert( a.v == 0 );
		assert( a.inf == 0 );
		a = to_inf_inf( 2 );
		fprintf( stderr, "+inf => %s,%s\n", ( s = id_inspect( a ) ), ( t = id_to_s( a, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		assert( a.v == 0 );
		assert( a.inf == 1 );

		/* str_to_inf_double */
		{
			a = to_inf_double( 123.4 );
			t = id_to_s( a, "%g" );
			s = mallocprintf( "%sTRAILING", t );
			if( t ) free( t );
			fprintf( stderr, "%s => %s => ", t = id_to_s( a, "%g" ), s );
			if( t ) free( t );
			b = str_to_inf_double( s, &u );
			assert( id_is_equal_to( a, b, 1e-10 ) );
			assert( u > s );
			fprintf( stderr, "%s + %s, ", t = id_to_s( b, "%g" ), u );
			assert( strcmp( u, "TRAILING" ) == 0 );
			if( s ) free( s );
			if( t ) free( t );
		}
		{
			a = to_inf_double( -123.4 );
			t = id_to_s( a, "%g" );
			s = mallocprintf( "%sTRAILING", t );
			if( t ) free( t );
			fprintf( stderr, "%s => %s => ", t = id_to_s( a, "%g" ), s );
			if( t ) free( t );
			b = str_to_inf_double( s, &u );
			assert( id_is_equal_to( a, b, 1e-10 ) );
			assert( u > s );
			fprintf( stderr, "%s + %s, ", t = id_to_s( b, "%g" ), u );
			assert( strcmp( u, "TRAILING" ) == 0 );
			if( s ) free( s );
			if( t ) free( t );
		}
		{
			a = to_inf_inf( -1 );
			t = id_to_s( a, "%g" );
			s = mallocprintf( "%sTRAILING", t );
			if( t ) free( t );
			fprintf( stderr, "%s => %s => ", t = id_to_s( a, "%g" ), s );
			if( t ) free( t );
			b = str_to_inf_double( s, &u );
			assert( id_is_equal_to( a, b, 1e-10 ) );
			assert( u > s );
			fprintf( stderr, "%s + %s, ", t = id_to_s( b, "%g" ), u );
			assert( strcmp( u, "TRAILING" ) == 0 );
			if( s ) free( s );
			if( t ) free( t );
		}
		{
			a = to_inf_inf( 1 );
			t = id_to_s( a, "%g" );
			s = mallocprintf( "%sTRAILING", t );
			if( t ) free( t );
			fprintf( stderr, "%s => %s => ", t = id_to_s( a, "%g" ), s );
			if( t ) free( t );
			b = str_to_inf_double( s, &u );
			assert( id_is_equal_to( a, b, 1e-10 ) );
			assert( u > s );
			fprintf( stderr, "%s + %s, ", t = id_to_s( b, "%g" ), u );
			assert( strcmp( u, "TRAILING" ) == 0 );
			if( s ) free( s );
			if( t ) free( t );
		}

		/* to double */
		a = to_inf_double( -1 );
		assert( id_to_double( a ) == -1 );
		a = to_inf_double( 0 );
		assert( id_to_double( a ) == 0 );
		a = to_inf_double( 1 );
		assert( id_to_double( a ) == 1 );
		a = to_inf_inf( -1 );
		fprintf( stderr, "-inf => %s,%g\n", ( s = id_to_s( a, "%g" ) ), id_to_double( a ) );
		assert( id_to_double( a ) == -DBL_MAX );
		if( s ) free( s );
		a = to_inf_inf( 1 );
		fprintf( stderr, "inf => %s,%g\n", ( s = id_to_s( a, "%g" ) ), id_to_double( a ) );
		assert( id_to_double( a ) == DBL_MAX );
		if( s ) free( s );

		/* id_is_zero */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_zero( a ) ? "zero" : "not zero" );
		assert( !id_is_zero( a ) );
		a = to_inf_double( 0 );
		fprintf( stderr, "0 is %s, ", id_is_zero( a ) ? "zero" : "not zero" );
		assert( id_is_zero( a ) );
		a = to_inf_inf( 1 );
		fprintf( stderr, "+inf is %s, ", id_is_zero( a ) ? "zero" : "not zero" );
		assert( !id_is_zero( a ) );
		a = to_inf_inf( -1 );
		fprintf( stderr, "-inf is %s\n", id_is_zero( a ) ? "zero" : "not zero" );
		assert( !id_is_zero( a ) );

		/* id_is_p_inf */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_p_inf( a ) ? "+inf" : "not +inf" );
		assert( !id_is_p_inf( a ) );
		a = to_inf_double( 0 );
		fprintf( stderr, "0 is %s, ", id_is_p_inf( a ) ? "+inf" : "not +inf" );
		assert( !id_is_p_inf( a ) );
		a = to_inf_double( -1 );
		fprintf( stderr, "-1 is %s, ", id_is_p_inf( a ) ? "+inf" : "not +inf" );
		assert( !id_is_p_inf( a ) );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0 is %s, ", id_is_p_inf( b ) ? "+inf" : "not +inf" );
		assert( id_is_p_inf( b ) );
		c = id_invert( b );
		fprintf( stderr, "-1/0 is %s\n", id_is_p_inf( c ) ? "+inf" : "not +inf" );
		assert( !id_is_p_inf( c ) );

		/* id_is_m_inf */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_m_inf( a ) ? "-inf" : "not -inf" );
		assert( !id_is_m_inf( a ) );
		a = to_inf_double( 0 );
		fprintf( stderr, "0 is %s, ", id_is_m_inf( a ) ? "-inf" : "not -inf" );
		assert( !id_is_m_inf( a ) );
		a = to_inf_double( -1 );
		fprintf( stderr, "-1 is %s, ", id_is_m_inf( a ) ? "-inf" : "not -inf" );
		assert( !id_is_m_inf( a ) );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0 is %s, ", id_is_m_inf( b ) ? "-inf" : "not -inf" );
		assert( !id_is_m_inf( b ) );
		c = id_invert( b );
		fprintf( stderr, "-1/0 is %s\n", id_is_m_inf( c ) ? "-inf" : "not -inf" );
		assert( id_is_m_inf( c ) );

		/* id_is_finite */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_finite( a ) ? "finite" : "not finite" );
		assert( id_is_finite( a ) );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0 is %s\n", id_is_finite( b ) ? "finite" : "not finite" );
		assert( !id_is_finite( b ) );

		/* id_is_plus */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_plus( a ) ? "plus" : "not plus" );
		assert( id_is_plus( a ) );
		a = to_inf_double( 0 );
		fprintf( stderr, "0 is %s, ", id_is_plus( a ) ? "plus" : "not plus" );
		assert( id_is_plus( a ) );
		a = to_inf_double( -1 );
		fprintf( stderr, "-1 is %s, ", id_is_plus( a ) ? "plus" : "not plus" );
		assert( !id_is_plus( a ) );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0 is %s, ", id_is_plus( b ) ? "plus" : "not plus" );
		assert( id_is_plus( b ) );
		c = id_invert( b );
		fprintf( stderr, "-1/0 is %s\n", id_is_plus( c ) ? "plus" : "not plus" );
		assert( !id_is_plus( c ) );

		/* id_is_minus */
		a = to_inf_double( 1 );
		fprintf( stderr, "1 is %s, ", id_is_minus( a ) ? "minus" : "not minus" );
		assert( !id_is_minus( a ) );
		a = to_inf_double( 0 );
		fprintf( stderr, "0 is %s, ", id_is_minus( a ) ? "minus" : "not minus" );
		assert( !id_is_minus( a ) );
		a = to_inf_double( -1 );
		fprintf( stderr, "-1 is %s, ", id_is_minus( a ) ? "minus" : "not minus" );
		assert( id_is_minus( a ) );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0 is %s, ", id_is_minus( b ) ? "minus" : "not minus" );
		assert( !id_is_minus( b ) );
		c = id_invert( b );
		fprintf( stderr, "-1/0 is %s\n", id_is_minus( c ) ? "minus" : "not minus" );
		assert( id_is_minus( c ) );

		/* sign */
		a = to_inf_double( 1 );
		fprintf( stderr, "sign of 1 is %d, ", id_sign( a ) );
		assert( id_sign( a ) == 1 );
		a = to_inf_double( 0 );
		fprintf( stderr, "sign of 0 is %d, ", id_sign( a ) );
		assert( id_sign( a ) == 0 );
		a = to_inf_double( -1 );
		fprintf( stderr, "sign of -1 is %d, ", id_sign( a ) );
		assert( id_sign( a ) == -1 );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		fprintf( stderr, "sign of 1/0 is %d, ", id_sign( b ) );
		assert( id_sign( b ) == 1 );
		c = id_invert( b );
		fprintf( stderr, "sign of -1/0 is %d\n", id_sign( c ) );
		assert( id_sign( c ) == -1 );

		/* id_is_equal_to */
		fputs( "id_is_equal_to...\n", stderr );
		a = to_inf_double( 0 );
		b = to_inf_double( 0 );
		assert( id_is_equal_to( a, b, 0 ) );
		b = to_inf_double( 1 );
		assert( !id_is_equal_to( a, b, 0 ) );
		b = to_inf_inf( -1 );
		assert( !id_is_equal_to( a, b, 0 ) );
		b = to_inf_inf( 0 );
		assert( id_is_equal_to( a, b, 0 ) );
		b = to_inf_inf( 1 );
		assert( !id_is_equal_to( a, b, 0 ) );
		a = to_inf_inf( 1 );
		b = to_inf_inf( -1 );
		assert( !id_is_equal_to( a, b, 0 ) );
		b = to_inf_inf( 0 );
		assert( !id_is_equal_to( a, b, 0 ) );
		b = to_inf_inf( 1 );
		assert( id_is_equal_to( a, b, 0 ) );

		/* id_invert */
		a = to_inf_double( 0 );
		b = id_invert( a );
		fprintf( stderr, "-0 = %g, ", b.v );
		assert( id_is_zero( b ) );
		a = to_inf_double( 1 );
		b = id_invert( a );
		fprintf( stderr, "-1 = %g, ", b.v );
		assert( b.v == -1 );
		c = id_invert( b );
		fprintf( stderr, "-(-1) = %g, ", c.v );
		assert( c.v == 1 );
		a = to_inf_double( 0 );
		b = id_reciprocal( a );
		c = id_invert( b );
		fprintf( stderr, "-(1/0) is %s, ", id_is_minus( c ) ? "minus" : "not minus" );
		assert( id_is_minus( c ) );
		c = id_invert( c );
		fprintf( stderr, "-(-(1/0)) is %s\n", id_is_plus( c ) ? "plus" : "not plus" );
		assert( id_is_plus( c ) );

		/* id_reciprocal */
		a = to_inf_double( 1 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/1 = %g, ", b.v );
		assert( b.v == 1 );
		a = to_inf_double( 2 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/2 = %g, ", b.v );
		assert( b.v == 0.5 );
		a = to_inf_double( 0.5 );
		b = id_reciprocal( a );
		fprintf( stderr, "1/0.5 = %g\n", b.v );
		assert( b.v == 2 );

		/* id_add */
		a = to_inf_double( 1 );
		b = to_inf_double( 1 );
		c = id_add( a, b );
		fprintf( stderr, "1+1 = %g, ", c.v );
		assert( c.v == 2 );
		a = to_inf_double( 1 );
		b = to_inf_double( -1 );
		c = id_add( a, b );
		fprintf( stderr, "1+(-1) = %g, ", c.v );
		assert( c.v == 0 );
		a = to_inf_double( 0 );
		a = id_reciprocal( a );
		c = id_add( a, a );
		fprintf( stderr, "1/0+1/0 is %s, ", id_is_p_inf( c ) ? "+inf" : "not +inf" );
		assert( id_is_p_inf( c ) );
		b = id_invert( a );
		c = id_add( a, b );
		fprintf( stderr, "1/0+(-1/0) is %s\n", id_is_zero( c ) ? "zero" : "not zero" );
		assert( id_is_zero( c ) );

		/* id_sub */
		a = to_inf_double( 1 );
		b = to_inf_double( 1 );
		c = id_sub( a, b );
		fprintf( stderr, "1-1 = %g, ", c.v );
		assert( c.v == 0 );
		a = to_inf_double( 1 );
		b = to_inf_double( -1 );
		c = id_sub( a, b );
		fprintf( stderr, "1-(-1) = %g, ", c.v );
		assert( c.v == 2 );
		a = to_inf_double( 0 );
		a = id_reciprocal( a );
		c = id_sub( a, a );
		fprintf( stderr, "1/0-1/0 is %s, ", id_is_zero( c ) ? "zero" : "not zero" );
		assert( id_is_zero( c ) );
		b = id_invert( a );
		c = id_sub( a, b );
		fprintf( stderr, "1/0-(-1/0) is %s\n", id_is_p_inf( c ) ? "+inf" : "not +inf" );
		assert( id_is_p_inf( c ) );

		/* id_mul */
		a = to_inf_double( 1 );
		b = to_inf_double( 1 );
		c = id_mul( a, b );
		fprintf( stderr, "1*1 = %g, ", c.v );
		assert( c.v == 1 );
		a = to_inf_double( 2 );
		b = to_inf_double( 3 );
		c = id_mul( a, b );
		fprintf( stderr, "2*3 = %g, ", c.v );
		assert( c.v == 6 );
		a = to_inf_double( 1 );
		b = to_inf_double( -1 );
		c = id_mul( a, b );
		fprintf( stderr, "1*(-1) = %g, ", c.v );
		assert( c.v == -1 );
		a = to_inf_inf( 1 );
		c = id_mul( a, a );
		fprintf( stderr, "+inf*+inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_inf( -1 );
		c = id_mul( a, b );
		fprintf( stderr, "+inf*-inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_m_inf( c ) );
		a = to_inf_inf( -1 );
		c = id_mul( a, b );
		fprintf( stderr, "-inf*-inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_double( 1 );
		c = id_mul( a, b );
		fprintf( stderr, "+inf*1 => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_double( 0 );
		c = id_mul( a, b );
		fprintf( stderr, "+inf*0 => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_zero( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_double( -1 );
		c = id_mul( a, b );
		fprintf( stderr, "+inf*-1 => %s\n", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_m_inf( c ) );

		/* id_div */
		a = to_inf_double( 1 );
		b = to_inf_double( 1 );
		c = id_div( a, b );
		fprintf( stderr, "1/1 = %g, ", c.v );
		assert( c.v == 1 );
		a = to_inf_double( 2 );
		b = to_inf_double( 3 );
		c = id_div( a, b );
		fprintf( stderr, "2/3 = %g, ", c.v );
		assert( c.v == 2.0/3.0 );
		a = to_inf_double( 1 );
		b = to_inf_double( -1 );
		c = id_div( a, b );
		fprintf( stderr, "1/(-1) = %g, ", c.v );
		assert( c.v == -1 );
		a = to_inf_inf( 1 );
		c = id_div( a, a );
		fprintf( stderr, "+inf/+inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_finite( c ) );
		assert( c.v == 1 );
		a = to_inf_inf( 1 );
		b = to_inf_inf( -1 );
		c = id_div( a, b );
		fprintf( stderr, "+inf/-inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_finite( c ) );
		assert( c.v == -1 );
		a = to_inf_inf( -1 );
		c = id_div( a, b );
		fprintf( stderr, "-inf/-inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_finite( c ) );
		assert( c.v == 1 );
		a = to_inf_inf( 1 );
		b = to_inf_double( 1 );
		c = id_div( a, b );
		fprintf( stderr, "+inf/1 => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_double( 0 );
		c = id_div( a, b );
		fprintf( stderr, "+inf/0 => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_p_inf( c ) );
		a = to_inf_inf( 1 );
		b = to_inf_double( -1 );
		c = id_div( a, b );
		fprintf( stderr, "+inf/-1 => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_m_inf( c ) );
		a = to_inf_double( 1 );
		b = to_inf_inf( 1 );
		c = id_div( a, b );
		fprintf( stderr, "1/+inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_zero( c ) );
		a = to_inf_double( 0 );
		b = to_inf_inf( 1 );
		c = id_div( a, b );
		fprintf( stderr, "0/+inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_zero( c ) );
		a = to_inf_double( -1 );
		b = to_inf_inf( 1 );
		c = id_div( a, b );
		fprintf( stderr, "-1/+inf => %s, ", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_zero( c ) );
		a = to_inf_double( -1 );
		b = to_inf_inf( -1 );
		c = id_div( a, b );
		fprintf( stderr, "-1/-inf => %s\n", ( s = id_inspect( c ) ) );
		if( s ) free( s );
		assert( id_is_zero( c ) );
	}

	/* lens_s */
	{
		lens_s l;
		char *s, *t, *u;
		inf_double_s o, i, m;

		to_lens( &l, 100, 0 );
		fprintf( stderr, "Lens f:100 t:0 => %s\n", (s = lens_inspect( &l ) ) );
		if( s ) free( s );

		o = to_inf_inf( 1 );
		i = lens_img( &l, o );
		m = lens_mag( &l, o );
		fprintf( stderr, "obj:%s => img:%s,mag:%s\n", ( s = id_to_s( o, "%g" ) ), ( t = id_to_s( i, "%g" ) ), ( u = id_to_s( m, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		if( u ) free( u );
		assert( id_is_finite( i ) );
		assert( id_is_finite( m ) );
		assert( i.v == 100 );
		assert( m.v == 0 );

		o = to_inf_double( 200 );
		i = lens_img( &l, o );
		m = lens_mag( &l, o );
		fprintf( stderr, "obj:%s => img:%s,mag:%s\n", ( s = id_to_s( o, "%g" ) ), ( t = id_to_s( i, "%g" ) ), ( u = id_to_s( m, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		if( u ) free( u );
		assert( id_is_finite( i ) );
		assert( id_is_finite( m ) );
		assert( i.v == 200 );
		assert( m.v == 1 );

		o = to_inf_double( 100 );
		i = lens_img( &l, o );
		m = lens_mag( &l, o );
		fprintf( stderr, "obj:%s => img:%s,mag:%s\n", ( s = id_to_s( o, "%g" ) ), ( t = id_to_s( i, "%g" ) ), ( u = id_to_s( m, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		if( u ) free( u );
		assert( id_is_p_inf( i ) );
		assert( id_is_p_inf( m ) );

		o = to_inf_double( 50 );
		i = lens_img( &l, o );
		m = lens_mag( &l, o );
		fprintf( stderr, "obj:%s => img:%s,mag:%s\n", ( s = id_to_s( o, "%g" ) ), ( t = id_to_s( i, "%g" ) ), ( u = id_to_s( m, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		if( u ) free( u );
		assert( id_is_finite( i ) );
		assert( id_is_finite( m ) );
		assert( i.v == -100 );
		assert( m.v == -2 );

		o = to_inf_double( 0 );
		i = lens_img( &l, o );
		m = lens_mag( &l, o );
		fprintf( stderr, "obj:%s => img:%s,mag:%s\n", ( s = id_to_s( o, "%g" ) ), ( t = id_to_s( i, "%g" ) ), ( u = id_to_s( m, "%g" ) ) );
		if( s ) free( s );
		if( t ) free( t );
		if( u ) free( u );
		assert( id_is_finite( i ) );
		assert( id_is_finite( m ) );
		assert( i.v == 0 );
		assert( m.v == 1 );
	}

	/* lenses */
	{
		int nlens = 1;
		int nobjs = 5;
		int i;
		lens_s lenses[nlens], *l_ptr[nlens];
		lenses_s opt;
		inf_double_s objs[nobjs], imgs[nobjs], mags[nobjs];
		inf_double_s obj;
		char *s;

		objs[0] = to_inf_inf( 1 );
		objs[1] = to_inf_double( 200 );
		objs[2] = to_inf_double( 100 );
		objs[3] = to_inf_double( 50 );
		objs[4] = to_inf_double( 0 );

		imgs[0] = to_inf_double( 100 );
		imgs[1] = to_inf_double( 200 );
		imgs[2] = to_inf_inf( 1 );
		imgs[3] = to_inf_double( -100 );
		imgs[4] = to_inf_double( 0 );

		mags[0] = to_inf_double( 0 );
		mags[1] = to_inf_double( 1 );
		mags[2] = to_inf_inf( 1 );
		mags[3] = to_inf_double( -2 );
		mags[4] = to_inf_double( 1 );

		for( i = 0; i < nlens; i++) l_ptr[i] = &lenses[i];
		to_lens( &lenses[0], 100, 0 );
		opt.n = nlens;
		opt.lenses = l_ptr;

		for( i = 0; i < nobjs; i++)
			{
				obj = objs[i];

				fprintf( stderr, "%s\n", s = lenses_params( &opt, obj ) );
				if( s ) free( s );

				assert( id_is_equal_to( lenses_img( &opt, obj, -1 ), imgs[i], 1e-5 ) );
				assert( id_is_equal_to( lenses_mag( &opt, obj, -1 ), mags[i], 1e-5 ) );
			}
	}
	
	{
		int nlens = 2;
		int i;
		lens_s lenses[nlens], *l_ptr[nlens];
		lenses_s opt;
		inf_double_s star, pupil;
		inf_double_s img, mag;
		char *s;

		for( i = 0; i < nlens; i++)
			{
				l_ptr[i] = &lenses[i];
			}

		to_lens( &lenses[0], 100, 300 );
		to_lens( &lenses[1], 200, 200 );

		opt.n = nlens;
		opt.lenses = l_ptr;

		star = to_inf_double( 100 );
		pupil = to_inf_inf( 1 );

		fprintf( stderr, "star : %s\n", ( s = lenses_params( &opt, star ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &opt, pupil ) ) );
		if( s ) free( s );

		img = lenses_img( &opt, star, -1 );
		mag = lenses_mag( &opt, star, -1 );
		assert( id_is_equal_to( img, to_inf_double( 0 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( -2 ), 1e-5 ) );

		img = lenses_img( &opt, pupil, -1 );
		mag = lenses_mag( &opt, pupil, -1 );
		assert( id_is_equal_to( img, to_inf_inf( 1 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( 0.5 ), 1e-5 ) );
	}

	{
		int nlens = 4;
		int i;
		lens_s lenses[nlens], *l_ptr[nlens];
		lenses_s opt;
		inf_double_s star, pupil;
		inf_double_s img, mag;
		char *s;

		for( i = 0; i < nlens; i++)
			{
				l_ptr[i] = &lenses[i];
			}

		to_lens( &lenses[0], 300, 300 );
		to_lens( &lenses[1], 150, 300 );
		to_lens( &lenses[2], 75, 200 );
		to_lens( &lenses[3], 400.0/3.0, 400 );

		opt.n = nlens;
		opt.lenses = l_ptr;

		star = to_inf_double( 0 );
		pupil = to_inf_inf( 1 );

		fprintf( stderr, "%s\n", ( s = lenses_inspect( &opt ) ) );
		if( s ) free( s );
		fprintf( stderr, "star : %s\n", ( s = lenses_params( &opt, star ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &opt, pupil ) ) );
		if( s ) free( s );

		img = lenses_img( &opt, star, -1 );
		mag = lenses_mag( &opt, star, -1 );
		assert( id_is_equal_to( img, to_inf_double( 0 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( 2 ), 1e-5 ) );

		img = lenses_img( &opt, pupil, -1 );
		mag = lenses_mag( &opt, pupil, -1 );
		assert( id_is_equal_to( img, to_inf_double( -800 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( 0 ), 1e-5 ) );
	}
	
	{
		int nlens = 2;
		int i;
		lens_s lenses[nlens], *l_ptr[nlens];
		inf_double_s pars[2];	// position_from
		lenses_s opt;
		inf_double_s star, pupil;
		inf_double_s img, mag;
		char *s;

		for( i = 0; i < nlens; i++)
			{
				l_ptr[i] = &lenses[i];
			}

		to_lens( &lenses[0], 100, 400 );
		to_lens( &lenses[1], 300, 0 );

		opt.n = nlens;
		opt.lenses = l_ptr;

		star = to_inf_double( 100 );
		pupil = to_inf_inf( 1 );

		fputs( "BEFORE lenses_reconfigure()\n", stderr );
		fprintf( stderr, "%s\n", ( s = lenses_inspect( &opt ) ) );
		if( s ) free( s );
		fprintf( stderr, "star : %s\n", ( s = lenses_params( &opt, star ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &opt, pupil ) ) );
		if( s ) free( s );

		lenses[1].thick_type = position_from;
		lenses[1].thick_pars = pars;
		pars[0] = to_inf_double( 0 );
		pars[1] = to_inf_double( 700 );
		assert( lenses_reconfigure( &opt ) == -1 );

		fputs( "AFTER lenses_reconfigure() with absolute surface number\n", stderr );
		fprintf( stderr, "%s\n", ( s = lenses_inspect( &opt ) ) );
		if( s ) free( s );
		fprintf( stderr, "star : %s\n", ( s = lenses_params( &opt, star ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &opt, pupil ) ) );
		if( s ) free( s );

		img = lenses_img( &opt, star, -1 );
		mag = lenses_mag( &opt, star, -1 );
		assert( id_is_equal_to( img, to_inf_double( 0 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( -3 ), 1e-5 ) );

		img = lenses_img( &opt, pupil, -1 );
		mag = lenses_mag( &opt, pupil, -1 );
		assert( id_is_equal_to( img, to_inf_inf( 1 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( 0.3333333333333 ), 1e-5 ) );

		to_lens( &lenses[1], 300, 0 );
		lenses[1].thick_type = position_from;
		lenses[1].thick_pars = pars;
		pars[0] = to_inf_double( -1 );
		pars[1] = to_inf_double( 700 );
		assert( lenses_reconfigure( &opt ) == -1 );

		fputs( "AFTER lenses_reconfigure() with relative surface number\n", stderr );
		fprintf( stderr, "%s\n", ( s = lenses_inspect( &opt ) ) );
		if( s ) free( s );
		fprintf( stderr, "star : %s\n", ( s = lenses_params( &opt, star ) ) );
		if( s ) free( s );
		fprintf( stderr, "pupil: %s\n", ( s = lenses_params( &opt, pupil ) ) );
		if( s ) free( s );

		img = lenses_img( &opt, star, -1 );
		mag = lenses_mag( &opt, star, -1 );
		assert( id_is_equal_to( img, to_inf_double( 0 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( -3 ), 1e-5 ) );

		img = lenses_img( &opt, pupil, -1 );
		mag = lenses_mag( &opt, pupil, -1 );
		assert( id_is_equal_to( img, to_inf_inf( 1 ), 1e-5 ) );
		assert( id_is_equal_to( mag, to_inf_double( 0.3333333333333 ), 1e-5 ) );

	}

	fputs( "\ntests successfully finished.\n", stderr );
	return EXIT_SUCCESS;
}
