/* lens.h : calcuates lens */

#ifndef lens_h
#define lens_h

static char *lens_h_rcsid __attribute__ ((unused)) =
	"$Id: lens.h,v 1.1.1.1 2004/10/27 20:14:07 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define PLUS_INFINITY "+Infinity"
#define MINUS_INFINITY "-Infinity"

/* extended double being able to handle +/- Infinity */
typedef struct {
	double v;	// usual value
	int inf;	// 0: usual value, -1: -infinite, 1: +infinite
} inf_double_s;

inf_double_s to_inf_double( const double x );	// converts into extended double
inf_double_s to_inf_inf( const int sign );	// +/-inf or zero
inf_double_s str_to_inf_double( const char *string, char **endptr );	// parses the string, endptr points to the next char after parsing
double id_to_double( const inf_double_s x );	// returns double
int id_is_zero( const inf_double_s x );	// 1 if x is ~zero
int id_is_p_inf( const inf_double_s x );	// 1 if x is +infinite
int id_is_m_inf( const inf_double_s x );	// 1 if x is -infinite
int id_is_finite( const inf_double_s x );	// 1 if x is finite
int id_is_plus( const inf_double_s x );	// 1 if x is plus
int id_is_minus( const inf_double_s x );	// 1 if x is minus
int id_sign( const inf_double_s x );	// 1 or 0 or -1
int id_is_equal_to( const inf_double_s x, const inf_double_s y, double precision );	// x == y
inf_double_s id_invert( const inf_double_s x );	// x * -1
inf_double_s id_reciprocal( const inf_double_s x );	// reciprocal number for x
inf_double_s id_add( const inf_double_s a, const inf_double_s b);	// a + b
inf_double_s id_sub( const inf_double_s a, const inf_double_s b);	// a - b
inf_double_s id_mul( const inf_double_s a, const inf_double_s b);	// a * b
inf_double_s id_div( const inf_double_s a, const inf_double_s b);	// a / b
inf_double_s id_abs( const inf_double_s x );	// |x|
char *id_inspect( const inf_double_s x );	// inspects the structure
char *id_to_s( const inf_double_s x, const char *format );	// formats the value
	// these functions malloc() inside. you have to free() it afterwards.

/* a lens */
typedef enum {
	normal_feff = 0,	// feff is treated as focal length
} feff_type_t;

typedef enum {
	normal_thick = 0,	// thick is treated as thickness
	position_from,	// thick is calculated
		// if thick_pars[0] < 0, it is treated as a relative surface number
		// if thick_pars[0] >= 0, it is treated as an absolute surface number
		// thick_pars[1] holds relative position from the specified surface
} thick_type_t;

/* number of parameters needed for each type */
#define return_feff_type_npars( type ) \
	{ switch( type ) { \
		case normal_feff: return 0; \
	} return -1; }

#define return_thick_type_npars( type ) \
	{ switch( type ) { \
		case normal_thick: return 0; \
		case position_from: return 2; \
	} return -1; }

typedef struct {
	inf_double_s feff;	// focal length
	int feff_is_free;	// 1 if feff is a free parameter
	feff_type_t feff_type;	// special parameter?
	inf_double_s *feff_pars;	// parameters
	inf_double_s thick;	// thickness
	int thick_is_free;	// 1 if thick is a free parameter
	thick_type_t thick_type;	// special parameter?
	inf_double_s *thick_pars;	// parameters
	char *comment;	// comment in the file (may be NULL)
} lens_s;

int feff_type_npars( feff_type_t type );	// returns number of special parameters
int thick_type_npars( thick_type_t type );	// returns number of special parameters

void to_lens( lens_s *l, const double feff, const double thick );	// sets the parameters
inf_double_s lens_img( const lens_s *l, const inf_double_s object );	// image distance from lens
inf_double_s lens_mag( const lens_s *l, const inf_double_s object );	// magnification
char *lens_inspect( const lens_s *l );	// inspects the lens
	// this function malloc()s inside. you have to free() it afterwards.
void lens_free( lens_s *l );	// frees the special parameters

/* lenses */
typedef struct {
	unsigned int n;	// number of lenses
	lens_s **lenses;	// pointer to an array of lenses
} lenses_s;

int lenses_reconfigure( lenses_s *l );	// sets feff and thick from special parameters
int lenses_feedback( lenses_s *l );	// sets special parameters from the current geometry
	// returns -1 if successful, surface number if an error occured
inf_double_s lenses_img( const lenses_s *l, const inf_double_s object, int last_surface );	// image distance from the last surface of the lenses (including thickness)
inf_double_s lenses_mag( const lenses_s *l, const inf_double_s object, int last_surface );	// magnification
	// you have to lenses_reconfigure() before calculating these.
	// If last_surface >= 0, result is calculated through the specified surface
	// otherwise, surface is counted from the last one
	// (specify -1 for the last surface)
char *lenses_params( const lenses_s *l, const inf_double_s object );	// shows the image location and magnification
char *lenses_inspect( const lenses_s *l );	// shows the lenses
	// this function malloc()s inside. you have to free() it afterwards.

#endif	/* lens_h */
