/* lens.c : calculates lens */

static char *rcsid __attribute__((unused)) =
	"$Id: lens.c,v 1.1.1.1 2004/10/27 20:14:07 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef DEBUG
#include <stdio.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <float.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include "lens.h"
#include "mallocprintf.h"

/* extended double being able to handle +/- Infinity */
/* converts into extended double */
inf_double_s
to_inf_double( const double x )
{
	if( x == DBL_MAX )
		{
			return to_inf_inf( 1 );
		}
	else
		{
			if( x == -DBL_MAX )
				{
					return to_inf_inf( -1 );
				}
			else
				{
					inf_double_s r;
					r.v = x;
					r.inf = 0;
					return r;
				}
		}
}

/* +/-inf or zero */
inf_double_s
to_inf_inf( const int sign )
{
	inf_double_s r;
	r.v = 0;
	if( sign > 0 )
		{
			r.inf = 1;
		}
	else
		{
			if( sign < 0 )
				{
					r.inf = -1;
				}
			else
				{
					r.inf = 0;
				}
		}
	return r;
}

/* parses the string */
inf_double_s
str_to_inf_double( const char *string, char **endptr )
{
	inf_double_s r;
	if( strncmp(string, PLUS_INFINITY, strlen( PLUS_INFINITY ) ) == 0 )
		{
			if( endptr ) *endptr = (char *) ( string + strlen( PLUS_INFINITY ) );
			return to_inf_inf( 1 );
		}
	if( strncmp(string, MINUS_INFINITY, strlen( MINUS_INFINITY ) ) == 0 )
		{
			if( endptr ) *endptr = (char *) ( string + strlen( MINUS_INFINITY ) );
			return to_inf_inf( -1 );
		}
	r = to_inf_double( strtod( string, endptr ) );
  return r;
}

/* returns double */
double
id_to_double( const inf_double_s x )
{
	if( id_is_finite( x ) )
		{
			return x.v;
		}
	else
		{
			switch( id_sign( x ) )
				{
					case 1: return DBL_MAX;
					case -1: return -DBL_MAX;
				}
		}
	assert( 0 );
	return 0;	// never reach here
}

/* 1 if x is ~zero */
int
id_is_zero( const inf_double_s x )
{
	if( !x.inf && -DBL_EPSILON * 2 < x.v && x.v < DBL_EPSILON * 2 ) return 1;
	return 0;
}

/* 1 if x is +infinite */
int
id_is_p_inf( const inf_double_s x )
{
	if( x.inf > 0 ) return 1;
	return 0;
}

/* 1 if x is -infinite */
int
id_is_m_inf( const inf_double_s x )
{
	if( x.inf < 0 ) return 1;
	return 0;
}

/* 1 if x is finite */
int
id_is_finite( const inf_double_s x )
{
	if( x.inf == 0 ) return 1;
	return 0;
}

/* 1 if x is plus */
int
id_is_plus( const inf_double_s x )
{
	if( x.inf > 0 ) return 1;
	if( x.inf == 0 && x.v >= 0 ) return 1;
	return 0;
}

/* 1 if x is minus */
int
id_is_minus( const inf_double_s x )
{
	if( x.inf < 0 ) return 1;
	if( x.inf == 0 && x.v < 0 ) return 1;
	return 0;
}

/* 1 or 0 or -1 */
int
id_sign( const inf_double_s x )
{
	if( id_is_zero( x ) ) return 0;
	if( id_is_plus( x ) ) return 1;
	return -1;
}

/* x == y */
int
id_is_equal_to( const inf_double_s x, const inf_double_s y, double precision )
{
	if( id_is_finite( x ) && id_is_finite( y ) )
		{
			return fabs( x.v - y.v ) <= precision;
		}
	else
		{
			return x.inf == y.inf;
		}
}

/* x * -1 */
inf_double_s
id_invert( const inf_double_s x )
{
	inf_double_s r;
	if( id_is_finite( x ) )
		{
			r =  to_inf_double( -x.v );
		}
	else
		{
			r = x;
			r.inf *= -1;
		}
	return r;
}

/* id_reciprocal number for x */
inf_double_s
id_reciprocal( const inf_double_s x )
{
	if( id_is_zero( x ) )
		{
			return to_inf_inf( id_sign( x ) < 0 ? -1 : 1 );
		}
	else
		{
			if( !id_is_finite( x ) )
				{
					return to_inf_double( 0 );
				}
			else
				{
					return to_inf_double( 1 / x.v );
				}
		}
}

/* a + b */
inf_double_s
id_add( const inf_double_s a, const inf_double_s b )
{
	if( !id_is_finite( a ) )
		{
			if( !id_is_finite( b ) && ( id_sign( a ) * id_sign( b ) < 0 ) )
				{
					return to_inf_double( 0 );	// should be NaN
				}
			else
				{
					return a;
				}
		}
	else
		{
			if( !id_is_finite( b ) )
				{
					return b;
				}
			else
				{
					return to_inf_double( a.v + b.v );
				}
		}
}

/* a - b  */
inf_double_s
id_sub( const inf_double_s a, const inf_double_s b )
{
	return id_add( a, id_invert( b ) );
}

/* a * b */
inf_double_s
id_mul( const inf_double_s a, const inf_double_s b )
{
	if( id_is_finite( a ) && id_is_finite( b ) )
		{
			return to_inf_double( a.v * b.v );
		}
	else
		{
			return to_inf_inf( id_sign( a ) * id_sign( b ) );
		}
}

/* a / b */
inf_double_s
id_div( const inf_double_s a, const inf_double_s b )
{
	if( id_is_zero( b ) ) return to_inf_inf( id_sign( a ) );

	if( id_is_finite( a ) && id_is_finite( b ) )
		{
			return to_inf_double( a.v / b.v );
		}
	else
		{
			if(! id_is_finite( b ) )
				{
					if( id_is_finite( a ) )
						{
							return to_inf_double( 0 );
						}
					else
						{
							return to_inf_double( id_sign( a ) * id_sign( b ) );
						}
				}
			else
				{
					return to_inf_inf( id_sign( a ) * id_sign( b ) );
				}
		}
}

/* |x| */
inf_double_s
id_abs( const inf_double_s x )
{
	inf_double_s r;
	r.v = fabs( x.v );
	r.inf = abs( x.inf );
	return r;
}

char *
id_inspect( const inf_double_s x )
{
	return mallocprintf( "v:%g inf:%d", x.v, x.inf );
}

char *
id_to_s( const inf_double_s x, const char *format )
{
	if( id_is_finite( x ) )
		{
			return mallocprintf( format, x.v );
		}
	else
		{
			if( id_is_p_inf( x ) )
				{
					return mallocprintf( PLUS_INFINITY );
				}
			else
				{
					return mallocprintf( MINUS_INFINITY );
				}
		}
}

/* a lens */
/* returns number of special parameters */
int feff_type_npars( feff_type_t type )
{
	return_feff_type_npars( type );
}

int thick_type_npars( thick_type_t type )
{
	return_thick_type_npars( type );
}


/* sets the parameters */
void
to_lens( lens_s *l, const double feff, const double thick )
{
	l->feff = to_inf_double( feff );
	l->thick = to_inf_double( thick );
	l->feff_is_free = 0;
	l->thick_is_free = 0;
	l->feff_type = normal_feff;
	l->thick_type = normal_thick;
	l->feff_pars = NULL;
	l->thick_pars = NULL;
	l->comment = NULL;
}

/* image distance from lens */
inf_double_s 
lens_img( const lens_s *l, const inf_double_s object )
{
	return id_reciprocal( id_sub( id_reciprocal( l->feff ), id_reciprocal( object ) ) );
}

/* magnification */
inf_double_s
lens_mag( const lens_s *l, const inf_double_s object )
{
	inf_double_s img;
	if( id_is_zero( object ) ) return to_inf_double( 1 );
	img = lens_img( l, object );
	if( id_is_zero( img ) ) return to_inf_double( 1 );
	return id_div( img, object );
}

/* inspects the lens */
char *
lens_inspect( const lens_s *l )
{
	char *r, *f, *t;
	f = id_to_s( l->feff, "%g" );
	t = id_to_s( l->thick, "%g" );
	r = mallocprintf( "f:%s%s t:%s%s", f, l->feff_is_free ? "V" : "", t, l->thick_is_free ? "V" : "" );
	if( f ) free( f );
	if( t ) free( t );
	return r;
}

/* frees the specil parameters */
void
lens_free( lens_s *l )
{
	if( l->feff_pars ) free( l->feff_pars );
	if( l->thick_pars ) free( l->thick_pars );
	if( l->comment ) free( l->comment );
}

/* lenses */
/* set feff and thick according to lens special parameters */
int
lenses_reconfigure( lenses_s *l )
{
	int i;
	for(i = 0; i < l->n; i++)
		{
			switch( l->lenses[i]->feff_type )
				{
					case normal_feff: break;	// nothing to do
					default: assert(0);
				}
			switch( l->lenses[i]->thick_type )
				{
					case normal_thick: break;	// nothing to do
					case position_from:
						{
							int orig, j;

							/* define original surface */
							orig = (int) id_to_double( l->lenses[i]->thick_pars[0] );
							if( orig < 0 ) orig += i;	// original surface number
							if( orig < 0 || orig > i )	return i;	// error

							/* calculate required thickeness */
							l->lenses[i]->thick = l->lenses[i]->thick_pars[1];
							for( j = orig; j < i; j++ )
								{
									l->lenses[i]->thick = id_sub( l->lenses[i]->thick, l->lenses[j]->thick );
								}
						}
						break;
					default: assert(0);
				}
		}

	return -1;
}

/* set lens special parameters according to feff and thick */
int
lenses_feedback( lenses_s *l )
{
	int i;
	for(i = 0; i < l->n; i++)
		{
			switch( l->lenses[i]->feff_type )
				{
					case normal_feff: break;	// nothing to do
					default: assert(0);
				}
			switch( l->lenses[i]->thick_type )
				{
					case normal_thick: break;	// nothing to do
					case position_from:
						{
							int orig, j;

							/* define original surface */
							orig = (int) id_to_double( l->lenses[i]->thick_pars[0] );
							if( orig < 0 ) orig += i;	// original surface number
							if( orig < 0 || orig >= i )	return i;	// error

							/* calculate required thickeness */
							l->lenses[i]->thick_pars[1] = to_inf_double( 0 );
							for( j = orig; j <= i; j++ )
								{
									l->lenses[i]->thick_pars[1] = id_add( l->lenses[i]->thick_pars[1], l->lenses[j]->thick );
								}
						}
						break;
					default: assert(0);
				}
		}

	return -1;
}

/* distance from the last surface of the lenses (including thickness) */
inf_double_s
lenses_img( const lenses_s *l, const inf_double_s object, int last_surface )
{
	int i, last;
	inf_double_s obj, img;	// object and image distances on current lens

	last = ( last_surface < 0 ) ? l->n + last_surface : last_surface;
	if( last < 0 ) last = 0;
	if( last >= l->n ) last = l->n - 1;
	obj = object;
	for(i = 0; i <= last; i++)
		{
			img = lens_img( l->lenses[i], obj );
			obj = id_sub( l->lenses[i]->thick, img );
		}
	return id_invert( obj );
}

/* magnification */
inf_double_s
lenses_mag( const lenses_s *l, const inf_double_s object, int last_surface )
{
	int i, last;
	int inf_sign, inf_num;	// sign and number of infinities
	inf_double_s finite_mag;	// current magnification from finite distacnes
	inf_double_s obj, img;	// object and image distances on current lens

	last = ( last_surface < 0 ) ? l->n + last_surface : last_surface;
	finite_mag = to_inf_double( 1 );
	inf_sign = 1;
	inf_num = 0;

	obj = object;
	for(i = 0; i <= last; i++)
		{
			if( id_is_finite( obj ) && !id_is_zero( obj ) )
				{
					finite_mag = id_div( finite_mag, obj );
				}
			else
				{
					inf_num -= 1;
					if( id_is_m_inf( obj ) ) inf_sign *= -1;
				}

			img = lens_img( l->lenses[i], obj );
			if( id_is_finite( img ) && !id_is_zero( img ) )
				{
					finite_mag = id_mul( finite_mag, img );
				}
			else
				{
					inf_num += 1;
					if( id_is_m_inf( img ) ) inf_sign *= -1;
				}

			obj = id_sub( l->lenses[i]->thick, img );
		}

	if( inf_num > 0 ) return to_inf_inf( inf_sign );
	if( inf_num < 0 ) return to_inf_double( 0 );

	if( inf_sign < 0 ) finite_mag = id_invert( finite_mag );
	return finite_mag;
}

/* shows the image location and magnification */
char *
lenses_params( const lenses_s *l, const inf_double_s object )
{
	char *r, *o, *i, *m;
	o = id_to_s( object, "%g" );
	i = id_to_s( lenses_img( l, object, -1 ), "%g" );
	m = id_to_s( lenses_mag( l, object, -1 ), "%g" );
	r = mallocprintf( "object:%s image:%s magnification:%s", o, i, m );
	if( o ) free( o );
	if( i ) free( i );
	if( m ) free( m );
	return r;
}

/* shows the lenses */
char *
lenses_inspect( const lenses_s *l )
{
	char *r, **num, **lens;
	size_t total_len;
	unsigned int i;

	num = (char **) malloc( l->n * sizeof( char * ) );
	lens = (char **) malloc( l->n * sizeof( char * ) );
	if( !num || !lens )
		{
			fprintf( stderr, "%s:%d could not allocate memory for strings\n", __FILE__, __LINE__ );
			exit( EXIT_FAILURE );
		}

	total_len = l->n;	// spaces between lenses and terminating \0
	for( i = 0; i < l->n; i++ )
		{
			num[i] = mallocprintf( "[L%d]", i );
			lens[i] = lens_inspect( l->lenses[i] );
			total_len += strlen( num[i] ) + strlen( lens[i] );
		}

	r = (char *) malloc( total_len * sizeof( char ) );
	if( !r )
		{
			fprintf( stderr, "%s:%d could not allocate memory for strings\n", __FILE__, __LINE__ );
			exit( EXIT_FAILURE );
		}

	r[0] = '\0';
	for( i = 0; i < l->n; i++ )
		{
			if( i > 0 ) strcat( r, " " );
			strcat( r, num[i] );
			strcat( r, lens[i] );
			if( num[i] ) free( num[i] );
			if( lens[i] ) free( lens[i] );
		}

	free( num );
	free( lens );

	return r;
}
