/* conffile.c configuration file I/O */

static char *rcsid __attribute__((unused)) =
	"$Id: conffile.c,v 1.2 2004/10/27 20:50:43 tomono Exp $";

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "conffile.h"
#include "optimize.h"

#define LENS_FORMAT "%g"
#define MF_FORMAT "%.2g"
#define READ_BUFSIZE 256
#define READ_FIRST_NLENS 16
#define PRECISION 1e-5
#define DELIM " \t\n"
#define COMMENT "#"

/* local prototypes */
optics_minimizer_s * _lenses_shallow_allocate( const unsigned int nlens );	// does not initialize lenses and other parameters
void _lenses_init( optics_minimizer_s * opt );	// initialize flat parameters

/* local functions */
/* copies the first word at cptr. eptr points next to the end of the word */
words_s *
word_get( const char *start, char **next )
{
	words_s *r;
	char *cptr;
	size_t l;

	cptr = (char *) start;

	/* skip spaces */
	cptr += strspn( cptr, DELIM );

	/* check where we are */
	if( *cptr == '\0' || strchr( COMMENT, *cptr ) )
		{
			if( next ) *next = NULL;
			return NULL;
		}

	/* copy the word */
	r = (words_s *) malloc( sizeof( words_s ) );
	if( !r )
		{
			perror( "word_get()" );
			if( next ) *next = cptr;
			return NULL;
		}
	l = strcspn( cptr, DELIM );
	r->word = (char *) malloc( ( l + 1 ) * sizeof( char ) );
	if( !r->word )
		{
			free( r );
			perror( "word_get()" );
			if( next ) *next = cptr;
			return NULL;
		}
	strncpy( r->word, cptr, l );
	r->word[l] = '\0';
	r->next = NULL;

	if( next ) *next = cptr + l;
	return r;
}

/* copies the first value or word at cptr */
words_s *
value_get( const char *start, char **next )
{
	words_s *r;
	char *cptr, *nptr, *nnptr;
	size_t l;

	cptr = (char *) start;

	/* skip spaces */
	cptr += strspn( cptr, DELIM );

	/* check where we are */
	if( *cptr == '\0' || strchr( COMMENT, *cptr ) )
		{
			if( next ) *next = NULL;
			return NULL;
		}

	/* first, we try to get a value */

	/* then, try to get a non-value word */
	nptr = cptr;
	while( *nptr != '\0' )
		{
			if( isnan( id_to_double( str_to_inf_double( nptr, &nnptr ) ) ) )
				{
					nnptr = nptr;	// we do not want to get NaN's
				}
			if( nnptr > nptr ) break;	// we are at the start of next number
			nptr++;	// FIX ME: try with next charactor
		}
	if( nptr == cptr ) nptr = nnptr;

	/* copy the string or value */
	r = (words_s *) malloc( sizeof( words_s ) );
	if( !r )
		{
			perror( "word_get()" );
			if( next ) *next = cptr;
			return NULL;
		}
	l = nptr - cptr;
	r->word = (char *) malloc( ( l + 1 ) * sizeof( char ) );
	if( !r->word )
		{
			free( r );
			perror( "word_get()" );
			if( next ) *next = cptr;
			return NULL;
		}
	strncpy( r->word, cptr, l );
	r->word[l] = '\0';
	r->next = NULL;

	if( next ) *next = cptr + l;
	return r;
}

/* scans for words(mode='w') or values(mode='v') */
words_s *
words_s_scan( const char *line, const int mode, char **comment )
{
	words_s *r, *p, *c;
	char *cptr, *nptr;
	words_s * (* parser) ( const char *, char ** );

	switch( mode )
		{
			case 'w': parser = &word_get; break;
			case 'v': parser = &value_get; break;
			default: parser = NULL;	// should never reach here
		}

	cptr = (char *) line;
	p = NULL;
	r = NULL;
	while( cptr )
		{
			c = parser( cptr, &nptr );
			if( c )
				{
					if( p ) p->next = c;
					if( !r ) r = c;
					p = c;
				}
			else
				{
					if( comment )
						{
							cptr += strspn( cptr, DELIM );
							if( *cptr != '\0' && strchr( COMMENT, *cptr ) )
								{
									*comment = cptr;
								}
							else
								{
									*comment = NULL;
								}
						}
				}
			cptr = nptr;
		}

	return r;
}

/* scans words from a line, make a list, and returns the first words_s */
words_s *
word_scan( const char *line, char **comment )
{
	return words_s_scan( line, 'w', comment );
}

/* scans values from a line, make a list, and returns the first words_s */
words_s *
value_scan( const char *line )
{
	return words_s_scan( line, 'v', NULL );
}

/* free the memories allocated by word_scan or word_get */
void
word_free( words_s *words )
{
	words_s *c, *n;

	c = words;
	while( c )
		{
			n = c->next;
			free( c->word );
			free( c );
			c = n;
		}
}

/* count the number of words */
size_t
word_n( const words_s *words )
{
	size_t r;
	const words_s *c;

	r = 0;
	c = words;
	while( c )
		{
			r++;
			c = c->next;
		}

	return r;
}

/* next word */
words_s *
word_next( const words_s *word )
{
	return word->next;
}

/* the word */
const char*
word_the( const words_s *word )
{
	return word->word;
}

/* the i-th word */
const char*
word_ith( const words_s *words, const size_t n )
{
	size_t i;
	const words_s *c;

	i = 0;
	c = words;
	while( i < n && c )
		{
			i++;
			c = c->next;
		}
	
	return c ? c->word : NULL;
}

/* usage */
void
lenses_file_usage( FILE *stream )
{
	fputs( "\
Format the lens file as follows -\n\
\n\
* General rules\n\
- use space(s) or tab(s) as a delimiter\n\
- begin comments with a #: strings in a line following it are ignored\n\
\n\
Lenses and other parameters: format the conf file as follows\n\
\n\
--\n\
L focal_length[V] thickness[V]  # comment\n\
  # a lens, stored in the sequence it appears\n\
  # if a `V' follows the numbers, the parameter is free and will be optimized\n\
  # `+Infinity' and `-Infinity' are allowed\n\
  # a comment can be added after each line\n\
star object_distance_for_stellar_image\n\
pupil object_distance_for_pupil_image\n\
  # object distances, `+Infinity' and `-Infinity' are allowed\n\
\n\
# Optional followings are optional\n\
star_img   goal_image_distance [weight [surface [object_distance]]]\n\
pupil_img  goal_image_distance [weight [surface [object_distance]]]\n\
star_mag   goal_magnification  [weight [surface [object_distance]]]\n\
pupil_mag  goal_magnification  [weight [surface [object_distance]]]\n\
star_amag  goal_magnification  [weight [surface [object_distance]]]\n\
pupil_amag goal_magnification  [weight [surface [object_distance]]]\n\
  # a comment can be added after each line\n\
--\n\
\n\
* Merit function operands\n\
If the first word ends with _img, it defines an operand for image\n\
distance while _mag for magnification, and _amag for absolute\n\
magnification.\n\
\n\
If no surface is specified, operands are optimized for the last lens.\n\
Surface >= 0 means aboslute surface number starting with zero, otherwise\n\
surface number is counted from the last one (-1 means the last one).\n\
\n\
Merit functions beginning with star/pupil uses object distances defined\n\
with star/pupil line. If object distance is specified, it overrides the\n\
above.\n\
\n\
`+Infinity' and `-Infinity' are allowed for the goals and object\n\
distances. ignored while optimization if weight <= 0.\n\
\n\
* Surface constraints\n\
In place of thickness in lens defnition, one can write\n\
  <distance>from<surface>[V]\n\
which means to set the thickness of the lens so that after going through\n\
its own thickness, the location of the surface is <distance> from the\n\
starting surface of the <surface>.\n\
\n\
If <surface> is zero or more than zero, it is treated as an absolute\n\
surface number (starting from zero). If it is negative, it is relative\n\
to the current surface.\n", stream );
}

/* writes the OPTICS into the file PATH */
int
lenses_write( const optics_minimizer_s* optics, const char *path )
{
	FILE *outfile;
	time_t now;
	char timebuf[27];

	if( strcmp( "-", path ) == 0 )
		{
			outfile = stdout;
		}
	else
		{
			if( !( outfile = fopen( path, "w" ) ) )
				{
					perror( path );
					return errno;
				}
		}

	lenses_reconfigure( optics->optics );

	time( &now );
	fprintf( outfile, "# optics file by %s-%s on %s", PACKAGE, VERSION, ctime_r( &now, timebuf ) );

	{
		unsigned int l;
		char *s;
		fputs( "\n#lenses: feff thickness\n", outfile );
		for( l = 0; l < optics->optics->n; l++)
			{
				fputs( "L\t", outfile );
				switch( optics->optics->lenses[l]->feff_type )
					{
						case normal_feff:
							fputs( s = id_to_s( optics->optics->lenses[l]->feff, LENS_FORMAT ), outfile );
							if( s ) free( s );
							break;
						default:
							fputs( "unknown-focal-length-type!", outfile );
							break;
					}
				if( optics->optics->lenses[l]->feff_is_free ) fputs( "V", outfile);
				fputs( " ", outfile );
				switch( optics->optics->lenses[l]->thick_type )
					{
						case normal_thick:
							fputs( s = id_to_s( optics->optics->lenses[l]->thick, LENS_FORMAT ), outfile );
							if( s ) free( s );
							break;
						case position_from:
							fputs( s = id_to_s( optics->optics->lenses[l]->thick_pars[1], LENS_FORMAT ), outfile );
							if( s ) free( s );
							fputs( "from", outfile );
							fputs( s = id_to_s( optics->optics->lenses[l]->thick_pars[0], LENS_FORMAT ), outfile );
							if( s ) free( s );
							break;
						default:
							fputs( "unknown-thickness-type!", outfile );
							break;
					}
				if( optics->optics->lenses[l]->thick_is_free ) fputs( "V", outfile);
				if( optics->optics->lenses[l]->comment )
					{
						fprintf( outfile, "\t%s", optics->optics->lenses[l]->comment );
					}
				fputs( "\n", outfile );
				
			}
	}

	{
		char *s;
		fputs( "\n#objects\n", outfile );
		fprintf( outfile, "star\t%s\n", s = id_to_s( optics->star_obj, LENS_FORMAT ) );
		free( s );
		fprintf( outfile, "pupil\t%s\n", s = id_to_s( optics->pupil_obj, LENS_FORMAT ) );
		free( s );
	}

	if( optics->mf )
		{
			char *s;
			double delta, total_mf;
			merit_function_operands_s *cur;

			fputs( "\n#current status\n", outfile );
			fprintf( outfile, "# optimization status: %s\n", optimization_status_to_s( optics->final_status ) );
			if( optics->final_status != lens_opt_notdone )
				{
					total_mf = optics->final_merit_function;
					fprintf( outfile, "# final merit function: " LENS_FORMAT "\n", total_mf ) ;
				}
			else
				{
					total_mf = merit_function( optics );
					fprintf( outfile, "# current merit function: " LENS_FORMAT "\n", total_mf ) ;
				}

			cur = optics->mf;
			while( cur )
				{
					delta = mf_operand_delta( cur, optics->optics );
					fprintf( outfile, "# current %s\t%s\td=%.3g",
						cur->name,
						s = id_to_s( mf_operand_value( cur, optics->optics ), MF_FORMAT ),
						delta
					);
					if( total_mf > 0 ) 
						{
							fprintf( outfile, "\t(%6.2f%%)", delta / total_mf * 100 );
						}
					fputs( "\n", outfile );
					if( s ) free( s );
					cur = cur->next;
				}
		}

	if( optics->mf )
		{
			char *s;
			merit_function_operands_s *cur;

			fputs( "\n#merit function: goal weight [surface object]\n", outfile );
			cur = optics->mf;
			while( cur )
				{
					fprintf( outfile, "%s\t%s " MF_FORMAT,
						cur->name,
						s = id_to_s( cur->goal, MF_FORMAT ),
						cur->weight );
					if( s ) free( s );
					if( cur->surface != -1 )
						{
							fprintf( outfile, " %d", cur->surface );
						}
					if( cur->object_number == -1 )
						{
							if( cur->surface == -1 )
								{
									fprintf( outfile, " %d", cur->surface );
								}
							fprintf( outfile, " %s",
								s = id_to_s( cur->object, MF_FORMAT ) );
							if( s ) free( s );
						}
					if( cur->comment )
						{
							fprintf( outfile, "\t%s", cur->comment );
						}
					fputs( "\n", outfile );
					cur = cur->next;
				}
		}
	
	if( strcmp( "-", path ) != 0 ) fclose( outfile );
	return 0;
}

/* does not initialize lenses and other parameters */
optics_minimizer_s *
_lenses_shallow_allocate( const unsigned int nlens )
{
	optics_minimizer_s *r;

	/* memory allocation for optics_minimizer_s */
	r = (optics_minimizer_s *) malloc( sizeof( optics_minimizer_s ) );
	if( !r )
		{
			perror( "optics_minimizer_s" );
			return NULL;
		}

	r->optics = (lenses_s *) malloc( sizeof( lenses_s ) );
	if( !r->optics )
		{
			perror( "lenses_s" );
			free( r );
			return NULL;
		}

	r->optics->lenses = (lens_s **) malloc( nlens * sizeof( lens_s * ) );
	if( !r->optics->lenses )
		{
			perror( "lens_s" );
			free( r->optics );
			free( r );
			return NULL;
		}

	r->optics->n = 0;
	return r;
}

/* alloctes memory for optics and initializes it */
optics_minimizer_s *
lenses_allocate( const unsigned int nlens )
{
	optics_minimizer_s *r;
	unsigned int i;

	r = _lenses_shallow_allocate( nlens );
	if( !r ) return r;

	r->mf = NULL;

	/* initialization of the lenses */
	for( i = 0; i < nlens; i++ )
		{
			r->optics->lenses[i] = (lens_s *) malloc( sizeof( lens_s ) );
			if( !r->optics->lenses[i] )
				{
					unsigned int j;
					if( r->optics->lenses[i] ) free( r->optics->lenses[i] );
					for( j = 0; j < i - 1; j++ )
						{
							free( r->optics->lenses[j] );
						}
					perror( "lens_s" );
					free( r->optics->lenses );
					free( r->optics );
					free( r );
					return NULL;
				}
			r->optics->lenses[i]->feff = to_inf_inf( 1 );
			r->optics->lenses[i]->thick = to_inf_double( 0 );
			r->optics->lenses[i]->feff_is_free = 0;
			r->optics->lenses[i]->thick_is_free = 0;
			r->optics->lenses[i]->comment = NULL;
		}
	r->optics->n = nlens;

	_lenses_init( r );

	return r;
}

/* initialize flat parameters */
void
_lenses_init( optics_minimizer_s * opt )
{
	if( !opt ) return;

	/* other parameters */
	opt->star_obj = to_inf_double( 0 );
	opt->pupil_obj = to_inf_inf( 1 );
	opt->mf = NULL;
	opt->final_status = lens_opt_notdone;
	opt->final_merit_function = 0;
}

/* free the memories allocated by lenses_read() or lenses_allocate() */
void
lenses_free( optics_minimizer_s* optics )
{
	unsigned int i;
	if( !optics ) return;

	/* each lens */
	for( i = 0; i < optics->optics->n; i++ ) {
		if( optics->optics->lenses[i] )
		{
			lens_free( optics->optics->lenses[i] );
			free( optics->optics->lenses[i] );
		}
	}

	/* lenses */
	if( optics->optics->lenses ) free( optics->optics->lenses );

	if( optics->optics ) free( optics->optics );

	/* merit functions */
	mf_operand_free( optics->mf );

	/* self */
	free( optics );
}

/* reads the PATH and al locates and sets the optics */
optics_minimizer_s *
lenses_read( const char *path )
{
	optics_minimizer_s *r;
	unsigned int ilens, nalloc;
	FILE *infile;
	char buf[READ_BUFSIZE];
	size_t iline;
	words_s *words;

	if( strcmp( "-", path ) == 0 )
		{
			infile = stdin;
		}
	else
		{
			if( !( infile = fopen( path, "r" ) ) )
				{
					perror( path );
					return NULL;
				}
		}

	r = _lenses_shallow_allocate( READ_FIRST_NLENS );
	if( !r )
		{
			perror( "optics_read" );
			return NULL;
		}
	_lenses_init( r );
	nalloc = READ_FIRST_NLENS;

	ilens = 0;
	iline = 0;
	while( 1 )
		{
			char *comment;

			/* read a line */
			fgets( buf, READ_BUFSIZE, infile );
			if( feof( infile ) ) break;
			iline++;

			/* check the eol */
			if( buf[strlen( buf ) - 1] != '\n' )
				{
					fprintf( stderr, "%s:%d line to long:'%.20s ...'\n\tSorry. Please increase READ_BUFSIZE in %s.\n", path, iline, buf, __FILE__ );
					lenses_free( r );
					return NULL;
				}

			/* parse the line */
			words = word_scan( buf, &comment );
			if( !words ) continue;

			/* add a lens */
			if( strcmp( "L", word_ith( words, 0 ) ) == 0 )
				{
					/* check number of parameters */
					if( word_n( words ) != 3 )
						{
							fprintf( stderr, "%s:%d parse error for a lens: number of parameters wrong.\n", path, iline );
							word_free( words );
							lenses_free( r );
							return NULL;
						}

					/* allocate the lens array */
					if( nalloc < r->optics->n + 1 )
						{
							r->optics->lenses = (lens_s **) realloc( r->optics->lenses, nalloc * 2 * sizeof( lens_s * ) );
							if( !r->optics->lenses )
								{
									perror( "lens_s *" );
									lenses_free( r );
									return NULL;
								}
							nalloc *= 2;
						}

					/* allocate a lens */
					r->optics->lenses[r->optics->n] = (lens_s *) malloc( sizeof( lens_s ) );
					if( !r->optics->lenses[r->optics->n] )
						{
							perror( "lens_s" );
							lenses_free( r );
							return NULL;
						}
					r->optics->n++;

					/* copy the comment */
					if( comment )
						{
							char *c;
							c = strdup( comment );
							if( c[strlen(c) - 1] == '\n' ) c[strlen(c) - 1] = '\0';
							r->optics->lenses[r->optics->n - 1]->comment = c;
						}
					else
						{
							r->optics->lenses[r->optics->n - 1]->comment = NULL;
						}

					/* set the lens */
					{
						words_s *parsed;
						int is_free;

						/* feff */
						parsed = value_scan( word_ith( words, 1) );
						if( !parsed )
							{
								perror( "lens_s" );
								lenses_free( r );
								return NULL;
							}
						if( word_n( parsed ) > 1 &&
							( strcmp( word_ith( parsed, word_n( parsed ) - 1 ), "V" ) == 0 ||
							strcmp( word_ith( parsed, word_n( parsed ) - 1 ), "v" ) == 0 ) )
							{
								is_free = 1;
							}
						else
							{
								is_free = 0;
							}
						r->optics->lenses[r->optics->n - 1]->feff_is_free = is_free;
						switch( word_n( parsed ) - is_free )
							{
								case 1:	// normal surface
									r->optics->lenses[r->optics->n - 1]->feff = str_to_inf_double( word_ith( parsed, 0 ), NULL );
									r->optics->lenses[r->optics->n - 1]->feff_type = normal_feff;
									r->optics->lenses[r->optics->n - 1]->feff_pars = NULL;
									break;
								default:	// error
									fprintf( stderr, "%s:%d error parsing focal length.\n", path, iline );
									word_free( parsed );
									word_free( words );
									lenses_free( r );
									return NULL;
							}
						word_free( parsed );

						/* thick */
						parsed = value_scan( word_ith( words, 2) );
						if( !parsed )
							{
								perror( "lens_s" );
								lenses_free( r );
								return NULL;
							}
						if( word_n( parsed ) > 1 &&
							( strcmp( word_ith( parsed, word_n( parsed ) - 1 ), "V" ) == 0 ||
							strcmp( word_ith( parsed, word_n( parsed ) - 1 ), "v" ) == 0 ) )
							{
								is_free = 1;
							}
						else
							{
								is_free = 0;
							}
						r->optics->lenses[r->optics->n - 1]->thick_is_free = is_free;
						switch( word_n( parsed ) - is_free )
							{
								case 1:	// normal surface
									r->optics->lenses[r->optics->n - 1]->thick = str_to_inf_double( word_ith( parsed, 0 ), NULL );
									r->optics->lenses[r->optics->n - 1]->thick_type = normal_feff;
									r->optics->lenses[r->optics->n - 1]->thick_pars = NULL;
									break;
								case 3:
									if( strcmp( word_ith( parsed, 1 ), "from" ) == 0 )
										{
											r->optics->lenses[r->optics->n - 1]->thick = to_inf_double( 0 );
											r->optics->lenses[r->optics->n - 1]->thick_type = position_from;
											r->optics->lenses[r->optics->n - 1]->thick_pars = (inf_double_s *) malloc( thick_type_npars( position_from ) * sizeof( inf_double_s ) );
											if( !r->optics->lenses[r->optics->n - 1]->thick_pars )
												{
													word_free( parsed );
													word_free( words );
													lenses_free( r );
													return NULL;
												}
											r->optics->lenses[r->optics->n - 1]->thick_pars[0] = str_to_inf_double( word_ith( parsed, 2 ), NULL );
											r->optics->lenses[r->optics->n - 1]->thick_pars[1] = str_to_inf_double( word_ith( parsed, 0 ), NULL );
											break;
										}
									// fall through to error
								default:	// error
									fprintf( stderr, "%s:%d error parsing thickness.\n", path, iline );
									word_free( parsed );
									word_free( words );
									lenses_free( r );
									return NULL;
							}
						word_free( parsed );
					}

					word_free( words );
					continue;
				}

			/* object distance */
			{
				inf_double_s *par;
				par = NULL;
				if( strcmp( "star", word_ith( words, 0 ) ) == 0 ) par = &r->star_obj;
				if( strcmp( "pupil", word_ith( words, 0 ) ) == 0 ) par = &r->pupil_obj;
				if( par )
					{
						if( word_n( words ) != 2 )
							{
								fprintf( stderr, "%s:%d parse error: invalid number of parameters: %d.\n", path, iline, word_n( words ) );
								word_free( words );
								lenses_free( r );
								return NULL;
							}
						*par = str_to_inf_double( word_ith( words, 1 ), NULL );
						continue;
					}
			}

			/* merit function */
			{
				int object_number = -1;
				mf_operand_t type;
				char *typeptr;

				type = mf_none;
				typeptr = strrchr( word_ith( words, 0 ), '_' );
				if( strcmp( typeptr, "_img" ) == 0 ) type = mf_img;
				if( strcmp( typeptr, "_mag" ) == 0 ) type = mf_mag;
				if( strcmp( typeptr, "_amag" ) == 0 ) type = mf_amag;
				if( strncmp( word_ith( words, 0 ), "star", strcspn( word_ith( words, 0 ), "_" ) ) == 0 ) object_number = 0;
				if( strncmp( word_ith( words, 0 ), "pupil", strcspn( word_ith( words, 0 ), "_" ) ) == 0 ) object_number = 1;
				if( type != mf_none && word_n( words ) >= 2 )
					{
						char *cpcomment;
						int surface;
						inf_double_s object, goal;
						double weight;
						goal = str_to_inf_double( word_ith( words, 1 ), NULL );
						if (word_n( words ) >= 3)
							weight = atof( word_ith( words, 2 ) );
						else
							weight = 1.0;
						object = to_inf_double( 0 );
						if( word_n( words ) > 3 )
							{
								surface = atoi( word_ith( words, 3 ) );
								if( word_n( words ) > 4 )
									{
										object_number = -1;
										object = str_to_inf_double( word_ith( words, 4 ), NULL );
									}
							}
						else
							{
								surface = -1;
							}

						/* copy the comment */
						if( comment )
							{
								cpcomment = strdup( comment );
								if( cpcomment[strlen(cpcomment) - 1] == '\n' ) cpcomment[strlen(cpcomment) - 1] = '\0';
							}
						else
							{
								cpcomment = 0;
							}

						r->mf = mf_operand_add( r->mf, word_ith( words, 0 ), object, object_number, type, goal, surface, weight, cpcomment );
						continue;
					}
			}

			fprintf( stderr, "%s:%d parse error: invalid number of parameters: %d.\n", path, iline, word_n( words ) );
			word_free( words );
			lenses_free( r );
			return NULL;
		}

	/* set object distance on merit functions */
	{
		merit_function_operands_s *cur;

		cur = r->mf;
		while( cur )
			{
				switch( cur->object_number )
					{
						case 0:	// star
							cur->object = r->star_obj;
							break;
						case 1:	// pupil
							cur->object = r->pupil_obj;
							break;
					}
				cur = cur->next;
			}
	}

	if( strcmp( "-", path ) != 0 ) fclose( infile );
	return r;
}




