/* fitting.c : fits a circle onto a set of pixels */
/*
 * Copyright (C) 2003 Daigo Tomono <tomono at mpe.mpg.de>
 *
 * Permission is granted for use, copying, modification, distribution,
 * and distribution of modified versions of this work under the terms of
 * GPL version 2 or later.
 */
static char *rcsid __attribute__ ((unused)) =
	"$Id: fitting.c,v 1.7 2003/04/21 22:50:07 mos Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_multifit.h>
#include <gsl/gsl_multifit_nlin.h>
#include <gsl/gsl_statistics_double.h>
#include <gsl/gsl_fit.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_sort.h>
#include <gsl/gsl_statistics.h>

#include "fitting.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

/*
 *
 * defnitinos and prototypes
 *
 */

#define CIRCLE_NPARS 3	// x, y, and r

/* internal data structures */

typedef struct {
	double x; 	// x-coordinate of the pixel
	double y; 	// y-coordinate of the pixel
	double dx;	// x_center - x
	double dy;	// y_center - y 
	double f; 	// radius difference between model and the pixel
	int use;  	// 0:ignored in fitting 1:used in fitting
} pixel_on_circle_s;

typedef struct {
	size_t n;       	// number of pixels
	size_t n_used;   	// number of pixels used in calculation
	size_t n_changed;	// number of pixels changed usage status
	double sigma;   	// sigma = sqrt( chi-squared ) / N
	double chisq;    	// chi-squared
	pixel_on_circle_s * pixels;	// pixel data
	double r;       	// current r
	double r2;       	// current r*r
} pixels_on_circle_s;

/* chi-square calculations */
inline void _update_pixels_on_circle( pixels_on_circle_s * pixels, const
	gsl_vector * params );
inline double _circle_f( const pixels_on_circle_s * pixels, const size_t i );
inline void _set_circle_f_diff( gsl_matrix * J, const gsl_vector * params,
	const pixels_on_circle_s * pixels, const size_t idat );

/* functinos called from GSL */
/* x[0]: x-center, x[1]: y-center, x[2]: radius */
int _gsl_fit_f( const gsl_vector * x, void * params, gsl_vector * f );
int _gsl_fit_df( const gsl_vector * x, void * params, gsl_matrix * J );
int _gsl_fit_fdf( const gsl_vector * x, void * params, gsl_vector * f,
	gsl_matrix * J );

/* pixel management */
inline void _update_stat_on_circle( pixels_on_circle_s * pixels, const
	gsl_multifit_fdfsolver * s );
inline void _update_usage_on_circle( pixels_on_circle_s * pixels, const
	gsl_multifit_fdfsolver * s, const double sigma_rej ); 

/*
 *
 * implementatinos
 *
 */

char *
fit_str_error( fit_error_t err )
{
	switch( err ) {
		case FIT_GOOD:	return "no error.";
		case FIT_NOMEMORY:	return "out of memory.";
		case FIT_TOO_MANY_ITER:	return "too many iterations.";
		case FIT_TOO_FEW_PIX:	return "not enough data to be fit.";
		default: return "unknown error.";
	}
}

/*
 * difference between observation and model
 * we fit f([x,y,r],[xi,yi]) = (x-xi)^2 + (y-yi)^2 - r^2
 */
inline void
_update_pixels_on_circle( pixels_on_circle_s * pixels,
	const gsl_vector * params )
{
	size_t i;
	double x, y, r, t;

	#ifdef DEBUG
		fputs( "[_update_pixels_on_circle():", stderr );
	#endif

	x = gsl_vector_get( params, 0 );
	y = gsl_vector_get( params, 1 );
	r = gsl_vector_get( params, 2 );
	pixels->r = r;
	pixels->r2 = r*r;
	#ifdef DEBUG
		fprintf( stderr, " x:%g y:%g r:%g", x, y, r);
	#endif

	for(i = 0; i < pixels->n; i++)
		if( pixels->pixels[i].use )
			{
				pixels->pixels[i].dx = x - pixels->pixels[i].x;
				pixels->pixels[i].dy = y - pixels->pixels[i].y;
				t = pixels->pixels[i].dx * pixels->pixels[i].dx
					+ pixels->pixels[i].dy * pixels->pixels[i].dy
					- pixels->r2;
				pixels->pixels[i].f = t;
			}

	#ifdef DEBUG
		fputs( "]\n", stderr );
	#endif
}

/* difference */
inline double
_circle_f( const pixels_on_circle_s * pixels, const size_t i )
{
	if(! pixels->pixels[i].use )
		return 0;

	return pixels->pixels[i].f;
}

/* differentials */
inline void
_set_circle_f_diff( gsl_matrix * J, const gsl_vector * params,
	const pixels_on_circle_s * pixels, const size_t idat )
{
	if(! pixels->pixels[idat].use )
		{
			gsl_matrix_set( J, idat, 0, 0 );
			gsl_matrix_set( J, idat, 1, 0 );
			gsl_matrix_set( J, idat, 2, 0 );
			return;
		}

	gsl_matrix_set( J, idat, 0, 2 * pixels->pixels[idat].dx );
	gsl_matrix_set( J, idat, 1, 2 * pixels->pixels[idat].dy );
	gsl_matrix_set( J, idat, 2, -2 * pixels->r );
}

/*
 * updates
 */
inline void
_update_stat_on_circle( pixels_on_circle_s * pixels,
	const gsl_multifit_fdfsolver * s )
{
	pixels->chisq = pow( gsl_blas_dnrm2( s->f ), 2 );
	pixels->sigma = sqrt( pixels->chisq ) / pixels->n_used;
}

inline void
_update_usage_on_circle( pixels_on_circle_s * pixels,
	const gsl_multifit_fdfsolver * s, const double sigma_rej )
{
	size_t i;
	double thresh;
	double x, y, r2, dx, dy, delta;

	thresh = pixels->sigma * sigma_rej;
	x = gsl_vector_get( s->x, 0 );
	y = gsl_vector_get( s->x, 1 );
	r2 = gsl_vector_get( s->x, 2 );
	r2 *= r2;

	pixels->n_used = 0;
	pixels->n_changed = 0;
	for( i = 0; i < pixels->n; i++ )
		{
			dx = pixels->pixels[i].x - x;
			dy = pixels->pixels[i].y - y;
			delta = dx*dx + dy*dy - r2;
			if( sigma_rej >= 0.0 && fabs( delta ) > thresh )
				{
					if( pixels->pixels[i].use )
						pixels->n_changed++;
					pixels->pixels[i].use = 0;
				}
			else
				{
					if(! pixels->pixels[i].use )
						pixels->n_changed++;
					pixels->pixels[i].use = 1;
					pixels->n_used++;
				}
		}
}

/*
 * functinos called from GSL
 */
int
_gsl_fit_f( const gsl_vector * x, void * params, gsl_vector * f )
{
	size_t i;
	pixels_on_circle_s * pixels =  params;

	#ifdef DEBUG
		fputs( "[n_gsl_fit_f():", stderr );
	#endif

	_update_pixels_on_circle( pixels, x );
	for( i = 0; i < pixels->n; i++ )
		{
			gsl_vector_set( f, i, _circle_f( pixels, i ) );
			#ifdef DEBUG
				fprintf( stderr, " %d:%g", i, gsl_vector_get( f, i ) );
			#endif
		}

	#ifdef DEBUG
		fputs( "]\n", stderr );
	#endif

	return GSL_SUCCESS;
}

int
_gsl_fit_df( const gsl_vector * x, void * params, gsl_matrix * J )
{
	size_t idat, ipar;
	pixels_on_circle_s * pixels =  params;

	_update_pixels_on_circle( pixels, x );
	for( idat = 0; idat < pixels->n; idat++ )
		for( ipar = 0; ipar < CIRCLE_NPARS; ipar++ )
			_set_circle_f_diff( J, x, pixels, idat );

	return GSL_SUCCESS;
}

int
_gsl_fit_fdf( const gsl_vector * x, void * params, gsl_vector * f,
	gsl_matrix * J )
{
	size_t idat, ipar;
	pixels_on_circle_s * pixels =  params;

	_update_pixels_on_circle( pixels, x );
	for( idat = 0; idat < pixels->n; idat++ )
		{
			gsl_vector_set( f, idat, _circle_f( pixels, idat ) );
			for( ipar = 0; ipar < CIRCLE_NPARS; ipar++ )
				_set_circle_f_diff( J, x, pixels, idat );
		}

	return GSL_SUCCESS;
}


/*
 * public functions: fitting a circle
 */

void
guess_circle( circle_s * guessed_circle,
  const size_t npix, const double x[], const double y[] )
{
	size_t i;
	double * r;

	/* initialize */
	guessed_circle->x = 0;
	guessed_circle->y = 0;
	guessed_circle->r = 0;
	guessed_circle->dx = 0;
	guessed_circle->dy = 0;
	guessed_circle->dr = 0;

	/* guess center as centroid of the pixels */
	for( i = 0; i < npix; i++)
		{
			guessed_circle->x += x[i];
			guessed_circle->y += y[i];
		}
	guessed_circle->x /= npix;
	guessed_circle->y /= npix;

  r = (double *) malloc( sizeof( double ) * npix );
	if( !r )
		{
			/* guess radius as average of distances if not enough memory */
			double dx, dy;
			guessed_circle->r = 0;
			for( i = 0; i < npix; i++)
				{
					dx = x[i] - guessed_circle->x;
					dy = y[i] - guessed_circle->y;
					guessed_circle->r = dx * dx + dy * dy;
				}
			guessed_circle->r /= npix;
			guessed_circle->r = sqrt( guessed_circle->r );
		}
	else
		{
			/* guess radius as median of distances */
			double dx, dy;
			for( i = 0; i < npix; i++)
				{
					dx = x[i] - guessed_circle->x;
					dy = y[i] - guessed_circle->y;
					r[i] = dx * dx + dy * dy;
				}
			gsl_sort( r, 1, npix );
			guessed_circle->r = sqrt( gsl_stats_median_from_sorted_data( r, 1, npix ) );
		}
}

fit_error_t
fit_circle( circle_s * fitted_circle, fit_result_s * fit_result,
	const circle_s * initial_circle,
	const size_t npix, const double x[], const double y[],
	const size_t imax, const double sigma_rej, const int vf, const double
	eps_abs, const double eps_rel )
{
	size_t i, iter;
	pixels_on_circle_s pixels;
	int status;
	gsl_vector * initpars;
	gsl_multifit_function_fdf f;
	const gsl_multifit_fdfsolver_type *T;
	gsl_multifit_fdfsolver *s;

	/* initialization of fitting results */
	*fitted_circle = *initial_circle;
	fitted_circle->dx = 0;
	fitted_circle->dy = 0;
	fitted_circle->dr = 0;
	fit_result->iter = 0;
	fit_result->n_used = 0;
	fit_result->n_changed = 0;
	fit_result->chisq = 0;
	fit_result->sigma = 0;
	fit_result->err = FIT_GOOD;

	/* initialization of internal data structure */
	pixels.n = npix;
	pixels.n_used = npix;
	pixels.n_changed = 0;
	pixels.sigma = 0;
	pixels.chisq = 0;
	if( npix < CIRCLE_NPARS )
		{
			fit_result->err = FIT_TOO_FEW_PIX;
			return( fit_result->err );
		}
	pixels.pixels = ( pixel_on_circle_s * )
		malloc( sizeof( pixel_on_circle_s ) * npix );
	if( pixels.pixels == NULL )
		{
			fit_result->err = FIT_NOMEMORY;
			return( fit_result->err );
		}
	for(i = 0; i < npix; i++)
		{
			pixels.pixels[i].x = x[i];
			pixels.pixels[i].y = y[i];
			pixels.pixels[i].use = 1;
		}

	/* initialization for the GSL multifit library */
	f.f = &_gsl_fit_f;
	f.df = &_gsl_fit_df;
	f.fdf = &_gsl_fit_fdf;
	f.n = npix;
	f.p = CIRCLE_NPARS;
	f.params = &pixels;

	/* initialization of the GSL resolver */
	T = gsl_multifit_fdfsolver_lmsder;
	s = gsl_multifit_fdfsolver_alloc( T, f.n, f.p );
	initpars = gsl_vector_alloc( CIRCLE_NPARS );
	gsl_vector_set( initpars, 0, initial_circle->x );
	gsl_vector_set( initpars, 1, initial_circle->y );
	gsl_vector_set( initpars, 2, initial_circle->r );
	gsl_multifit_fdfsolver_set( s, &f, initpars );
	gsl_vector_free( initpars );
	initpars = NULL;

	/* iterations */
	iter = 0;
	do
		{
			iter++;
			if( vf > 2 )
				fprintf( stderr, "%3dth iteration:", iter );

			/* iterate */
			status = gsl_multifit_fdfsolver_iterate( s );
			_update_stat_on_circle( &pixels, s );

			/* check continuation */
			if( status == GSL_CONTINUE || status == GSL_SUCCESS )
				status = gsl_multifit_test_delta( s->dx, s->x, eps_abs, eps_rel );
			if( vf > 2 )
				{
					if( vf > 3 )
						{
							fprintf( stderr, " |f(x)|=%8g (%d data): %s\n",
								gsl_blas_dnrm2( s->f ), pixels.n_used,
								gsl_strerror( status ) );
							if( vf > 4 )
								{
									size_t i;
									for( i = 0; i < CIRCLE_NPARS; i++ )
										{
											fprintf( stderr, "\t\tp%d: %+9.3g => %-9.3g\n", i,
												gsl_vector_get( s->dx, i ),
												gsl_vector_get( s->x, i ) );
										}
								}
						}
					else
						{
							fputs( "\n", stderr );
						}
				}

			/* update pixel status */
			if( status == GSL_CONTINUE )
				{
					_update_usage_on_circle( &pixels, s, sigma_rej );
					if( vf > 3 )
						fprintf( stderr, "\t%d pixels changed their status.\n",
							pixels.n_changed);
				}
			else
				{
					if( vf > 3 )
						fputs( "\titeration finished.\n", stderr );
				}

		} while( status == GSL_CONTINUE && iter < imax
			&& pixels.n_used >= CIRCLE_NPARS );

	/* results */
	fit_result->iter = iter;
	fit_result->n_used = pixels.n_used;
	fit_result->n_changed = pixels.n_changed;
	fit_result->chisq = pixels.chisq;
	fit_result->sigma = pixels.sigma;

	fitted_circle->x = gsl_vector_get( s->x, 0 );
	fitted_circle->y = gsl_vector_get( s->x, 1 );
	fitted_circle->r = gsl_vector_get( s->x, 2 );

	{
		gsl_matrix *covar;
		covar = gsl_matrix_alloc( CIRCLE_NPARS, CIRCLE_NPARS );
		if(! covar )
			{
				fit_result->err = FIT_NOMEMORY;
				return( fit_result->err );
			}
		gsl_multifit_covar( s->J, 0.0, covar );
		fitted_circle->dx = sqrt( gsl_matrix_get( covar, 0, 0 ) );
		fitted_circle->dy = sqrt( gsl_matrix_get( covar, 1, 1 ) );
		fitted_circle->dr = sqrt( gsl_matrix_get( covar, 2, 2 ) );
		gsl_matrix_free( covar );
	}

	/* check iteration status */
	if( iter >= imax )
		fit_result->err = FIT_TOO_MANY_ITER;
	if( pixels.n_used < CIRCLE_NPARS )
		fit_result->err = FIT_TOO_FEW_PIX;

	free( pixels.pixels );
	pixels.pixels = 0;
	gsl_multifit_fdfsolver_free( s );
	s = 0;

	return( fit_result->err );
}

/* display */
void
show_circle( const circle_s * circle, FILE * stream )
{
  fprintf( stream, "center: %7.2f+-%6.2f, %7.2f+-%6.2f radius: %7.2f+-%6.2f\n",
		circle->x, circle->dx, circle->y, circle->dy, circle->r, circle->dr
		);
}

void
show_fit_result( const fit_result_s * result, FILE * stream )
{
	fprintf( stream, "%d iterations on %d data (final usage change %d):\n",
		result->iter, result->n_used, result->n_changed );
	fprintf( stream, "  chi^2:%g sigma:%g status:%s\n",
		result->chisq, result->sigma, fit_str_error( result->err ) );
}
