/* fits.h : header file for fits.h */
/*
 * Copyright (C) 2003 Daigo Tomono <tomono at mpe.mpg.de>
 *
 * Permission is granted for use, copying, modification, distribution,
 * and distribution of modified versions of this work under the terms of
 * GPL version 2 or later.
 */
#ifndef fits_h
#define fits_h

static char *fits_h_rcsid __attribute__ ((unused)) =
	"$Id: fits.h,v 1.6 2003/04/23 12:51:48 mos Exp $";

#include <stdlib.h>
#include <cfitsio/fitsio.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* error codes */
typedef enum {
	FITS_GOOD = 0,  	// no error : 0
	FITS_CFITSIO_ERR,	// error in cfitsio library
	FITS_NOMEMORY,  	// out of memory
} fits_error_t;

/* data structures */
struct _boundary_pixel_s {
	int dim;                       	// dimension of the pixels
	long *x;                       	// array of coordinates
	struct _boundary_pixel_s *next;	// next boundary_pixel_s
};
typedef struct _boundary_pixel_s boundary_pixel_s;

typedef struct {
	struct _boundary_pixel_s *first;	// first pixel in the list
	struct _boundary_pixel_s *last; 	// last pixel in the list
	fits_error_t err;  	// error status
	int cfitsio_status;	// status code from cfitsio library
} boundary_pixels_s;

/* functions */
char * fits_str_error( fits_error_t err );
char * fits_boundary_str_error( const boundary_pixels_s *pixels );
void show_pos( FILE *stream, long *pos, int naxis );

/* additions to cfitsio: header handling after copying */
void fits_decrease_dimension( fitsfile *file, int *status );
void fits_set_bitpix( fitsfile *file, const int bitpix, int *status );
long *fits_get_image_dimension( fitsfile *infits, int *naxis, int *status );

/* simple image arrays */
void fits_set_unsigned_char_pix( unsigned char *image,
	const unsigned char value, const long *position, const int naxis,
	const long *naxes );
unsigned char *fits_alloc_unsigned_char_image_buffer( const int naixs,
	const long *naxes );
double fits_get_double_pix( const double *image,
	const long *position, const int naxis, const long *naxes );
long fits_npix( const int naxis, const long *naxes );

/*
 * We will rely on cfitsio library for basic I/O and data handling of
 * FITS files. See cfitsio documents for details. Here, we only define
 * additional functions which accepts FITS file through fitsfile * which
 * is defined in cfitsio/fitso.h.
 */

/*
fits_get_boundary_upper_pixels() picks up pixels which are
(1) with value equal or more than VALUE_MINIMUM, and
(2) with more than LOWER_PIXELS * number of adjacent pixels (eight in
    case of a 2D image: right, left, up, down, and four corner pixels)
		which have values lower than VALUE_MINIMUM.

outputs:
	pixels: pixels which are picked up. This function malloc()s memory for
	        pixels->x and pixels->y. Free them using
	        fits_boundary_pixels_free().

inputs:
	fitsfile:      The fits file to be looked in. It has to be fits_open_file()ed.
	value_minimum: minimum value of the boundary pixels (see above).
	lower_pixels:  rario of number of adjacent pixels with lower values
	               (see above).
	projection:    only use PROJECTIONth plane (1, 2, ...) if PROJECTION
	               is more than zero. Otherwise, use all the dimensions.
 */

fits_error_t fits_get_boundary_upper_pixels( boundary_pixels_s * pixels,
	fitsfile * fitsfile, const double value_minimum, const double lower_pixels,
	const long projection );

/* pixel handling */
boundary_pixel_s *fits_boundary_pixel_new( const int axis, const long *axes );
void fits_boundary_pixel_delete( boundary_pixel_s *pixel );

/* pixels handling */
boundary_pixels_s *fits_boundary_pixels_new( void );
void fits_boundary_pixels_clear( boundary_pixels_s *pixels );
void fits_boundary_pixels_delete( boundary_pixels_s *pixels );

/* number of pixels */
long fits_boundary_pixels_length( const boundary_pixels_s *pixels );

/* add one pixel */
void fits_boundary_pixels_add( boundary_pixels_s *pixels,
  boundary_pixel_s *pixel );

/* iterator */
boundary_pixel_s *fits_boundary_pixels_first(
	const boundary_pixels_s *pixels );
boundary_pixel_s *fits_boundary_pixels_next( const boundary_pixel_s *pixel );
void firstpix( long *curpos, const int naxis );
int nextpix( long *curpos, const int naxis, const long *naxes );

#endif /* fits_h */
