/* fits.c : handles FITS data */
/*
 * Copyright (C) 2003 Daigo Tomono <tomono at mpe.mpg.de>
 *
 * Permission is granted for use, copying, modification, distribution,
 * and distribution of modified versions of this work under the terms of
 * GPL version 2 or later.
 */
static char *rcsid __attribute__ ((unused)) =
	"$Id: fits.c,v 1.11 2003/09/26 16:38:04 mos Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <cfitsio/fitsio.h>

#include "fits.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

/*
 *
 * defnitinos and prototypes
 *
 */

fits_error_t _cfitsio_error_check( boundary_pixels_s * pixels );
fits_error_t _boundary_pixels_malloc_check(
	boundary_pixels_s * pixels, void * allocated );

inline int _read_int_pix( int *image, long *position, int naxis, long *naxes );

/* iterators */
inline void _firstpix( long *curpos, const int naxis );
inline int _nextpix( long *curpos, const int naxis, const long *naxes );

inline void _adjacent_init( long *curpos, const long *centerpos,
	const int naxis, long *shift );
inline int _adjacent_next( long *curpos, const long *centerpos,
	const int naxis, const long *naxes, long *shift );
	// 0: end, 1: continue
inline int _adjacent_check( long *curpos, const long *centerpos,
	const int naxis, const long *naxes, long *shift );
	// 0: curpos at good location, 1: at bad location

/*
 *
 * implementatinos
 *
 */

fits_error_t
_cfitsio_error_check( boundary_pixels_s *pixels )
{
	if( pixels->cfitsio_status )
		pixels->err = FITS_CFITSIO_ERR;
	return pixels->err;
}

fits_error_t
_boundary_pixels_malloc_check( boundary_pixels_s *pixels, void *allocated )
{
	if( allocated == NULL )
		pixels->err = FITS_NOMEMORY;
	return pixels->err;
}

void
show_pos( FILE *stream, long *pos, int naxis )
{
	int i;
	for( i = 0; i < naxis; i++ )
		{
			if( i > 0 ) fputs( ",", stream );
			fprintf( stream, "%ld", pos[i] );
		}
}

inline int
_read_int_pix( int *image, long *position, int naxis, long *naxes )
{
	int axis;
	long pos;

	pos = position[naxis - 1] - 1;
	for( axis = naxis - 2; axis >= 0; axis-- )
		{
			pos *= naxes[axis + 1];
			pos += position[axis] - 1;
		}
	return image[pos];
}

void
firstpix( long *curpos, const int naxis )
{
	_firstpix( curpos, naxis );
}

int
nextpix( long *curpos, const int naxis, const long *naxes )
{
	return _nextpix( curpos, naxis, naxes );
}

inline void
_firstpix( long *curpos, const int naxis )
{
	int i;
	for( i = 0; i < naxis; i++ )
		curpos[i] = 1;
}

inline int
_nextpix( long *curpos, const int naxis, const long *naxes )
{
	int i;

	i = 0;
	do
		{
			curpos[i]++;
			if( curpos[i] > naxes[i] )
				{
					curpos[i] = 1;
					i++;
				}
			else
				break;
		} while( i < naxis );

	if( i < naxis )
		return 1;
	else
		return 0;
}

inline void
_adjacent_init( long *curpos, const long *centerpos, const int naxis,
	long *shift )
{
	int i;

	for( i = 0; i < naxis; i++ )
		{
			shift[i] = -1;
			curpos[i] = centerpos[i] + shift[i];
		}
}

inline int
_adjacent_next( long *curpos, const long *centerpos, const int naxis,
	const long *naxes, long *shift )
{
	int i;

	i = 0;
	do
		{
			shift[i]++;
			curpos[i] = centerpos[i] + shift[i];
			if( shift[i] > 1 )
				{
					shift[i] = -1;
					curpos[i] = centerpos[i] + shift[i];
					i++;
				}
			else
				break;
		} while( i < naxis );

	if( i < naxis )
		return 1;
	else
		return 0;
}

inline int
_adjacent_check( long *curpos, const long *centerpos, const int naxis,
	const long *naxes, long *shift )
{
	int i, allzero;

	for( i = 0, allzero = 1; i < naxis; i++ )
		{
			if( curpos[i] < 1 ) return 1;
			if( curpos[i] > naxes[i] ) return 1;
			if( shift[i] != 0 ) allzero = 0;
		}

	return allzero;
}

/*
 * public functions
 */

char *
fits_str_error( fits_error_t err )
{
	switch( err )
		{
			case FITS_GOOD:	return "no error.";
			case FITS_NOMEMORY:	return "out of memory.";
			case FITS_CFITSIO_ERR:	return "error in cfitsio library.";
			default:	return "unknown error.";
		}
}

char *
fits_boundary_str_error( const boundary_pixels_s *pixels )
{
	static char cfitsio_error[FLEN_STATUS];
	if( pixels->err == FITS_CFITSIO_ERR )
		{
			fits_get_errstatus( pixels->cfitsio_status, cfitsio_error );
			return cfitsio_error;
		}
	else
		{
			return fits_str_error( pixels->err );
		}
}

fits_error_t
fits_get_boundary_upper_pixels( boundary_pixels_s *pixels,
	fitsfile *fitsfile, const double value_minimum,
	const double lower_pixels, const long projection )
{
	int naxis, projaxis;
	long *iaxes = NULL, *adjpos = NULL, *shift = NULL, *naxes = NULL;
	int *flagimage = NULL;

	pixels->err = FITS_GOOD;
	pixels->cfitsio_status = 0;

	/* dimension of the fits data */
	#ifdef DEBUG
		fputs( "dimension\n", stderr );
	#endif
	fits_get_img_dim( fitsfile, &naxis, &(pixels->cfitsio_status) );
	if( _cfitsio_error_check( pixels ) ) return pixels->err;
	if( projection > 0 )
		projaxis = naxis - 1;
	else
		projaxis = naxis;
	if( pixels->err ) return pixels->err;
	naxes = (long *) malloc( sizeof( long ) * naxis );
	iaxes = (long *) malloc( sizeof( long ) * naxis );
	adjpos = (long *) malloc( sizeof( long ) * naxis );
	shift = (long *) malloc( sizeof( long ) * projaxis );
	if( _boundary_pixels_malloc_check( pixels, naxes )
		|| _boundary_pixels_malloc_check( pixels, iaxes )
		|| _boundary_pixels_malloc_check( pixels, adjpos )
		|| _boundary_pixels_malloc_check( pixels, shift ) )
		{
			if( naxes != NULL ) free( naxes );
			if( iaxes != NULL ) free( iaxes );
			if( adjpos != NULL ) free( adjpos );
			if( shift != NULL ) free( shift );
			return pixels->err;
		}
	fits_get_img_size( fitsfile, naxis, naxes, &(pixels->cfitsio_status));
	if( _cfitsio_error_check( pixels ) )
		{
			free( naxes );
			free( iaxes );
			free( adjpos );
			free( shift );
			return pixels->err;
		}

	/* copy the image */
	{
		long i, npixel;
		double *image;
		for( i = 0, npixel = 1; i < projaxis; i++ )
			npixel *= naxes[i];
		image = (double *) malloc( sizeof( double ) * npixel );
		flagimage = (int *) malloc( sizeof( int ) * npixel );
		if( _boundary_pixels_malloc_check( pixels, image )
			|| _boundary_pixels_malloc_check( pixels, flagimage ) )
			{
				if( image != NULL ) free( image );
				if( flagimage != NULL ) free( flagimage );
				free( naxes );
				free( iaxes );
				free( adjpos );
				free( shift );
				return pixels->err;
			}
		_firstpix( iaxes, projaxis );
		if( projection > 0 )
			iaxes[naxis - 1] = projection;
		fits_read_pix( fitsfile, TDOUBLE, iaxes, npixel, NULL, image, NULL,
			&(pixels->cfitsio_status) );
		if( _cfitsio_error_check( pixels ) )
			{
				free( image );
				free( naxes );
				free( iaxes );
				free( adjpos );
				free( shift );
				return pixels->err;
			}
		for( i = 0; i < npixel; i++ )
			flagimage[i] = ( image[i] >= value_minimum ) ? 1 : 0;
		free( image );
	}

	/* scan through the image */
	{
		size_t nthresh, nhigher;

		#ifdef DEBUG
			fputs( "scanning\n", stderr );
		#endif

		nthresh = (size_t) (lower_pixels * (pow( 3, (double) projaxis )  - 1 ));
		_firstpix( iaxes, projaxis );
		if( projection > 0 )
			{
				iaxes[naxis - 1] = 1;
				adjpos[naxis - 1] = 1;
			}
		do	
			{
				#ifdef DEBUG_ADJACENT
					fputs("adjacent of ", stderr);
					show_pos( stderr, iaxes, naxis );
					fputs(":", stderr);
				#endif
				nhigher = 0;
				if (! _read_int_pix( flagimage, iaxes, naxis, naxes ) )
					{
						int more = 1;
						_adjacent_init( adjpos, iaxes, projaxis, shift );
						while( more
							&& _adjacent_check( adjpos, iaxes, projaxis, naxes, shift ) )
							more = _adjacent_next( adjpos, iaxes, projaxis, naxes, shift );
						while( more )
							{
								#ifdef DEBUG_ADJACENT
									fputs(" ", stderr);
									show_pos( stderr, adjpos, naxis );
								#endif
								if( _read_int_pix( flagimage, adjpos, naxis, naxes ) )
									{
										nhigher++;
										#ifdef DEBUG_ADJACENT
											fputs( "-", stderr );
										#endif
									}
								else
									{
										#ifdef DEBUG_ADJACENT
											fputs( "-", stderr );
										#endif
									}
								more = _adjacent_next( adjpos, iaxes, projaxis, naxes, shift );
								while( more
									&& _adjacent_check( adjpos, iaxes, projaxis, naxes, shift ) )
									more = _adjacent_next( adjpos, iaxes, projaxis, naxes,
										shift );
							}
						#ifdef DEBUG_ADJACENT
							fputs( "\n", stderr );
						#endif
					}
				else
					{
						#ifdef DEBUG_ADJACENT
							fputs( "\n", stderr );
						#endif
					}
				if( _cfitsio_error_check( pixels ) )
					{
						free( naxes );
						free( iaxes );
						free( adjpos );
						free( shift );
						return pixels->err;
					}
				if( nhigher > nthresh )
					{
						fits_boundary_pixels_add( pixels,
							fits_boundary_pixel_new( projaxis, iaxes ) );
					}
			} while( _nextpix( iaxes, projaxis, naxes ) );
	}

	free( naxes );
	free( iaxes );
	free( adjpos );
	free( shift );
	free( flagimage );
	return pixels->err;
}

/* pixel handling */
boundary_pixel_s *
fits_boundary_pixel_new( const int axis, const long *axes )
{
	boundary_pixel_s *r;
	r = (boundary_pixel_s *) malloc( sizeof( boundary_pixel_s ) );
	if( r == NULL )
		return NULL;

	r->dim = axis;
	r->next = NULL;
	r->x = (long *) malloc( sizeof( long ) * axis );
	if( r->x == NULL )
		{
			free( r );
			return NULL;
		}
	memcpy( r->x, axes, sizeof( long ) * axis );
	return r;
}

void
fits_boundary_pixel_delete( boundary_pixel_s *pixel )
{
	free( pixel->x );
	free( pixel );
}

/* pixels handling */
boundary_pixels_s *
fits_boundary_pixels_new( void )
{
	boundary_pixels_s *r;
	r = (boundary_pixels_s *) malloc( sizeof( boundary_pixels_s ) );
	if( r == NULL )
		return NULL;
	r->first = NULL;
	r->last = NULL;
	r->err = FITS_GOOD;
	r->cfitsio_status = 0;
	return r;
}

void
fits_boundary_pixels_clear( boundary_pixels_s *pixels )
{
	boundary_pixel_s *current, *next;
	next = pixels->first;
	while( next != NULL )
		{
			current = next;
			next = current->next;
			fits_boundary_pixel_delete( current );
		}
	pixels->first = NULL;
	pixels->last = NULL;
}

void
fits_boundary_pixels_delete( boundary_pixels_s *pixels )
{
	fits_boundary_pixels_clear( pixels );
	free( pixels );
}

long
fits_boundary_pixels_length( const boundary_pixels_s *pixels )
{
	long n = 0;
	boundary_pixel_s *next;
	next = pixels->first;
	while( next != NULL )
		{
			n++;
			next = next->next;
		}
	return n;
}

void
fits_boundary_pixels_add( boundary_pixels_s *pixels,
	boundary_pixel_s *pixel )
{
	if( pixels->first == NULL )
		{
			pixels->first = pixel;
		}
	else
		{
			pixels->last->next = pixel;
		}
	pixels->last = pixel;
}

boundary_pixel_s *
fits_boundary_pixels_first( const boundary_pixels_s *pixels )
{
	return pixels->first;
}

boundary_pixel_s *
fits_boundary_pixels_next( const boundary_pixel_s *pixel )
{
	return pixel->next;
}

#define NAXISi "NAXISi"
void
fits_decrease_dimension( fitsfile *file, int *status )
{
	int naxis;
	char *naxis_last, *naxis_str = "NAXIS";
	naxis_last = (char *) malloc( sizeof( NAXISi ) );
	if( naxis_last == NULL )
		{
			fprintf( stderr, "%s:%d: could not allocate memory.\n",
				__FILE__, __LINE__ );
			exit( EXIT_FAILURE );
		}
	fits_get_img_dim( file, &naxis, status );
	if( naxis > 9 )
		{
			fprintf( stderr, "%s:%d: input file has too many dimension.\n",
				__FILE__, __LINE__ );
			fputs( "We have to modify the source file, sorry.\n", stderr );
			exit( EXIT_FAILURE );
		}
	if( naxis < 0 )
		{
			return;	// to protect snprintf
			// I hope we can rely on FITS file check for naxis < 2 case
		}
	snprintf( naxis_last, sizeof( NAXISi ), "NAXIS%d", naxis );
	fits_delete_key( file, naxis_last, status );
	if( *status )
		return;
	fits_modify_key_lng( file, naxis_str, naxis - 1, "", status );
}

void
fits_set_bitpix( fitsfile *file, const int bitpix, int *status )
{
	fits_modify_key_lng( file, "BITPIX", bitpix, "", status );
}

inline void
fits_set_unsigned_char_pix( unsigned char *image,
	const unsigned char value, const long *position,
	const int naxis, const long *naxes )
{
	int axis;
	long pos;

	pos = position[naxis - 1] - 1;
	for( axis = naxis - 2; axis >= 0; axis-- )
		{
			pos *= naxes[axis + 1];
			pos += position[axis] - 1;
		}
	image[pos] = value;
}

double
fits_get_double_pix( const double *image,
	const long *position, const int naxis, const long *naxes )
{
	int axis;
	long pos;

	pos = position[naxis - 1] - 1;
	for( axis = naxis - 2; axis >= 0; axis-- )
		{
			pos *= naxes[axis + 1];
			pos += position[axis] - 1;
		}
	return image[pos];
}

unsigned char *
fits_alloc_unsigned_char_image_buffer( const int naxis, const long *naxes )
{
	return (unsigned char *) malloc( sizeof( unsigned char ) * fits_npix
		( naxis, naxes ) );
}

long
fits_npix( const int naxis, const long *naxes )
{
	int i;
	long npix = 1;
	for( i = 0; i < naxis; i++ )
		npix *= naxes[i];
	return npix;
}

long *
fits_get_image_dimension( fitsfile *infits, int *naxis, int *status )
{ 
	long *naxes;
	/* image dimension */
	fits_get_img_dim( infits, naxis, status );
	if( *status )
		return NULL;
	naxes = (long *) malloc( sizeof( long ) * *naxis );
	if( naxes == NULL )
		return NULL;
	fits_get_img_size( infits, *naxis, naxes, status );
	if( *status )
		{
			free( naxes );
			return NULL;
		}
	return naxes;
} 

