/* fiber_photo.c : fiber photometry tool */
/*
 * Copyright (C) 2003 Daigo Tomono <tomono at mpe.mpg.de>
 *
 * Permission is granted for use, copying, modification, distribution,
 * and distribution of modified versions of this work under the terms of
 * GPL version 2 or later.
 */
static char *rcsid __attribute__ ((unused)) =
	"$Id: fiberphoto.c,v 1.9 2003/09/26 16:38:04 mos Exp $";

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <libgen.h>
#include <unistd.h>
#include <getopt.h>
#include <cfitsio/fitsio.h>
#include <gsl/gsl_statistics_double.h>
#include <gsl/gsl_sort.h>
#include <gsl/gsl_statistics.h>

#include "fits.h"
#include "fitting.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

/*
 *
 * defnitinos and prototypes
 *
 */

#define DEF_UPPER_RATIO 0.99
#define DEF_LOWER_RATIO 0.5
#define DEF_RADIUS_RATIO 0.8
#define DEF_BOUNDARY_RATIO 0.3
#define DEF_SIGMA_REJ 2.0
#define DEF_IMAX 100
#define DEF_EPS_ABS 1e-10
#define DEF_EPS_REL 1e-10

typedef struct
{
	char *locfilename;
	int autothresh;
	double manualthresh;
	double upper;
	double lower;
	double scale;
	double boundratio;
	circle_s *predef_aperture;
	char *boundaryfileext;
	char *aperturefileext;
	int vf;
	long imax;
	double sigma_rej;
	double eps_abs;
	double eps_rel;
	int nohead;
} fiberphoto_config_s;

typedef struct
{
	double total_in;
	double total_out;
	long n_in;
	long n_out;
} fiberphoto_photometry_s;

typedef struct
{
	char *filename;
	long projection;
	fiberphoto_photometry_s photo;
	circle_s *aperture;
	circle_s *circle;
	fit_result_s *fitting;
	char *locfilename;
	double threshold;
} fiberphoto_result_s;

#define malloc_check(ptr) \
{\
	if( (ptr) == NULL )\
		{\
			fprintf(stderr, "malloc error in %s:%d\n", __FILE__, __LINE__);\
			exit(EXIT_FAILURE);\
		}\
}

#define cfits_check(status) \
{\
	if( (status) )\
		{\
			fprintf(stderr, "%s:%d ", __FILE__, __LINE__);\
			fits_report_error( stderr, status );\
			exit(EXIT_FAILURE);\
		}\
}

/* command usage and configuration */
void usage_short( FILE *stream, char *argv[] );
void usage_long( FILE *stream, char *argv[] );
int fiberphoto_get_options( int argc, char *argv[],
	fiberphoto_config_s *config );
void fiberphoto_set_defaults( fiberphoto_config_s *config );
circle_s *str_to_circle( char *string );
char *fibermeas_ext_filename( const char *origfilename,
	const char *extension, long projection );

/* automatic recognization of the image */
double get_thresh( fitsfile *infits, const long projection,
	const double r1, const double r2);
fit_error_t fiber_fit_circle( fitsfile *infits, const long projection,
	circle_s *circle, fit_result_s *result, 
	const double thresh, const double boundaryratio,
	const size_t imax, const double simga_rej, const int vf,
	const double eps_abs, const double eps_rel,
	const char *filename, const char *boundaryfileext );

/* photometry */
void fiberphoto_measure( fitsfile *infits, const long projection,
	fiberphoto_photometry_s *result, const circle_s *aperture );
void show_result( FILE *stream, const fiberphoto_result_s *result );

/* image handling */
inline int in_aperture( const long *x, const circle_s *aperture );
void create_boundary_image( fitsfile *infits, long projection,
	const boundary_pixels_s *pixels, const char *infilename,
	const char *fileext );
void create_aperture_image( fitsfile *infits, long projection,
	const circle_s *aperture, const char *infilename, const char *fileext );

/*
 *
 * implementatinos
 *
 */

/* command line options */
#define OPTIONS "f:t:u:l:s:c:b:a:n:hHvqrko"
#define OPTIONS_LONG {\
	{"location", 1, NULL, 'f'},\
	{"threshold", 1, NULL, 't'},\
	{"upper", 1, NULL, 'u'},\
	{"lower", 1, NULL, 'l'},\
	{"scale", 1, NULL, 's'},\
	{"adjacent", 1, NULL, 'n'},\
	{"circle", 1, NULL, 'c'},\
	{"boundary", 1, NULL, 'b'},\
	{"aperture", 1, NULL, 'a'},\
	{"help", 0, NULL, 'H'},\
	{"verbose", 0, NULL, 'v'},\
	{"quiet", 0, NULL, 'q'},\
	{"sigmarej", 1, NULL, 'r'},\
	{"no-head", 0, NULL, 'k'},\
	{0, 0, NULL, 0}\
}

void
usage_short( FILE *stream, char *argv[] )
{
	char *buffer, *cmdname;
	buffer = strdup( argv[0] );
	malloc_check( buffer );
	cmdname = basename( buffer );
	fprintf( stream, "\
usage: %s [options] fits-file-name(s)\n\
\n\
This program finds the location of an image of the fiber output, fits a\n\
circle, and counts the pixel values inside and outside the circle.\n\
%s\n\
\n\
Options [defaults]:\n\
-f, --location=FITS-FILE-NAME [from the image being measured]\n\
-t, --threshold=THRESH-HOLD [automatically adujsted]\n\
-u, --upper=RATIO [%.3f]\n\
-l, --lower=RATIO [%.3f]\n\
-r, --sigmarej=SIGMA [%.3f]\n\
-s, --scale=RATIO [%.3f]\n\
-n, --adjacent=RATIO [%.3f]\n\
-c, --circle=X,Y,R [best fit on the image]\n\
-b, --boundary=EXTENSION\n\
-a, --aperture=EXTENSION\n\
-o, --no-head\n\
-v, --verbose\n\
-q, --quiet\n\
-H, --help shows detailed usage.\n\
",
	cmdname, rcsid, DEF_UPPER_RATIO, DEF_LOWER_RATIO, DEF_SIGMA_REJ,
	DEF_RADIUS_RATIO, DEF_BOUNDARY_RATIO );

	free( buffer );
}

void
usage_long( FILE *stream, char *argv[] )
{
	char *buffer, *cmdname;
	buffer = strdup( argv[0] );
	malloc_check( buffer );
	cmdname = basename( buffer );
	fprintf( stream, "\
usage: %s [options] fits-file-name(s)\n\
\n\
This program finds the location of an image of the fiber output, fits a\n\
circle, and counts the pixel values inside and outside the circle.\n\
%s\n\
\n\
Input fits files can be 2D or 3D image. For 3D images, photometry is\n\
applied on each 2D images. Results are shown in the standard output with\n\
the following fields separated with a tab:\n\
	 1. filename\n\
	 2. plane number (1 for 2D images)\n\
	 3. total ADU inside aperture\n\
	 4. total ADU outside aperture\n\
	 5. aperture center X (1...)\n\
	 6. aperture center Y (1...)\n\
	 7. aperture radius\n\
	 8. number of pixels inside aperture\n\
	 9. number of pixels outside aperture\n\
	 9. image circle radius\n\
	10. 1-sigma fitting error of aperture center X\n\
	11. 1-sigma fitting error of aperture cneter Y\n\
	12. 1-sigma fitting error of image circle radius\n\
	13. filename for image location detection\n\
	14. threshold\n\
\n\
Options [defaults]:\n\
\n\
Automatic detection of the fiber image location\n\
-f, --location=FITS-FILE-NAME\n\
	Finds the location from this 2D fits file.\n\
	[from the image being measured]\n\
\n\
-t, --threshold=THRESH-HOLD\n\
	Sets the threshhold value to be used in finding the circle.\n\
	[automatically adujsted]\n\
-u, --upper=RATIO\n\
	Picks up (ratio * number of pixels)-th the dakest pixel\n\
	as the value for fiber image. [%.3f]\n\
-l, --lower=RATIO\n\
	Picks up (ratio * number of pixels)-th the dakest pixel\n\
	as the value for fiber image. [%.3f]\n\
-r, --sgimarej=SIGMA\n\
	Sigma rejection limit for circle fitting. [%.3f]\n\
-s, --scale=RATIO\n\
	Scale of the aperture circle radius to the detected image\n\
	radius. [%.3f]\n\
-n, --adjacent=RATIO\n\
	Define the pixel if RATIO ratio of adjacent pixels have values\n\
	lower than the threshold. [%.3f]\n\
\n\
Manual defnition of the aperture\n\
-c, --circle=X,Y,R\n\
	Sets circle center coordinate (1..) and radius in pixels.\n\
	[best fit on the image]\n\
\n\
Check images\n\
-b, --boundary=EXTENSION\n\
	Creates FITS file(s) of image(s) of detected circle\n\
	boundary pixels. EXTENSION is added before the '.fits'\n\
	extension of the original file name. Add a dot to EXTENSION\n\
	when needed. [does not create]\n\
-a, --aperture=EXTENSION\n\
	Creates FITS file(s) of image(s) of the aperture used\n\
	to count the pixel values.\n\
\n\
-o, --no-head\n\
	Prints out no header.\n\
-v, --verbose\n\
-q, --quiet\n\
	Increase/decrease verbosity\n\
-h\n\
	Shows a list of options.\n\
",
	cmdname, rcsid, DEF_UPPER_RATIO, DEF_LOWER_RATIO, DEF_SIGMA_REJ,
	DEF_RADIUS_RATIO, DEF_BOUNDARY_RATIO );

	free( buffer );
}

int
fiberphoto_get_options( int argc, char *argv[], fiberphoto_config_s *config )
{
	int c, this_optind;
	static struct option long_options[] = OPTIONS_LONG;
	while( 1 )
		{
			this_optind = optind ? optind : 1;
			c = getopt_long( argc, argv, OPTIONS, long_options, NULL );
			if( c == -1 )
				break;
			switch( c )
				{
					case 'f':
						if( config->predef_aperture != NULL )
							{
								free( config->predef_aperture );
								config->predef_aperture = NULL;
							}
						config->locfilename = optarg;
						break;
					case 't':
						config->autothresh = 0;
						config->manualthresh = atof( optarg );
						break;
					case 'u':
						config->upper = atof( optarg );
						break;
					case 'l':
						config->lower = atof( optarg );
						break;
					case 's':
						config->scale = atof( optarg );
						break;
					case 'n':
						config->boundratio = atof( optarg );
						break;
					case 'c':
						if( config->predef_aperture != NULL )
							free( config->predef_aperture );
						config->predef_aperture = str_to_circle( optarg );
						config->locfilename = NULL;
						if( config->predef_aperture == NULL )
							{
								fprintf( stderr,
									"option %s: please provide comma seperated numbers for x, y, and r.\n",
									argv[this_optind] );
								exit( EXIT_FAILURE );
							}
						break;
					case 'b':
						config->boundaryfileext = optarg;
						break;
					case 'a':
						config->aperturefileext = optarg;
						break;
					case 'h':
						usage_short( stdout, argv );
						exit( EXIT_SUCCESS );
					case 'H':
						usage_long( stdout, argv );
						exit( EXIT_SUCCESS );
					case 'v':
						config->vf++;
						break;
					case 'q':
						config->vf--;
						break;
					case 'o':
						config->nohead = 1;
						break;
					case '?':
						usage_short( stderr, argv );
						exit( EXIT_FAILURE );
					default:
						fprintf( stderr, "oops: missing implementation for option: %s\n",
							argv[this_optind] );
						exit( EXIT_FAILURE );
				}
		}

	return optind;
}

void
fiberphoto_set_defaults( fiberphoto_config_s *config )
{
	config->locfilename = NULL; // NULL: use the image being measured
	config->autothresh = 1;
	config->manualthresh = 0;
	config->upper = DEF_UPPER_RATIO;
	config->lower = DEF_LOWER_RATIO;
	config->scale = DEF_RADIUS_RATIO;
	config->boundratio = DEF_BOUNDARY_RATIO;
	config->predef_aperture = NULL; // NULL: fit circle on image being measured
	config->boundaryfileext = NULL;
	config->aperturefileext = NULL; // NULL: no file creation
	config->vf = 1; // verbosity
	config->imax = DEF_IMAX;
	config->sigma_rej = DEF_SIGMA_REJ;
	config->eps_abs = DEF_EPS_ABS;
	config->eps_rel = DEF_EPS_REL;
	config->nohead = 0;
}
circle_s *
str_to_circle( char *string )
{
	circle_s *r;
	char *cptr, *nptr;

	r = (circle_s *) malloc( sizeof( circle_s ) );
	malloc_check( r );

	cptr = string;
	r->x = strtod( cptr, &nptr );
	if( cptr == nptr || *nptr != ',' )
		{
			free( r );
			return NULL;
		}
	cptr = nptr + 1;
	r->y = strtod( cptr, &nptr );
	if( cptr == nptr || *nptr != ',' )
		{
			free( r );
			return NULL;
		}
	cptr = nptr + 1;
	r->r = strtod( cptr, &nptr );
	if( cptr == nptr || *nptr != '\0' )
		{
			free( r );
			return NULL;
		}
	r->dx = 0;
	r->dy = 0;
	r->dr = 0;
	return r;
}

fit_error_t
fiber_fit_circle( fitsfile *infits, const long projection,
	circle_s *circle, fit_result_s *result, 
	const double thresh, const double boundaryratio,
	const size_t imax, const double sigma_rej, const int vf,
	const double eps_abs, const double eps_rel,
	const char *filename, const char *boundaryfileext )
{
	boundary_pixels_s *pixels;

	pixels = fits_boundary_pixels_new();
	malloc_check( pixels );

	/* boundary pixels */
	fits_get_boundary_upper_pixels( pixels, infits, thresh, boundaryratio,
		projection );
	if( pixels->err )
		{
			result->err = pixels->err;
			return pixels->err;
		}

	/* fitting */
	{
		long ipix, npix;
		double *x, *y;
		circle_s initial_circle;
		boundary_pixel_s *cpix;

		npix = fits_boundary_pixels_length( pixels );
		if( vf > 2 )
			fprintf( stderr, " %ld boundary pixels.\n", npix );
		x = (double *) malloc( sizeof( double ) * npix );
		malloc_check( x );
		y = (double *) malloc( sizeof( double ) * npix );
		malloc_check( y );

		cpix = fits_boundary_pixels_first( pixels );
		for( ipix = 0; ipix < npix; ipix++ )
			{
				x[ipix] = cpix->x[0];
				y[ipix] = cpix->x[1];
				cpix = fits_boundary_pixels_next( cpix );
			}

		guess_circle( &initial_circle, npix, x, y);
		if( vf > 3 )
			{
				fputs( "initial guess: ", stderr );
				show_circle( &initial_circle, stderr );
			}
		fit_circle( circle, result, &initial_circle, npix, x, y,
			imax, sigma_rej, vf, eps_abs, eps_rel );

		if( vf > 1 )
			{
				fputs( "\t", stderr );
				show_circle( circle, stderr );
			}
	}

	/* save the detected pixels if needed */
	if( boundaryfileext != NULL )
		{
			create_boundary_image( infits, projection, pixels, filename,
				boundaryfileext );
		}

	fits_boundary_pixels_delete( pixels );
	return result->err;
}

void
fiberphoto_measure( fitsfile *infits, const long projection,
	fiberphoto_photometry_s *result, const circle_s *aperture )
{
	double *values;
	int naxis, projaxis, status = 0;
	long npixel, *iaxes, *naxes;

	naxes = fits_get_image_dimension( infits, &naxis, &status );
	malloc_check( naxes );
	iaxes = (long *) malloc( sizeof( long ) * naxis );
	malloc_check( iaxes );
	if( projection > 0 )
		projaxis = naxis - 1;
	else
		projaxis = naxis;

  /* piexels */
	npixel = fits_npix( projaxis, naxes );
	firstpix( iaxes, projaxis );
	if( projection > 0 )
		iaxes[naxis - 1] = projection;
	values = (double *) malloc( sizeof( double ) * npixel );
	malloc_check( values );
	fits_read_pix( infits, TDOUBLE, iaxes, npixel, NULL, values, NULL,
		&status );
	cfits_check( status );

	/* count */
	result->total_in = 0;
	result->total_out = 0;
	result->n_in = 0;
	result->n_out = 0;
	firstpix( iaxes, projaxis );
	do
		{
			if( in_aperture( iaxes, aperture ) )
				{
					result->total_in += fits_get_double_pix( values, iaxes,
						projaxis, naxes );
					result->n_in++;
				}
			else
				{
					result->total_out += fits_get_double_pix( values, iaxes,
						projaxis, naxes );
					result->n_out++;
				}
		} while( nextpix( iaxes, projaxis, naxes ) );

	free( naxes );
}

inline int
in_aperture( const long *x, const circle_s *aperture )
{
	double dx, dy;
	dx = x[0] - aperture->x;
	dy = x[1] - aperture->y;
	return (dx*dx + dy*dy - aperture->r*aperture->r) <= 0;
}

double
get_thresh( fitsfile *infits, const long projection,
	const double r1, const double r2 )
{
	double v1, v2, *values;
	long npixel, *iaxes, *naxes;
	int projaxis, naxis;
	int status = 0;

	naxes = fits_get_image_dimension( infits, &naxis, &status );
	iaxes = (long *) malloc( sizeof( long ) * naxis );
	malloc_check( iaxes );
	if( projection > 0 )
		projaxis = naxis - 1;
	else
		projaxis = naxis;

  /* piexels */
	npixel = fits_npix( projaxis, naxes );
	firstpix( iaxes, projaxis );
	if( projection > 0 )
		iaxes[naxis - 1] = projection;
	values = (double *) malloc( sizeof( double ) * npixel );
	malloc_check( values );
	fits_read_pix( infits, TDOUBLE, iaxes, npixel, NULL, values, NULL,
		&status );
	cfits_check( status );

	/* sort and get percentiles */
	gsl_sort( values, 1, npixel );
	v1 = gsl_stats_quantile_from_sorted_data( values, 1, npixel, r1 );
	v2 = gsl_stats_quantile_from_sorted_data( values, 1, npixel, r2 );

	free( naxes );
	free( iaxes );
	free( values );

	return (v1 + v2) / 2;
}

char *
fibermeas_ext_filename( const char *origfilename, const char *extension,
	long projection )
{
	int total;
	char *result;
	char *basename, *dotfits, projectionstr[16], *fitsstr;

	/* basename is original filename without '.fits' or '.fit' */
	basename = strdup( origfilename );
	malloc_check( basename );
	dotfits = &(basename[strlen( basename ) - 5]);
	if( strcmp( dotfits, ".fits" ) == 0 )
		{
			fitsstr = ".fits";
			*dotfits = '\0';
		}
	else
		{
			dotfits = &(basename[strlen( basename ) - 4]);
			if( strcmp( dotfits, ".fit" ) == 0)
				{
					fitsstr = ".fit";
					*dotfits = '\0';
				}
			else
				{
					fitsstr = "";
				}
		}

	/* projection */
	if( projection > 0 )
		snprintf(projectionstr, sizeof( projectionstr ), ".%ld", projection );
	else
		*projectionstr = '\0';

	/* concatenate */
	total = strlen( basename ) + strlen( projectionstr ) + strlen( extension )
		+ strlen( fitsstr ) + 1;
	result = (char *) malloc( sizeof(char) * total );
	malloc_check( result );
	snprintf( result, total, "%s%s%s%s", basename, projectionstr,
		extension, fitsstr );

	free( basename );

	return result;
}

void
create_boundary_image( fitsfile *infits, long projection,
	const boundary_pixels_s *pixels, const char *infilename,
	const char *fileext )
{
	int status = 0;
	int naxis;
	long i, npix, *iaxes, *naxes;
	char *outfilename;
	unsigned char *image;
	fitsfile *outfits;
	boundary_pixel_s *curpixel;

	outfilename = fibermeas_ext_filename( infilename, fileext,
		projection );
	malloc_check( outfilename );
	unlink( outfilename );
	fits_create_file( &outfits, outfilename, &status );
	cfits_check( status );
	fits_copy_header( infits, outfits, &status );
	cfits_check( status );
	if( projection > 0 )
		{
			fits_decrease_dimension( outfits, &status );
			cfits_check( status );
		}

	naxes = fits_get_image_dimension( outfits, &naxis, &status );
	cfits_check( status );
	malloc_check( naxes );
	iaxes = (long *) malloc( sizeof( long ) * naxis );
	npix = fits_npix( naxis, naxes );
	image = fits_alloc_unsigned_char_image_buffer( naxis, naxes );
	malloc_check( image );
	for(i = 0; i < npix; i++)
		image[i] = 0;

	curpixel = fits_boundary_pixels_first( pixels );
	do
		{
			fits_set_unsigned_char_pix( image, 1, curpixel->x, naxis, naxes );
		} while( (curpixel = fits_boundary_pixels_next( curpixel )) != NULL );

	firstpix( iaxes, naxis );
	fits_write_pix( outfits, TBYTE, iaxes, npix, image, &status );
	cfits_check( status );
	fits_close_file( outfits, &status );
	cfits_check( status );
	free( outfilename );
	free( iaxes );
	free( naxes );
}

void
create_aperture_image( fitsfile *infits, long projection,
	const circle_s *aperture, const char *infilename,
	const char *fileext )
{
	int status = 0;
	int naxis;
	long npix, *iaxes, *naxes;
	char *outfilename;
	unsigned char *image;
	fitsfile *outfits;

	outfilename = fibermeas_ext_filename( infilename, fileext, projection );
	malloc_check( outfilename );
	unlink( outfilename );
	fits_create_file( &outfits, outfilename, &status );
	cfits_check( status );
	fits_copy_header( infits, outfits, &status );
	cfits_check( status );
	if( projection > 0 )
		{
			fits_decrease_dimension( outfits, &status );
			cfits_check( status );
		}

	naxes = fits_get_image_dimension( outfits, &naxis, &status );
	malloc_check( naxes );
	iaxes = (long *) malloc( sizeof( long ) * naxis );
	npix = fits_npix( naxis, naxes );
	image = fits_alloc_unsigned_char_image_buffer( naxis, naxes );
	malloc_check( image );

	firstpix( iaxes, naxis );
	do
		{
			if( in_aperture( iaxes, aperture ) )
				fits_set_unsigned_char_pix( image, 1, iaxes, naxis, naxes );
			else
				fits_set_unsigned_char_pix( image, 0, iaxes, naxis, naxes );
		} while( nextpix( iaxes, naxis, naxes ) );

	firstpix( iaxes, naxis );
	fits_write_pix( outfits, TBYTE, iaxes, npix, image, &status );
	cfits_check( status );
	fits_close_file( outfits, &status );
	cfits_check( status );
	free( outfilename );
	free( iaxes );
}

void show_result( FILE *stream, const fiberphoto_result_s *result )
{
	if( result == NULL )
		{
			fputs(
				"#file\tz\ttotal_in\ttotal_out\tx\ty\tr\tn_in\tn_out\tr_img\tdx\tdy\tdr_img\tsigma\tlocfile\tthresh\n",
				stream );
			return;
		}
	fprintf( stream,
		"%s\t%ld\t%.5g\t%.5g\t%.2f\t%.2f\t%.2f\t%ld\t%ld\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\t%s\t%.5g\n",
		result->filename,
		result->projection,
		result->photo.total_in,
		result->photo.total_out,
		result->aperture->x,
		result->aperture->y,
		result->aperture->r,
		result->photo.n_in,
		result->photo.n_out,
		(result->circle != NULL) ? result->circle->r: 0.0,
		(result->circle != NULL) ? result->circle->dx: 0.0,
		(result->circle != NULL) ? result->circle->dy: 0.0,
		(result->circle != NULL) ? result->circle->dr: 0.0,
		(result->fitting != NULL) ? result->fitting->sigma : 0.0,
		(result->locfilename != NULL) ? result->locfilename : "command_defined",
		result->threshold
	);
}

/*
 *
 * main routine
 *
 */
int
main( int argc, char *argv[])
{
	/* argv index for the input files */
	int argind;

	/* aperture defnition and measurement resuls */
	fiberphoto_result_s results;
	circle_s aperture, imagecircle, *locfileaperture = NULL;
	fit_result_s imagecirclefit;

	/* read command line options */
	fiberphoto_config_s config;
	fiberphoto_set_defaults( &config );
	argind = fiberphoto_get_options( argc, argv, &config );

	/* aperture pre-defnition */
	if( config.predef_aperture != NULL )
		{
			results.locfilename = NULL;
			results.circle = NULL;
			results.fitting = NULL;
			results.aperture = config.predef_aperture;
			results.threshold = 0;
			results.fitting = NULL;
		}
	if( config.locfilename != NULL )
		{
			int status = 0, naxis;
			long projection;
			fitsfile *infits;
			double thresh;

			if( config.vf > 0 )
				{
					fprintf( stderr, "Processing reference image: %s\n",
						config.locfilename );
				}

			fits_open_file( &infits, config.locfilename, READONLY, &status );
			cfits_check( status );
			fits_get_img_dim( infits, &naxis, &status );
			cfits_check( status );
			switch( naxis )
				{
					case 2:
						projection = 0;
						break;
					case 3:
						projection = 1;
						break;
					default:
						fprintf( stderr, "Location file %s has to be 2D or 3D.\n",
							config.locfilename );
						exit( EXIT_FAILURE );
				}

			/* fit to the image */
			if( config.autothresh )
				thresh = get_thresh( infits, projection, config.lower, config.upper );
			else
				thresh = config.manualthresh;
			fiber_fit_circle( infits, projection, &imagecircle,
				&imagecirclefit, thresh, config.boundratio, config.imax,
				config.sigma_rej, config.vf, config.eps_abs, config.eps_rel,
				config.locfilename, config.boundaryfileext );
			if( imagecirclefit.err )
				{
					fprintf( stderr, "%s\n", fit_str_error( imagecirclefit.err ) );
					exit( EXIT_FAILURE );
				}

			/* define aperture */
			locfileaperture = (circle_s *) malloc( sizeof( circle_s ) );
			malloc_check( locfileaperture );
			locfileaperture->x = imagecircle.x;
			locfileaperture->y = imagecircle.y;
			locfileaperture->r = imagecircle.r * config.scale;
			config.predef_aperture = locfileaperture;
				// we have to make aperture image later if required

			/* remember what we have done */
			results.locfilename = config.locfilename;
			results.circle = &imagecircle;
			results.fitting = &imagecirclefit;
			results.aperture = config.predef_aperture;
			results.threshold = thresh;
			results.fitting = &imagecirclefit;
		}

	/* header of the output */
	if( ! config.nohead )
		show_result( stdout, NULL );

	/* scan through the files and image planes */
	{
		char *infilename;
		fitsfile *infits;
		int status, naxis;
		long *naxes;
		long projection, projectionmax = 0;
		circle_s *cur_aperture;
		for( ; argind < argc; argind++ )
			{
				infilename = argv[argind];
				results.filename = infilename;

				/* open the file and check the dimension */
				status = 0;
				fits_open_file( &infits, infilename, READONLY, &status );
				cfits_check( status );
				fits_get_img_dim( infits, &naxis, &status );
				cfits_check( status );
				naxes = (long *) malloc( sizeof( long ) * naxis );
				malloc_check( naxes );
				fits_get_img_size( infits, naxis, naxes, &status );
				cfits_check( status );
				switch( naxis )
					{
						case 2:
							projection = 0;
							projectionmax = 0;
							break;
						case 3:
							projection = 1;
							projectionmax = naxes[naxis - 1];
							break;
						default:
							projection = -1;
							fprintf( stderr, "Input file %s has to be 2D or 3D. Skipping.\n", 
								infilename );
					}
				if( projection < 0 )
					{
						fits_close_file( infits, &status );
						cfits_check( status );
						continue;
					}

				/* scan through the image planes */
				for( ; projection <= projectionmax; projection++ )
					{
						if( config.vf > 0 )
							{
								if( projection > 0)
									fprintf( stderr, "Processing: %s[,,%ld]\n", infilename,
										projection );
								else
									fprintf( stderr, "Processing: %s\n", infilename );
							}
						results.projection = projection;

						/* aperture defnition */
						if( config.predef_aperture == NULL )
							{
								double thresh;

								if( config.vf > 1 )
									fputs( "defining aperture.\n", stderr );
								/* decide threshold */
								if( config.autothresh )
									thresh = get_thresh( infits, projection, config.lower,
										config.upper );
								else
									thresh = config.manualthresh;
								/* fit circle */
								fiber_fit_circle( infits, projection, &imagecircle,
									&imagecirclefit, thresh, config.boundratio,
									config.imax, config.sigma_rej, config.vf,
									config.eps_abs, config.eps_rel,
									infilename, config.boundaryfileext );
								if( imagecirclefit.err )
									{
										fprintf( stderr, "%s\n",
											fit_str_error( imagecirclefit.err ) );
										exit( EXIT_FAILURE );
									}
								/* define aperture */
								aperture.x = imagecircle.x;
								aperture.y = imagecircle.y;
								aperture.r = imagecircle.r * config.scale;
								cur_aperture = &aperture;
								/* remeber things */
								results.locfilename = infilename;
								results.circle = &imagecircle;
								results.fitting = &imagecirclefit;
								results.aperture = &aperture;
								results.threshold = thresh;
							}
						else
							{
								cur_aperture = config.predef_aperture;
							}
						
						/* measurement */
						fiberphoto_measure( infits, projection, &(results.photo),
							cur_aperture );

						/* output */
						if( config.aperturefileext != NULL )
							create_aperture_image( infits, projection, cur_aperture,
								infilename, config.aperturefileext );

						show_result( stdout, &results );
					}

				fits_close_file( infits, &status );
				cfits_check( status );
			}
	}

	/* after care */
  if( config.predef_aperture != NULL )
		free( config.predef_aperture );

	return EXIT_SUCCESS;
}
