/* circle-points.c : fit a circle onto points */
/* Copyright (C) 2006 Daigo Tomono <tomono at subaru.naoj.org> */
/* Licensed under the terms of GPL2 or later */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gsl/gsl_math.h>

#include "points.h"
#include "fitting.h"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#define CMD_NAME "circle-points"
#define XCOL 1
#define YCOL 2

#define SIGMA_REJ -1.0
#define IMAX 100
#define EPS_ABS 1e-5
#define EPS_REL 1e-5

void usage(FILE *stream);

/* command line options */
#define OPTIONS "hvq"

void
usage(FILE *stream)
{
	fputs("usage: " CMD_NAME " [options] [file]\n\
\n\
This program fits a circle onto points from input (stdin or FILE).\n\
\n\
options:\n\
-v: increase verbosity\n\
-q: decrease verbosity\n\
-h: show this\n\
", stream);
}

int
main(int argc, char *argv[])
{
	points_s *points;
	int vf = 1;

	/* command line option */
	{
		int c;
		while(1)
			{
				c = getopt(argc, argv, OPTIONS);
				if (c == -1) break;
				switch(c)
					{
						case 'h':
							usage(stdout);
							exit(EXIT_SUCCESS);
						case 'v': vf++; break;
						case 'q': vf--; break;
						case '?':
							usage(stderr);
							exit(EXIT_FAILURE);
					}
			}
	}
	
	/* read data file(s) or stdin */
	points = points_alloc();
	if (!points)
		{
			perror(CMD_NAME);
			exit(EXIT_FAILURE);
		}

	{
		int nfiles;
		nfiles = argc - optind;
		while(optind < argc || nfiles == 0)
			{
				FILE *stream;
				if (nfiles == 0)
					{
						if (vf > 0) fputs("taking data from stdin\n", stderr);
						nfiles = 1;
						stream = stdin;
					}
				else
					{
						stream = fopen(argv[optind], "r");
						if (!stream)
							{
								perror(CMD_NAME);
								exit(EXIT_FAILURE);
							}
					}

				if (!points_read(points, stream, XCOL, YCOL))
					{
						perror(CMD_NAME);
						exit(EXIT_FAILURE);
					}
				optind++;
			}

		if (vf > 2)
			{
				long i;
				fprintf(stderr, "%ld data read:\n", points->n);
				for(i = 0; i < points->n; i++)
					{
						fprintf(stderr, "%g %g\n", points->x[i], points->y[i]);
					}
			}
	}

	/* fitting */
	{
		fit_result_s result;
		circle_s circle_guess, circle_fit;
		double scale;

		guess_circle(&circle_guess, points->n, points->x, points->y);
		if (vf > 2)
			{
				fputs("\nInitial guess of circle:\n", stderr);
				show_circle(&circle_guess, stderr);
			}
		fit_circle(&circle_fit, &result, &circle_guess,
			points->n, points->x, points->y,
			IMAX, SIGMA_REJ, vf - 1, EPS_ABS, EPS_REL);
		if (vf > 1)
			{
				fputs("\nResult of fitting: ", stderr);
				show_fit_result(&result, stderr);
				show_circle(&circle_fit, stderr);
			}

		/* output */
		scale = result.sigma;
		fprintf(stdout,
			"#center: %7.2f+-%6.2f, %7.2f+-%6.2f radius: %7.2f+-%6.2f\n",
			circle_fit.x, circle_fit.dx*scale,
			circle_fit.y, circle_fit.dy*scale,
			circle_fit.r, circle_fit.dr*scale
		);
		fputs("# error bars scaled with the residual errors\n", stdout);
		fputs("#", stdout);
		show_circle(&circle_fit, stdout);
		fputs("# error bars not scaled\n", stdout);
		fputs("#x\ty\n", stdout);
		{
			double t;
			for(t = 0; t < M_PI*2.0; t += M_PI*2.0/64)
				{
					double x, y;
					x = circle_fit.x + circle_fit.r*cos(t);
					y = circle_fit.y + circle_fit.r*sin(t);
					fprintf(stdout, "%g\t%g\n", x, y);
				}
		}
	}

	points_free(points);

	return EXIT_SUCCESS;
}
